<?php
// app/baseKRIZAN/Modules/ModuleRegistry.php

namespace baseKRIZAN\Modules;

use baseKRIZAN\Error\Logger;

/**
 * Registar za praćenje registriranih modula
 * 
 * @package baseKRIZAN\Modules
 */
class ModuleRegistry
{
    /**
     * @var array Registrirani moduli
     */
    private static array $modules = [];
    
    /**
     * @var Logger|null
     */
    private static ?Logger $logger = null;
    
    /**
     * Inicijalizacija registra
     * 
     * @param Logger $logger
     */
    public static function init(Logger $logger): void
    {
        self::$logger = $logger;
    }
    
    /**
     * Registracija modula u registar
     * 
     * @param string $id ID modula
     * @param array $data Podaci o modulu
     * @return bool
     */
    public static function register(string $id, array $data): bool
    {
        if (self::$logger) {
            self::$logger->modules('Registering module in registry', [
                'id' => $id,
                'name' => $data['name'] ?? $id
            ]);
        }
        
        self::$modules[$id] = $data;
        return true;
    }
    
    /**
     * Provjera je li modul registriran
     * 
     * @param string $id ID modula
     * @return bool
     */
    public static function isRegistered(string $id): bool
    {
        return isset(self::$modules[$id]);
    }
    
    /**
     * Dohvaćanje modula prema ID-u
     * 
     * @param string $id ID modula
     * @return array|null Podaci o modulu ili null ako nije pronađen
     */
    public static function get(string $id): ?array
    {
        return self::$modules[$id] ?? null;
    }
    
    /**
     * Dohvaćanje svih registriranih modula
     * 
     * @return array
     */
    public static function getAll(): array
    {
        return self::$modules;
    }
    
    /**
     * Brisanje modula iz registra
     * 
     * @param string $id ID modula
     * @return bool
     */
    public static function unregister(string $id): bool
    {
        if (isset(self::$modules[$id])) {
            if (self::$logger) {
                self::$logger->modules('Unregistering module from registry', [
                    'id' => $id
                ]);
            }
            
            unset(self::$modules[$id]);
            return true;
        }
        
        return false;
    }
}