<?php
/**
 * Route Builder - fluent API za definiranje ruta
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use baseKRIZAN\Error\Logger;

class RouteBuilder {
    /**
     * Pohranjene definirane rute
     * @var array
     */
    private array $routes = [];
    
    /**
     * Trenutni prefix za rute
     * @var string|null
     */
    private ?string $prefix = null;
    
    /**
     * Middleware za primjenu
     * @var array
     */
    private array $middleware = [];
    
    /**
     * Namespace kontrolera
     * @var string|null
     */
    private ?string $namespace = null;
    
    /**
     * Je li potrebna autentikacija
     * @var bool|null
     */
    private ?bool $authRequired = null;
    
    /**
     * Potrebne dozvole
     * @var mixed
     */
    private $permissions = null;
    
    /**
     * Zadnje dodana putanja za chaining
     * @var string|null
     */
    private ?string $lastAddedPath = null;
    
    /**
     * Logger instanca
     * @var Logger|null
     */
    private ?Logger $logger = null;
    
    /**
     * Grupne konfiguracije za stack manjih grupa
     * @var array
     */
    private array $groupStack = [];
    
    /**
     * Metode za rukovanje rutama
     * @var array
     */
    private array $httpMethods = ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS', 'HEAD'];

    /**
     * Konstruktor
     *
     * @param Logger|null $logger Logger instanca
     */
    public function __construct(?Logger $logger = null)
    {
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->routing('RouteBuilder initialized');
        }
    }

    /**
     * Postavlja logger instancu
     *
     * @param Logger|null $logger Logger instanca
     * @return self
     */
    public function setLogger(?Logger $logger): self
    {
        $this->logger = $logger;
        return $this;
    }
    
    /**
     * Definira grupu ruta s zajedničkim atributima
     *
     * @param array $attributes Atributi grupe
     * @param \Closure $callback Callback za definiranje ruta
     * @return self
     */
    public function group(array $attributes, \Closure $callback): self {
        // Push current state to stack
        $this->pushToStack();

        // Apply group attributes
        if (isset($attributes['prefix'])) {
            $this->prefix = $this->prefix ? $this->prefix . '/' . trim($attributes['prefix'], '/') : trim($attributes['prefix'], '/');
        }
        
        if (isset($attributes['middleware'])) {
            $this->middleware = array_merge($this->middleware, (array)$attributes['middleware']);
        }
        
        if (isset($attributes['namespace'])) {
            $this->namespace = $attributes['namespace'];
        }
        
        if (isset($attributes['auth'])) {
            $this->authRequired = $attributes['auth'];
        }
        
        if (isset($attributes['permission'])) {
            $this->permissions = $attributes['permission'];
        }

        // Execute callback
        $callback($this);

        // Restore state from stack
        $this->popFromStack();

        return $this;
    }
    
    /**
     * Sprema trenutno stanje na stack
     */
    private function pushToStack(): void {
        $this->groupStack[] = [
            'prefix' => $this->prefix,
            'middleware' => $this->middleware,
            'namespace' => $this->namespace,
            'auth' => $this->authRequired,
            'permissions' => $this->permissions,
            'lastAddedPath' => $this->lastAddedPath
        ];
        
        // Reset lastAddedPath for new group context
        $this->lastAddedPath = null;
    }
    
    /**
     * Vraća zadnje stanje sa stacka
     */
    private function popFromStack(): void {
        if (empty($this->groupStack)) {
            return;
        }
        
        $state = array_pop($this->groupStack);
        $this->prefix = $state['prefix'];
        $this->middleware = $state['middleware'];
        $this->namespace = $state['namespace'];
        $this->authRequired = $state['auth'];
        $this->permissions = $state['permissions'];
        $this->lastAddedPath = $state['lastAddedPath'];
    }
    
    /**
     * Dodaje GET rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function get(string $path, array $action): self {
        return $this->addRoute('GET', $path, $action);
    }

    /**
     * Dodaje POST rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function post(string $path, array $action): self {
        return $this->addRoute('POST', $path, $action);
    }

    /**
     * Dodaje PUT rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function put(string $path, array $action): self {
        return $this->addRoute('PUT', $path, $action);
    }

    /**
     * Dodaje DELETE rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function delete(string $path, array $action): self {
        return $this->addRoute('DELETE', $path, $action);
    }
    
    /**
     * Dodaje PATCH rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function patch(string $path, array $action): self {
        return $this->addRoute('PATCH', $path, $action);
    }
    
    /**
     * Dodaje OPTIONS rutu
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function options(string $path, array $action): self {
        return $this->addRoute('OPTIONS', $path, $action);
    }

    /**
     * Označava rutu kao pattern (za rute s parametrima)
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function pattern(string $path, array $action): self {
        $route = $this->addRoute('GET', $path, $action);
        $this->routes[$path]['pattern'] = true;
        return $route;
    }

    /**
     * Dodaje dodatne HTTP metode zadnjoj definiranoj ruti
     *
     * @param array $methods HTTP metode
     * @return self
     */
    public function methods(array $methods): self {
        $lastPath = $this->lastAddedPath;
        if ($lastPath) {
            $lastRoute = $this->routes[$lastPath];
            
            foreach ($methods as $method) {
                if (isset($lastRoute[$method])) {
                    continue;
                }
                
                if (isset($lastRoute['GET'])) {
                    $this->routes[$lastPath][$method] = $lastRoute['GET'];
                }
            }
        }
        
        return $this;
    }

    /**
     * Dodaje rutu za sve standardne HTTP metode
     *
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    public function any(string $path, array $action): self {
        foreach ($this->httpMethods as $method) {
            $this->addRoute($method, $path, $action);
        }
        return $this;
    }

    /**
     * Dodaje novu rutu
     *
     * @param string $method HTTP metoda
     * @param string $path Putanja
     * @param array $action Controller i akcija
     * @return self
     */
    private function addRoute(string $method, string $path, array $action): self {
        // Apply prefix
        if ($this->prefix) {
            $path = $this->prefix . '/' . trim($path, '/');
        }
    
        // Normalize path
        $path = trim($path, '/');
        if (empty($path)) {
            $path = '/';
        }
    
        // Apply namespace to controller if it's a string and namespace is set
        if ($this->namespace && is_string($action[0]) && strpos($action[0], '\\') === false) {
            $action[0] = $this->namespace . '\\' . $action[0];
        }
    
        // Detect automatically routes with parameters
        $hasParams = strpos($path, '{') !== false && strpos($path, '}') !== false;
    
        if ($this->logger) {
            $this->logger->routing('Adding route', [
                'method' => $method,
                'path' => $path,
                'has_parameters' => $hasParams ? 'yes' : 'no'
            ]);
        }
    
        // Create route if not exists
        if (!isset($this->routes[$path])) {
            $this->routes[$path] = [
                'middleware' => $this->middleware,
                'login' => $this->authRequired,
                'auth' => $this->authRequired,
                'permissions' => $this->permissions,
                'pattern' => $hasParams,
                'name' => null
            ];
        }
    
        // Store controller as string identifier or keep object
        $controller = $action[0];
        $controllerValue = $controller;
        
        // Convert object controller to string identifier if needed
        if (is_object($controller)) {
            $className = get_class($controller);
            $parts = explode('\\', $className);
            $controllerName = end($parts);
            $controllerName = strtolower(str_replace('Controller', '', $controllerName));
            $controllerValue = $controllerName . 'Controller';
        }
    
        // Add method handler
        $this->routes[$path][$method] = [
            'controller' => $controllerValue,
            'action' => $action[1]
        ];
    
        // Set last added path for chaining methods
        $this->lastAddedPath = $path;
    
        return $this;
    }
    
    /**
     * Postavlja prefix za buduće rute
     *
     * @param string $prefix Prefix
     * @return self
     */
    public function prefix(string $prefix): self {
        $this->prefix = trim($prefix, '/');
        return $this;
    }

    /**
     * Dodaje middleware komponentu
     *
     * @param array|string $middleware Middleware
     * @return self
     */
    public function middleware(array|string $middleware): self {
        if (is_string($middleware)) {
            $middleware = [$middleware];
        }
        
        // If a route is already defined, add middleware to the last added route
        if ($this->lastAddedPath !== null) {
            if (!isset($this->routes[$this->lastAddedPath]['middleware'])) {
                $this->routes[$this->lastAddedPath]['middleware'] = [];
            }
            $this->routes[$this->lastAddedPath]['middleware'] = array_merge(
                $this->routes[$this->lastAddedPath]['middleware'],
                $middleware
            );
            // Deduplicate middleware
            $this->routes[$this->lastAddedPath]['middleware'] = array_unique($this->routes[$this->lastAddedPath]['middleware']);
        } else {
            // Otherwise set for future routes
            $this->middleware = array_merge($this->middleware, $middleware);
        }
        
        return $this;
    }

    /**
     * Postavlja namespace za kontrolere
     *
     * @param string $namespace Namespace
     * @return self
     */
    public function namespace(string $namespace): self {
        $this->namespace = $namespace;
        return $this;
    }

    /**
     * Postavlja zahtjevanu autentikaciju
     *
     * @param bool $required Potrebna autentikacija
     * @return self
     */
    public function auth(bool $required = true): self {
        // If a route is already defined, set auth on the last added route
        if ($this->lastAddedPath !== null) {
            $this->routes[$this->lastAddedPath]['login'] = $required;
            $this->routes[$this->lastAddedPath]['auth'] = $required;
        } else {
            // Otherwise set for future routes
            $this->authRequired = $required;
        }
        return $this;
    }

    /**
     * Postavlja potrebne dozvole
     *
     * @param mixed $permission Dozvole
     * @return self
     */
    public function permission($permission): self {
        // If a route is already defined, set permission on the last added route
        if ($this->lastAddedPath !== null) {
            $this->routes[$this->lastAddedPath]['permissions'] = $permission;
        } else {
            // Otherwise set for future routes
            $this->permissions = $permission;
        }
        return $this;
    }
    
    /**
     * Postavlja ime rute za lakši pristup
     *
     * @param string $name Ime rute
     * @return self
     */
    public function name(string $name): self {
        if ($this->lastAddedPath !== null) {
            $this->routes[$this->lastAddedPath]['name'] = $name;
        }
        return $this;
    }

    /**
     * Resetira stanje buildera
     *
     * @return self
     */
    public function reset(): self {
        $this->middleware = [];
        $this->authRequired = null;
        $this->permissions = null;
        $this->lastAddedPath = null;
        $this->prefix = null;
        $this->namespace = null;
        return $this;
    }

    /**
     * Dohvaća definirane rute
     *
     * @return array Rute
     */
    public function getRoutes(): array {
        return $this->routes;
    }
    
    /**
     * Zapisuje debug informacije o rutama
     *
     * @param string|null $logPath Putanja log datoteke
     * @return self
     */
    public function debugRoutes(string $logPath = null): self {
        $routes = $this->getRoutes();
        
        // If no path specified, use from env variable
        if ($logPath === null) {
            $logPath = \baseKRIZAN\Config\Config::get('paths.logs') . '/debug/routes.json';
        }
        
        if ($this->logger) {
            $this->logger->routing('Writing routes debug file', [
                'path' => $logPath,
                'route_count' => count($routes)
            ]);
        }
        
        // Check and create directory if it doesn't exist
        $dir = dirname($logPath);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        // Write JSON
        $result = file_put_contents(
            $logPath, 
            json_encode($routes, JSON_PRETTY_PRINT)
        );
        
        if ($this->logger) {
            if ($result !== false) {
                $this->logger->routing('Routes debug file written successfully', [
                    'bytes_written' => $result
                ]);
            } else {
                $this->logger->error('Failed to write routes debug file', [
                    'path' => $logPath
                ]);
            }
        }
        
        return $this;
    }
    
    /**
     * API resource routes shorthand
     * 
     * @param string $name Ime resursa
     * @param array $controller Kontroler i opcije
     * @return self
     */
    public function resource(string $name, array $controller): self {
        $name = trim($name, '/');
        
        // Index route - GET /resource
        $this->get($name, [$controller[0], 'index']);
        
        // Show route - GET /resource/{id}
        $this->get("$name/{id}", [$controller[0], 'show']);
        
        // Create route - GET /resource/create
        $this->get("$name/create", [$controller[0], 'create']);
        
        // Store route - POST /resource
        $this->post($name, [$controller[0], 'store']);
        
        // Edit route - GET /resource/{id}/edit
        $this->get("$name/{id}/edit", [$controller[0], 'edit']);
        
        // Update route - PUT /resource/{id}
        $this->put("$name/{id}", [$controller[0], 'update']);
        
        // Delete route - DELETE /resource/{id}
        $this->delete("$name/{id}", [$controller[0], 'destroy']);
        
        // Return for chaining
        return $this;
    }
}