<?php
/**
 * Route Collection - upravlja i pohranjuje definirane rute
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use baseKRIZAN\Error\Logger;

class RouteCollection {
    /**
     * Pohranjuje sve definirane rute
     * @var array
     */
    private array $routes = [];
    
    /**
     * Pohranjuje grupe middleware komponenti
     * @var array
     */
    private array $middlewareGroups = [];
    
    /**
     * Logger instanca
     * @var Logger|null
     */
    private ?Logger $logger = null;
    
    /**
     * Cache spremnih ruta za brži lookup
     * @var array
     */
    private array $resolvedRouteCache = [];
    
    /**
     * Maksimalna veličina cache-a
     * @var int
     */
    private int $cacheSize = 200;

    /**
     * Konstruktor
     *
     * @param Logger|null $logger Logger instanca za praćenje aktivnosti
     */
    public function __construct(?Logger $logger = null) {
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->routing('RouteCollection initialized');
        }
    }

    /**
     * Dodaje novu rutu u kolekciju
     *
     * @param string $path Putanja rute
     * @param string $method HTTP metoda (GET, POST, PUT, DELETE)
     * @param array $handler Konfiguracija handlera 
     * @param array $options Dodatne opcije za rutu
     * @return void
     */
    public function add(string $path, string $method, array $handler, array $options = []): void {
        // Normaliziraj putanju
        $path = trim($path, '/');
        
        // Invalidira cache za ovu putanju
        $this->invalidateRouteCache($path, $method);
        
        if ($this->logger) {
            $this->logger->routing('Adding route to collection', [
                'path' => $path,
                'method' => $method,
                'controller' => $handler['handler']['controller'] ?? 'unknown',
                'action' => $handler['handler']['action'] ?? 'unknown'
            ]);
        }
        
        // Initialize the route if it doesn't exist yet
        if (!isset($this->routes[$path])) {
            $this->routes[$path] = [
                'login' => $options['login'] ?? false,
                'auth' => $options['auth'] ?? null,
                'permissions' => $options['permissions'] ?? null,
                'middleware' => $options['middleware'] ?? [],
                'name' => $options['name'] ?? null,
                'pattern' => $options['pattern'] ?? false
            ];
        } else {
            // Merge options with existing route (don't overwrite if null)
            if (isset($options['login'])) {
                $this->routes[$path]['login'] = $options['login'];
            }
            
            // Add auth property
            if (isset($options['auth'])) {
                $this->routes[$path]['auth'] = $options['auth'];
            }
            
            if (isset($options['permissions'])) {
                $this->routes[$path]['permissions'] = $options['permissions'];
            }
            
            if (isset($options['middleware'])) {
                $existingMiddleware = $this->routes[$path]['middleware'] ?? [];
                $this->routes[$path]['middleware'] = array_unique(
                    array_merge($existingMiddleware, $options['middleware'])
                );
            }
            
            if (isset($options['name'])) {
                $this->routes[$path]['name'] = $options['name'];
            }
            
            if (isset($options['pattern'])) {
                $this->routes[$path]['pattern'] = $options['pattern'];
            }
        }
        
        // Add the method handler
        $this->routes[$path][$method] = $handler;
    }

    /**
     * Dodaje grupu middleware komponenti
     *
     * @param string $name Naziv grupe
     * @param array $middleware Niz middleware komponenti
     * @return void
     */
    public function addMiddlewareGroup(string $name, array $middleware): void {
        $this->middlewareGroups[$name] = $middleware;
        
        if ($this->logger) {
            $this->logger->routing('Added middleware group', [
                'name' => $name,
                'count' => count($middleware)
            ]);
        }
    }

    /**
     * Dohvaća sve registrirane middleware grupe
     *
     * @return array Middleware grupe
     */
    public function getMiddlewareGroups(): array {
        return $this->middlewareGroups;
    }

    /**
     * Dohvaća sve registrirane rute
     *
     * @return array Sve rute
     */
    public function getRoutes(): array {
        return $this->routes;
    }
    
    /**
     * Dohvaća rutu po putanji i metodi (optimizirani pristup)
     *
     * @param string $path Putanja rute
     * @param string $method HTTP metoda
     * @return array|null Ruta ili null ako nije pronađena
     */
    public function getRoute(string $path, string $method): ?array {
        $path = trim($path, '/');
        $cacheKey = $this->generateCacheKey($path, $method);
        
        // Check cache first
        if (isset($this->resolvedRouteCache[$cacheKey])) {
            if ($this->logger) {
                $this->logger->routing('Route found in cache', [
                    'path' => $path,
                    'method' => $method
                ]);
            }
            return $this->resolvedRouteCache[$cacheKey];
        }
        
        // Check direct match
        if (isset($this->routes[$path]) && isset($this->routes[$path][$method])) {
            $routeData = $this->routes[$path];
            
            // Add path to route data
            $routeData['path'] = $path;
            
            // Cache the route for faster lookup next time
            $this->cacheRoute($cacheKey, $routeData);
            
            return $routeData;
        }
        
        return null;
    }
    
    /**
     * Dohvaća rute s pattern matched parametrima
     *
     * @return array Rute s pattern parametrima
     */
    public function getPatternRoutes(): array {
        $patternRoutes = [];
        
        foreach ($this->routes as $path => $routeData) {
            if (isset($routeData['pattern']) && $routeData['pattern']) {
                $patternRoutes[$path] = $routeData;
            }
        }
        
        return $patternRoutes;
    }
    
    /**
     * Dohvaća rutu po imenu
     *
     * @param string $name Naziv rute
     * @return array|null Ruta ili null ako nije pronađena
     */
    public function getRouteByName(string $name): ?array {
        foreach ($this->routes as $path => $routeData) {
            if (isset($routeData['name']) && $routeData['name'] === $name) {
                $routeData['path'] = $path;
                return $routeData;
            }
        }
        
        return null;
    }
    
    /**
     * Čisti cache ruta
     *
     * @return void
     */
    public function clearRouteCache(): void {
        $this->resolvedRouteCache = [];
        
        if ($this->logger) {
            $this->logger->routing('Route cache cleared');
        }
    }
    
    /**
     * Invalidira cache za pojedinu rutu
     *
     * @param string $path Putanja rute
     * @param string $method HTTP metoda
     * @return void
     */
    private function invalidateRouteCache(string $path, string $method): void {
        $cacheKey = $this->generateCacheKey($path, $method);
        
        if (isset($this->resolvedRouteCache[$cacheKey])) {
            unset($this->resolvedRouteCache[$cacheKey]);
        }
    }
    
    /**
     * Sprema rutu u cache
     * 
     * @param string $cacheKey Ključ za cache
     * @param array $routeData Podaci rute
     * @return void
     */
    private function cacheRoute(string $cacheKey, array $routeData): void {
        // Implementiran LRU cache - ako premašimo limit, brišemo najstarije zapise
        if (count($this->resolvedRouteCache) >= $this->cacheSize) {
            array_shift($this->resolvedRouteCache);
        }
        
        $this->resolvedRouteCache[$cacheKey] = $routeData;
    }
    
    /**
     * Generira ključ za cache ruta
     *
     * @param string $path Putanja rute
     * @param string $method HTTP metoda
     * @return string Ključ za cache
     */
    private function generateCacheKey(string $path, string $method): string {
        return $method . ':' . $path;
    }
    
    /**
     * Postavlja maksimalnu veličinu cache-a
     * 
     * @param int $size Nova veličina cache-a
     * @return void
     */
    public function setCacheSize(int $size): void {
        if ($size > 0) {
            $this->cacheSize = $size;
        }
    }
    
    /**
     * Dodaje više ruta odjednom
     * 
     * @param array $routes Rute za dodati
     * @return void
     */
    public function addRoutes(array $routes): void {
        foreach ($routes as $path => $routeData) {
            foreach (['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS', 'HEAD'] as $method) {
                if (isset($routeData[$method])) {
                    $handler = $routeData[$method];
                    $options = [
                        'login' => $routeData['login'] ?? false,
                        'auth' => $routeData['auth'] ?? null,
                        'permissions' => $routeData['permissions'] ?? null,
                        'middleware' => $routeData['middleware'] ?? [],
                        'name' => $routeData['name'] ?? null,
                        'pattern' => $routeData['pattern'] ?? false
                    ];
                    
                    $this->add($path, $method, $handler, $options);
                }
            }
        }
    }
}