<?php
/**
 * Route Dispatcher - izvršava akcije kontrolera za pronađene rute
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Services\Container;

class RouteDispatcher {
    /**
     * Template renderer
     * @var TemplateRenderer
     */
    private TemplateRenderer $templateRenderer;
    
    /**
     * Logger instanca
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Development mod
     * @var bool
     */
    private bool $developmentMode;
    
    /**
     * Cache instanciranih kontrolera
     * @var array
     */
    private array $controllerCache = [];

    /**
     * Konstruktor
     *
     * @param TemplateRenderer $templateRenderer Renderer za view-ove
     * @param Logger $logger Logger za praćenje aktivnosti
     */
    public function __construct(TemplateRenderer $templateRenderer, Logger $logger) 
    {
        $this->templateRenderer = $templateRenderer;
        $this->logger = $logger;
        $this->developmentMode = \baseKRIZAN\Config\Config::get('environment') === "development";
        
        $this->logger->routing('RouteDispatcher initialized', [
            'development_mode' => $this->developmentMode ? 'yes' : 'no'
        ]);
    }

    /**
     * Izvršava akciju kontrolera za zadanu rutu
     *
     * @param Request $request HTTP zahtjev
     * @param array $routeData Podaci rute
     * @param Container $container DI container
     * @return Response HTTP odgovor
     */
    public function dispatch(Request $request, array $routeData, Container $container): Response {
        $this->logger->routing('Dispatching route', [
            'path' => $request->getPath(),
            'method' => $request->getMethod(),
            'controller' => $routeData['controller'] ?? 'unknown',
            'action' => $routeData['action'] ?? 'unknown'
        ]);
        
        // DODANO: Postavi tip zahtjeva iz rute ako je dostupan
        if (isset($routeData['middleware']) && $container->has('requestClassifier')) {
            $requestClassifier = $container->get('requestClassifier');
            $middlewareType = $routeData['middleware'];
            
            // Ako je niz, uzmi prvi element kao tip
            if (is_array($middlewareType) && !empty($middlewareType)) {
                $middlewareType = $middlewareType[0];
            }
            
            // Postavi tip u classifier ako je relevantan
            if (in_array($middlewareType, ['web', 'api'])) {
                $requestClassifier->setRouterDefinedType($middlewareType);
                
                // Također ažuriraj tip u request objektu
                $request->setAttribute('request_type', $middlewareType);
                
                $this->logger->routing('Router defined request type', [
                    'type' => $middlewareType
                ]);
            }
        }
        
        // Handle redirects
        if (isset($routeData['redirect'])) {
            $this->logger->routing('Route redirects to', [
                'redirect' => $routeData['redirect']
            ]);
            return (new Response())->setRedirect($routeData['redirect']);
        }

        // Get controller
        $controller = $this->resolveController($routeData['controller'], $container);
        if (!$controller) {
            $this->logger->error('Could not resolve controller', [
                'identifier' => $routeData['controller']
            ]);
            throw new \RuntimeException("Could not resolve controller: " . $routeData['controller']);
        }

        $action = $routeData['action'];
        $this->logger->routing('Controller resolved, invoking action', [
            'controller' => get_class($controller),
            'action' => $action
        ]);

        if (!method_exists($controller, $action)) {
            $this->logger->error('Action not found on controller', [
                'controller' => get_class($controller),
                'action' => $action
            ]);
            throw new \RuntimeException("Action {$action} does not exist in controller " . get_class($controller));
        }

        // Prepare arguments for action using reflection
        $args = $this->prepareActionArguments($controller, $action, $request, $container);

        // Call the action with the prepared arguments
        $this->logger->routing('Invoking controller action', [
            'args_count' => count($args)
        ]);
        
        try {
            $response = call_user_func_array([$controller, $action], $args);
            
            $this->logger->routing('Controller action executed successfully', [
                'response_type' => is_object($response) ? get_class($response) : gettype($response)
            ]);
            
            // Process the response
            return $this->processResponse($response, $request);
        } catch (\Throwable $e) {
            $this->logger->error('Error executing controller action', [
                'controller' => get_class($controller),
                'action' => $action,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Priprema argumente za akciju kontrolera
     *
     * @param object $controller Instanca kontrolera
     * @param string $action Naziv akcije
     * @param Request $request HTTP zahtjev
     * @param Container $container DI container
     * @return array Argumenti za akciju
     */
    private function prepareActionArguments(object $controller, string $action, Request $request, Container $container): array {
        // Get method parameters using reflection
        $reflectionMethod = new \ReflectionMethod($controller, $action);
        $parameters = $reflectionMethod->getParameters();
        
        $this->logger->routing('Action method signature analyzed', [
            'parameter_count' => count($parameters)
        ]);
        
        // Prepare arguments based on parameter types
        $args = [];
        foreach ($parameters as $parameter) {
            if (!$parameter->hasType()) {
                // Parameter has no type hint, add null or route params if matches name
                $paramName = $parameter->getName();
                if (isset($request->getParams()[$paramName])) {
                    $args[] = $request->getParams()[$paramName];
                } elseif ($parameter->isDefaultValueAvailable()) {
                    $args[] = $parameter->getDefaultValue();
                } else {
                    $args[] = null;
                }
                continue;
            }
            
            $type = $parameter->getType();
            $typeName = $type->getName();
            
            // Check for common types
            if ($typeName === Request::class) {
                $args[] = $request;
            } else if ($typeName === Container::class) {
                $args[] = $container;
            } else if ($type->isBuiltin()) {
                // Handle scalar parameters from route params if available
                $paramName = $parameter->getName();
                if (isset($request->getParams()[$paramName])) {
                    $value = $request->getParams()[$paramName];
                    
                    // Convert to correct type
                    if ($typeName === 'int') {
                        $value = (int) $value;
                    } elseif ($typeName === 'float') {
                        $value = (float) $value;
                    } elseif ($typeName === 'bool') {
                        $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                    }
                    
                    $args[] = $value;
                } elseif ($parameter->isDefaultValueAvailable()) {
                    $args[] = $parameter->getDefaultValue();
                } else {
                    // If no value and not nullable, this will cause an error
                    // but that's the expected behavior
                    $args[] = null;
                }
            } else {
                // Try to resolve dependency from container
                if ($container->has($typeName)) {
                    $args[] = $container->get($typeName);
                } elseif ($type->allowsNull()) {
                    $args[] = null;
                } else {
                    $this->logger->error('Could not resolve parameter for controller action', [
                        'parameter' => $parameter->getName(),
                        'type' => $typeName
                    ]);
                    throw new \RuntimeException("Could not resolve parameter {$parameter->getName()} of type {$typeName}");
                }
            }
        }
        
        return $args;
    }

    /**
     * Procesira odgovor kontrolera i garantira da je Response objekt
     *
     * @param mixed $response Odgovor kontrolera
     * @param Request $request Zahtjev
     * @return Response
     */
    private function processResponse($response, Request $request): Response {
        // If it's already a Response object, return it
        if ($response instanceof Response) {
            return $response;
        }
        
        // If response is an array or object, convert to JSON
        if (is_array($response) || is_object($response)) {
            // Check if request wants JSON
            if ($request->wantsJson() || $request->isAjax()) {
                return Response::json($response);
            }
            
            // Otherwise try to create a view response
            try {
                return Response::view('json_debug.html.php', ['data' => $response], 'Debug View');
            } catch (\Throwable $e) {
                // Fallback to plain JSON
                return Response::json($response);
            }
        }
        
        // If somehow we get a string, wrap it in a Response
        if (is_string($response)) {
            // Check if it might be HTML
            if (strpos($response, '<html') !== false || strpos($response, '<!DOCTYPE') !== false) {
                return new Response($response);
            }
            
            return Response::html($response);
        }
        
        // If it's a primitive (number, bool), convert to JSON
        return Response::json($response);
    }

    /**
     * Resolves a controller from string or object into an instance
     *
     * @param string|object $controllerIdentifier Controller name or instance
     * @param Container $container DI container
     * @return object|null Controller instance or null if not found
     */
    private function resolveController($controllerIdentifier, Container $container): ?object 
    {
        // If it's already an object, return it
        if (is_object($controllerIdentifier)) {
            return $controllerIdentifier;
        }
        
        // Check if we've already instantiated this controller
        $cacheKey = is_string($controllerIdentifier) ? $controllerIdentifier : spl_object_hash($controllerIdentifier);
        
        if (isset($this->controllerCache[$cacheKey])) {
            $this->logger->routing('Using cached controller instance', [
                'identifier' => $controllerIdentifier
            ]);
            return $this->controllerCache[$cacheKey];
        }
        
        $this->logger->routing('Resolving controller', [
            'identifier' => $controllerIdentifier
        ]);
        
        // Try direct container lookup
        if ($container->has($controllerIdentifier)) {
            $controller = $container->get($controllerIdentifier);
            $this->controllerCache[$cacheKey] = $controller;
            return $controller;
        }
        
        // Try common naming variations
        $possibleKeys = [
            $controllerIdentifier,
            strtolower($controllerIdentifier),
            str_replace('Controller', '', $controllerIdentifier) . 'Controller',
            strtolower(str_replace('Controller', '', $controllerIdentifier)) . 'Controller'
        ];
        
        foreach ($possibleKeys as $key) {
            if ($container->has($key)) {
                $controller = $container->get($key);
                $this->controllerCache[$cacheKey] = $controller;
                return $controller;
            }
        }
        
        // If still not found, try to instantiate by class name
        if (class_exists($controllerIdentifier)) {
            $this->logger->routing('Controller not found in container, instantiating by class name', [
                'class' => $controllerIdentifier
            ]);
            
            // Try to use constructor injection with common dependencies
            try {
                // Create reflection and check constructor parameters
                $reflection = new \ReflectionClass($controllerIdentifier);
                $constructor = $reflection->getConstructor();
                
                if (!$constructor) {
                    // No constructor, simple instantiation
                    $controller = new $controllerIdentifier();
                    $this->controllerCache[$cacheKey] = $controller;
                    return $controller;
                }
                
                // Get constructor parameters
                $params = [];
                foreach ($constructor->getParameters() as $param) {
                    $paramName = $param->getName();
                    $paramType = $param->getType();
                    
                    // Try to resolve from container by type
                    if ($paramType && !$paramType->isBuiltin()) {
                        $typeName = $paramType->getName();
                        
                        if ($container->has($typeName)) {
                            $params[] = $container->get($typeName);
                            continue;
                        }
                        
                        // Try common services by name
                        if ($paramName === 'logger' && $container->has('logger')) {
                            $params[] = $container->get('logger');
                            continue;
                        } elseif ($paramName === 'container') {
                            $params[] = $container;
                            continue;
                        } elseif (($paramName === 'sessionManager' || $paramName === 'session') && $container->has('sessionManager')) {
                            $params[] = $container->get('sessionManager');
                            continue;
                        } elseif (($paramName === 'authentication' || $paramName === 'auth') && $container->has('authentication')) {
                            $params[] = $container->get('authentication');
                            continue;
                        }
                    }
                    
                    // If no match and parameter has default value
                    if ($param->isDefaultValueAvailable()) {
                        $params[] = $param->getDefaultValue();
                        continue;
                    }
                    
                    // If the parameter is nullable
                    if ($paramType && $paramType->allowsNull()) {
                        $params[] = null;
                        continue;
                    }
                    
                    // We couldn't resolve this parameter
                    $this->logger->error('Cannot resolve constructor parameter', [
                        'controller' => $controllerIdentifier,
                        'parameter' => $paramName,
                        'type' => $paramType ? $paramType->getName() : 'unknown'
                    ]);
                    
                    // Return null to indicate failure
                    return null;
                }
                
                // Create instance with resolved parameters
                $controller = $reflection->newInstanceArgs($params);
                $this->controllerCache[$cacheKey] = $controller;
                return $controller;
                
            } catch (\Throwable $e) {
                $this->logger->error('Error instantiating controller', [
                    'controller' => $controllerIdentifier,
                    'error' => $e->getMessage()
                ]);
                return null;
            }
        }
        
        return null;
    }

    /**
     * Clear controller cache
     *
     * @return void
     */
    public function clearControllerCache(): void {
        $this->controllerCache = [];
    }
}