<?php
/**
 * Route Resolver - pronalazi i validira rute prema zahtjevima
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use \baseKRIZAN\Security\Authentication;
use \baseKRIZAN\Error\Logger;

class RouteResolver {
    /**
     * Authentication servis
     * @var Authentication
     */
    private Authentication $authentication;
    
    /**
     * Logger instanca
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Spremište za cache pattern matcheva
     * @var array
     */
    private array $patternMatchCache = [];
    
    /**
     * Maksimalna veličina cachea
     * @var int
     */
    private int $cacheSize = 100;

    /**
     * Konstruktor
     *
     * @param Authentication $authentication Auth servis za provjeru autentikacije
     * @param Logger $logger Logger za praćenje aktivnosti
     */
    public function __construct(Authentication $authentication, Logger $logger) {
        $this->authentication = $authentication;
        $this->logger = $logger;
        
        $this->logger->routing('RouteResolver initialized');
    }    

    /**
     * Pronalazi odgovarajuću rutu za zadanu putanju i metodu
     *
     * @param string $path Putanja zahtjeva
     * @param string $method HTTP metoda
     * @param array $routes Definirane rute
     * @return array|null Pronađeni podaci o ruti ili null ako nije pronađena
     */
    public function resolve(string $path, string $method, array $routes): ?array 
    {
        // Debug log to see what's in the routes array
        $this->logger->routing('Resolving route details', [
            'path' => $path,
            'method' => $method,
            'routes_count' => count($routes)
        ]);
        
        // Check cache first for pattern matches
        $cacheKey = $this->generatePatternCacheKey($path, $method);
        if (isset($this->patternMatchCache[$cacheKey])) {
            $this->logger->routing('Using cached pattern match result', [
                'path' => $path,
                'method' => $method
            ]);
            return $this->patternMatchCache[$cacheKey];
        }

        // Extract only routes with patterns for efficiency
        $patternRoutes = [];
        foreach ($routes as $routePath => $routeData) {
            if (isset($routeData['pattern']) && $routeData['pattern']) {
                $patternRoutes[$routePath] = $routeData;
            }
        }
        
        $this->logger->routing('Checking pattern routes', [
            'count' => count($patternRoutes)
        ]);
        
        foreach ($patternRoutes as $routePath => $routeData) {
            if (!isset($routeData[$method]) || !isset($routeData[$method]['handler'])) {
                continue;
            }
            
            // Try to match with pattern
            $params = $this->matchRoutePattern($routePath, $path);
            
            if ($params !== null) {
                // We have a match
                $this->logger->routing('Found pattern match', [
                    'pattern' => $routePath,
                    'path' => $path,
                    'params' => $params
                ]);
                
                $methodData = $routeData[$method];
                
                // Set the current path (not the pattern) for authentication check
                $routeData['path'] = $path;
                
                // Validate for authentication/permissions
                $validationResult = $this->validateRoute($routeData);
                if (isset($validationResult['redirect'])) {
                    $this->logger->routing('Route requires authentication or permissions, redirecting', [
                        'path' => $path,
                        'redirect' => $validationResult['redirect']
                    ]);
                    
                    // Cache the redirect result
                    $this->cachePatternMatch($cacheKey, $validationResult);
                    return $validationResult;
                }
                
                $result = [
                    'controller' => $methodData['handler']['controller'],
                    'action' => $methodData['handler']['action'],
                    'login' => $routeData['login'] ?? false,
                    'permissions' => $routeData['permissions'] ?? null,
                    'middleware' => $routeData['middleware'] ?? [],
                    'params' => $params
                ];
                
                $this->logger->routing('Route with parameters resolved to controller/action', [
                    'controller' => $result['controller'],
                    'action' => $result['action'],
                    'params' => $params
                ]);
                
                // Cache the successful result
                $this->cachePatternMatch($cacheKey, $result);
                return $result;
            }
        }

        // If we get here, no route was found - cache the negative result too
        $this->cachePatternMatch($cacheKey, null);
        $this->logger->routing('No route match found', ['path' => $path, 'method' => $method]);
        return null;
    }
    
    /**
     * Spremi rezultat matchanja pattern rute u cache
     *
     * @param string $cacheKey Ključ za cache
     * @param array|null $result Rezultat matchanja
     * @return void
     */
    private function cachePatternMatch(string $cacheKey, ?array $result): void {
        // Maintain cache size - implementiran LRU (Least Recently Used) pristup
        if (count($this->patternMatchCache) >= $this->cacheSize) {
            // Remove oldest item
            array_shift($this->patternMatchCache);
        }
        
        // Add to cache
        $this->patternMatchCache[$cacheKey] = $result;
    }
    
    /**
     * Generira ključ za cache pattern matcheva
     *
     * @param string $path Putanja
     * @param string $method HTTP metoda
     * @return string Ključ za cache
     */
    private function generatePatternCacheKey(string $path, string $method): string {
        return 'pattern:' . $method . ':' . $path;
    }
    
    /**
     * Postavi maksimalnu veličinu cahcea za pattern
     *
     * @param int $size Nova veličina cachea
     * @return void
     */
    public function setCacheSize(int $size): void {
        if ($size > 0) {
            $this->cacheSize = $size;
        }
    }
    
    /**
     * Čisti cache pattern matcheva
     *
     * @return void
     */
    public function clearPatternCache(): void {
        $this->patternMatchCache = [];
    }
    
    /**
     * Match a route path pattern against an actual path
     * 
     * @param string $routePath The route pattern to match
     * @param string $path The actual path to check
     * @return array|null Parameter values if matched, null otherwise
     */
    public function matchRoutePattern(string $routePath, string $path): ?array
    {
        $routeParts = explode('/', $routePath);
        $pathParts = explode('/', $path);
        
        // Brza provjera - broj dijelova se mora poklapati osim ako imamo opcionalne parametre
        if (count($routeParts) !== count($pathParts) && !$this->hasOptionalParameters($routePath)) {
            return null;
        }
        
        $params = [];
        
        // Compare each part
        for ($i = 0; $i < count($routeParts); $i++) {
            // If we're past the end of the path but the route has more parts, check if they're optional
            if (!isset($pathParts[$i])) {
                if ($this->isOptionalParameter($routeParts[$i])) {
                    // Skip this optional parameter
                    continue;
                }
                // Non-optional parameter but no matching path part - no match
                return null;
            }
            
            // Check if this part is a parameter
            if ($this->isParameter($routeParts[$i])) {
                // Extract parameter name and any type constraints
                [$paramName, $paramType] = $this->extractParameterInfo($routeParts[$i]);
                
                // Validate parameter type if specified
                if ($paramType && !$this->validateParameterType($pathParts[$i], $paramType)) {
                    return null;
                }
                
                // Add to parameters
                $params[$paramName] = $pathParts[$i];
                continue;
            }
            
            // Regular string comparison
            if ($routeParts[$i] !== $pathParts[$i]) {
                return null;
            }
        }
        
        // All parts matched
        return $params;
    }
    
    /**
     * Check if a route path has optional parameters
     * 
     * @param string $path Route path to check
     * @return bool True if has optional parameters
     */
    private function hasOptionalParameters(string $path): bool
    {
        return strpos($path, '?') !== false;
    }
    
    /**
     * Check if a segment is an optional parameter
     * 
     * @param string $segment Route segment to check
     * @return bool True if is an optional parameter
     */
    private function isOptionalParameter(string $segment): bool
    {
        return $this->isParameter($segment) && strpos($segment, '?') !== false;
    }
    
    /**
     * Check if a segment is a parameter
     * 
     * @param string $segment Route segment to check
     * @return bool True if is a parameter
     */
    private function isParameter(string $segment): bool
    {
        return strpos($segment, '{') === 0 && strrpos($segment, '}') === strlen($segment) - 1;
    }
    
    /**
     * Extract parameter name and type from segment
     * 
     * @param string $segment Route segment
     * @return array Array with [name, type]
     */
    private function extractParameterInfo(string $segment): array
    {
        // Remove braces
        $content = substr($segment, 1, -1);
        
        // Remove optional marker
        $content = str_replace('?', '', $content);
        
        // Check for type constraint
        if (strpos($content, ':') !== false) {
            [$name, $type] = explode(':', $content, 2);
            return [$name, $type];
        }
        
        return [$content, null];
    }
    
    /**
     * Validate parameter value against type constraint
     * 
     * @param string $value Parameter value
     * @param string $type Type constraint
     * @return bool True if valid
     */
    private function validateParameterType(string $value, string $type): bool
    {
        switch ($type) {
            case 'number':
                return is_numeric($value);
            case 'alpha':
                return ctype_alpha($value);
            case 'alnum':
                return ctype_alnum($value);
            case 'slug':
                return preg_match('/^[a-z0-9-]+$/', $value) === 1;
            case 'uuid':
                return preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $value) === 1;
            case 'date':
                return strtotime($value) !== false;
            default:
                // Koristimo regularne izraze za custom tipove
                if (substr($type, 0, 7) === 'regex:') {
                    $pattern = substr($type, 7);
                    return (bool)preg_match($pattern, $value);
                }
                return true; // No validation for unknown types
        }
    }

    /**
     * Validate route for authentication and permissions
     * 
     * @param array $route Route data
     * @return array Validated route or redirect info
     */
    public function validateRoute(array $route): array {
        // Check if global authentication is disabled
        $authEnabled = \baseKRIZAN\Config\Config::get('authentication.enabled', true);
        
        if (!$authEnabled) {
            // Skip authentication checks if globally disabled
            $this->logger->routing('Route authentication check skipped (globally disabled)', [
                'path' => $route['path'] ?? 'unknown'
            ]);
            return $route;
        }
        
        // Check if this route explicitly doesn't require auth
        if (isset($route['auth']) && $route['auth'] === false) {
            $this->logger->routing('Route does not require authentication', [
                'path' => $route['path'] ?? 'unknown'
            ]);
            return $route; // Skip authentication check
        }
        
        // Default to requiring login if 'login' is set to true
        if (isset($route['login']) && $route['login'] && !$this->authentication->isLoggedIn()) {
            $this->logger->routing('Route requires authentication but user is not logged in', [
                'path' => $route['path'] ?? 'unknown'
            ]);
            return ['redirect' => 'login/error'];
        }
    
        // Check permissions if they are specified
        if (isset($route['permissions']) && !$this->checkPermissions($route['permissions'])) {
            $this->logger->routing('User lacks required permissions for route', [
                'path' => $route['path'] ?? 'unknown',
                'required_permissions' => $route['permissions']
            ]);
            return ['redirect' => 'user/permissionserror'];
        }
    
        $this->logger->routing('Route validation successful', [
            'path' => $route['path'] ?? 'unknown'
        ]);
        return $route;
    }

    /**
     * Check if current user has required permissions
     * 
     * @param mixed $permissions Required permissions
     * @return bool True if has permission
     */
    public function checkPermissions($permissions): bool {
        $user = $this->authentication->getUser();
        if (!$user) return false;

        // Za optimizaciju, koristimo is_array umjesto (array) cast
        if (is_array($permissions)) {
            foreach ($permissions as $permission) {
                if ($user->hasPermission($permission)) {
                    return true;
                }
            }
        } else {
            // Jednostavna provjera za pojedinačnu dozvolu
            return $user->hasPermission($permissions);
        }
        
        return false;
    }
}