<?php
/**
 * Router - implementacija Routes interfacea
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Routing\RouteCollection;
use baseKRIZAN\Routing\RouteResolver;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Error\Logger;

class Router implements Routes 
{
    /**
     * Kolekcija ruta
     * @var RouteCollection
     */
    private RouteCollection $routeCollection;
    
    /**
     * Resolver za rute
     * @var RouteResolver
     */
    private RouteResolver $routeResolver;
    
    /**
     * Auth servis
     * @var Authentication
     */
    private Authentication $authentication;
    
    /**
     * Logger
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Container za servise
     * @var Container|null
     */
    private ?Container $container = null;
    
    /**
     * Cache za imena ruta
     * @var array
     */
    private array $namedRoutesCache = [];

    /**
     * Konstruktor
     *
     * @param Logger $logger Logger instanca
     * @param Container|null $container DI container
     */
    public function __construct(Logger $logger, Container $container = null) 
    {
        $this->logger = $logger;
        $this->container = $container ?? new Container();
        
        $this->logger->routing('Router initialized', [
            'container_exists' => $container ? 'yes' : 'no'
        ]);
        
        // Always expect authentication from container
        if (!$this->container->has('authentication')) {
            $this->logger->error('Authentication service not found in container');
            throw new \RuntimeException('Authentication service not found in container');
        }
        
        $this->authentication = $this->container->get('authentication');
        $this->routeCollection = new RouteCollection($logger);
        $this->routeResolver = new RouteResolver($this->authentication, $logger);
    }

    /**
     * Postavlja rute iz niza
     *
     * @param array $routes Rute
     * @return void
     */
    public function setRoutes(array $routes): void {
        // Resetiraj postojeće rute
        $this->routeCollection = new RouteCollection($this->logger);
        
        $this->logger->routing('Setting routes from array', [
            'route_count' => count($routes)
        ]);
        
        $routeCount = 0;
        
        foreach ($routes as $path => $routeData) {
            foreach (['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'] as $method) {
                if (isset($routeData[$method])) {
                    // Normaliziraj putanju
                    $normalizedPath = trim($path, '/');
                    
                    $routeConfig = [
                        'handler' => [
                            'controller' => $routeData[$method]['controller'],
                            'action' => $routeData[$method]['action']
                        ]
                    ];
                    
                    $options = [
                        'login' => $routeData['login'] ?? false,
                        'auth' => $routeData['auth'] ?? null, 
                        'permissions' => $routeData['permissions'] ?? null,
                        'middleware' => $routeData['middleware'] ?? [],
                        'name' => $routeData['name'] ?? null,
                        'pattern' => isset($routeData['pattern']) && $routeData['pattern'] === true
                    ];
                    
                    // Cache named route if name is set
                    if (!empty($options['name'])) {
                        $this->namedRoutesCache[$options['name']] = $normalizedPath;
                    }
                    
                    // Add to route collection
                    $this->routeCollection->add($normalizedPath, $method, $routeConfig, $options);
                    $routeCount++;
                }
            }
        }
        
        $this->logger->routing("Routes registered successfully", [
            'total_route_count' => $routeCount,
            'registered_routes' => array_keys($routes)
        ]);
    }

    /**
     * Dohvaća sve rute
     *
     * @return array Rute
     */
    public function getRoutes(): array 
    {
        return $this->routeCollection->getRoutes();
    }

    /**
     * Dohvaća authentication servis
     *
     * @return Authentication Auth servis
     */
    public function getAuthentication(): Authentication 
    {
        return $this->authentication;
    }

    /**
     * Provjerava ima li korisnik tražene dozvole
     *
     * @param mixed $permission Dozvola za provjeru
     * @return bool True ako korisnik ima dozvolu
     */
    public function checkPermission($permission): bool 
    {
        return $this->routeResolver->checkPermissions($permission);
    }

    /**
     * Pronalazi i vraća rutu prema zahtjevu
     *
     * @param string $path Putanja zahtjeva
     * @param string $method HTTP metoda
     * @return array|null Pronađena ruta ili null
     */
    public function resolveRoute(string $path, string $method): ?array 
    {
        $path = trim($path, '/');
        
        $this->logger->routing('Resolving route', [
            'path' => $path,
            'method' => $method,
            'all_routes' => array_keys($this->routeCollection->getRoutes())
        ]);
        
        // First check if we can get the route directly from collection
        $routeData = $this->routeCollection->getRoute($path, $method);
        
        if ($routeData) {
            // We found exact match, now validate it
            $routeData['path'] = $path;
            
            // Log detailed route match
            $this->logger->routing('Exact route match found', [
                'path' => $path,
                'method' => $method,
                'route_details' => $routeData
            ]);
            
            // Validate authentication/permissions
            $validationResult = $this->routeResolver->validateRoute($routeData);
            if (isset($validationResult['redirect'])) {
                return $validationResult;
            }
            
            // SAFETY CHECK: Make sure method and handler data exist
            if (!isset($routeData[$method]) || !isset($routeData[$method]['handler'])) {
                $this->logger->routing('Method handler data not found for route', [
                    'path' => $path,
                    'method' => $method
                ]);
                return null;
            }
            
            // SAFETY CHECK: Make sure controller and action exist
            if (!isset($routeData[$method]['handler']['controller']) || !isset($routeData[$method]['handler']['action'])) {
                $this->logger->routing('Controller or action not found for route', [
                    'path' => $path,
                    'method' => $method
                ]);
                return null;
            }
            
            // Create the result with safety checks for all properties
            $methodData = $routeData[$method];
            $result = [
                'controller' => $methodData['handler']['controller'],
                'action' => $methodData['handler']['action'],
                'login' => $routeData['login'] ?? false,
                'permissions' => $routeData['permissions'] ?? null,
                'middleware' => $routeData['middleware'] ?? []
            ];
            
            $this->logger->routing('Route resolved successfully (direct match)', [
                'path' => $path,
                'controller' => $result['controller'],
                'action' => $result['action']
            ]);
            
            return $result;
        }
        
        // If no direct match, try pattern matching through resolver
        $result = $this->routeResolver->resolve($path, $method, $this->getRoutes());
        
        if ($result) {
            // SAFETY CHECK: Make sure controller and action exist
            if (!isset($result['controller']) || !isset($result['action'])) {
                $this->logger->routing('Controller or action not found in resolved pattern route', [
                    'path' => $path,
                    'method' => $method
                ]);
                return null;
            }
            
            $this->logger->routing('Route resolved successfully (pattern match)', [
                'path' => $path,
                'controller' => $result['controller'],
                'action' => $result['action']
            ]);
        } else {
            $this->logger->routing('Route not found', [
                'path' => $path,
                'method' => $method
            ]);
        }
        
        return $result;
    }

    /**
     * Dodaje grupu middleware komponenti
     *
     * @param string $name Naziv grupe
     * @param array $middleware Komponente
     * @return void
     */
    public function addMiddlewareGroup(string $name, array $middleware): void 
    {
        $this->routeCollection->addMiddlewareGroup($name, $middleware);
    }

    /**
     * Dohvaća middleware grupe
     *
     * @return array Middleware grupe
     */
    public function getMiddlewareGroups(): array 
    {
        return $this->routeCollection->getMiddlewareGroups();
    }

    /**
     * Postavlja container za servise
     *
     * @param Container $container DI container
     * @return void
     */
    public function setContainer(Container $container): void
    {
        $this->container = $container;
        
        // If container has authentication, use it
        if ($this->container->has('authentication')) {
            $this->authentication = $this->container->get('authentication');
            $this->routeResolver = new RouteResolver($this->authentication, $this->logger);
        }
    }
    
    /**
     * Dohvaća URL za imenovanu rutu
     *
     * @param string $name Ime rute
     * @param array $params Parametri za zamjenu u putanji
     * @return string|null URL ili null ako nije pronađeno
     */
    public function getRouteUrl(string $name, array $params = []): ?string
    {
        // Check cache first
        if (isset($this->namedRoutesCache[$name])) {
            $path = $this->namedRoutesCache[$name];
        } else {
            // Look for route by name
            $route = $this->routeCollection->getRouteByName($name);
            if (!$route) {
                return null;
            }
            $path = $route['path'];
            
            // Cache this route
            $this->namedRoutesCache[$name] = $path;
        }
        
        // Replace parameters in path
        foreach ($params as $key => $value) {
            $path = str_replace('{' . $key . '}', $value, $path);
            $path = str_replace('{' . $key . '?}', $value, $path); // Optional params
        }
        
        // Get base URL
        $baseUrl = \baseKRIZAN\Config\Config::get('paths.app_url');
        
        return $baseUrl . '/' . $path;
    }
    
    /**
     * Čisti cache ruta
     *
     * @return void
     */
    public function clearRouteCache(): void
    {
        $this->routeCollection->clearRouteCache();
        $this->routeResolver->clearPatternCache();
        $this->namedRoutesCache = [];
    }
    
    /**
     * Validira i priprema rutu bez dohvata kontrolera
     *
     * @param string $path Putanja
     * @param string $method HTTP metoda
     * @return array|null Route data or null if not found
     */
    public function validateRoute(string $path, string $method): ?array
    {
        $path = trim($path, '/');
        
        // First try to get route from collection
        $routeData = $this->routeCollection->getRoute($path, $method);
        
        if ($routeData) {
            // Validate route
            $routeData['path'] = $path;
            return $this->routeResolver->validateRoute($routeData);
        }
        
        // Try pattern routes
        $patternRoutes = $this->routeCollection->getPatternRoutes();
        
        foreach ($patternRoutes as $routePath => $routeData) {
            if (!isset($routeData[$method])) {
                continue;
            }
            
            $params = $this->routeResolver->matchRoutePattern($routePath, $path);
            if ($params !== null) {
                $routeData['path'] = $path;
                $routeData['params'] = $params;
                return $this->routeResolver->validateRoute($routeData);
            }
        }
        
        return $this->resolveRoute($path, $method);
    }

    /**
     * Pretprocesira rutu - vraća samo osnovne podatke o ruti bez validacije i izvršavanja
     *
     * @param string $path Putanja zahtjeva
     * @param string $method HTTP metoda
     * @return array|null Osnovni podaci o ruti ili null ako nije pronađena
     */
    public function preProcessRoute(string $path, string $method): ?array
    {
        $path = trim($path, '/');
        
        $this->logger->routing('Pre-processing route', [
            'path' => $path,
            'method' => $method
        ]);
        
        // Prvo pokušaj direktno dohvatiti rutu iz kolekcije
        $routeData = $this->routeCollection->getRoute($path, $method);
        
        if ($routeData) {
            // Vrati samo osnovne podatke o ruti
            return [
                'middleware' => $routeData['middleware'] ?? [],
                'path' => $path
            ];
        }
        
        // Ako nema direktnog poklapanja, pokušaj s pattern rutama
        $patternRoutes = $this->routeCollection->getPatternRoutes();
        
        foreach ($patternRoutes as $routePath => $routeInfo) {
            if (!isset($routeInfo[$method])) {
                continue;
            }
            
            // Provjeri pattern matching
            $params = $this->routeResolver->matchRoutePattern($routePath, $path);
            if ($params !== null) {
                return [
                    'middleware' => $routeInfo['middleware'] ?? [],
                    'path' => $path,
                    'params' => $params
                ];
            }
        }
        
        return null;
    }
}