<?php
/**
 * URL Generator - pomoćna klasa za generiranje URL-ova
 *
 * @package baseKRIZAN\Routing
 * @author KRIZAN
 */

namespace baseKRIZAN\Routing;

use baseKRIZAN\Error\Logger;

class UrlGenerator
{
    /**
     * Routes instanca
     * @var Routes
     */
    private Routes $routes;
    
    /**
     * Logger instanca
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Base URL
     * @var string
     */
    private string $baseUrl;
    
    /**
     * Asset URL
     * @var string
     */
    private string $assetUrl;
    
    /**
     * Konstruktor
     *
     * @param Routes $routes Routes instanca
     * @param Logger|null $logger Logger instanca
     */
    public function __construct(Routes $routes, ?Logger $logger = null)
    {
        $this->routes = $routes;
        $this->logger = $logger;
        
        // Initialize URLs from config
        $this->baseUrl = rtrim(\baseKRIZAN\Config\Config::get('paths.app_url', ''), '/');
        $this->assetUrl = rtrim(\baseKRIZAN\Config\Config::get('paths.app_url', ''), '/');
    }
    
    /**
     * Generira URL za rutu po imenu
     *
     * @param string $routeName Ime rute
     * @param array $params Parametri za zamjenu u putanji
     * @param array $queryParams GET parametri
     * @return string URL
     */
    public function route(string $routeName, array $params = [], array $queryParams = []): string
    {
        $url = $this->routes->getRouteUrl($routeName, $params);
        
        if ($url === null) {
            if ($this->logger) {
                $this->logger->routing('Route not found by name', [
                    'name' => $routeName
                ]);
            }
            
            // Fallback - consider the name as a direct path
            $url = $this->baseUrl . '/' . ltrim($routeName, '/');
        }
        
        // Add query parameters if any
        if (!empty($queryParams)) {
            $url .= '?' . http_build_query($queryParams);
        }
        
        return $url;
    }
    
    /**
     * Generira URL za asset
     *
     * @param string $path Putanja do asseta
     * @param bool $versioned Dodaj verziju (cache busting)
     * @return string URL asseta
     */
    public function asset(string $path, bool $versioned = true): string
    {
        $path = ltrim($path, '/');
        $url = $this->assetUrl . '/' . $path;
        
        // Check if we need to add versioning (cache busting)
        if ($versioned) {
            // Try to get the actual file path
            $localPath = \baseKRIZAN\Config\Config::get('paths.public') . '/' . $path;
            
            if (file_exists($localPath)) {
                // Add file modification time as version
                $mtime = filemtime($localPath);
                $url .= '?v=' . $mtime;
            } else {
                // Fallback to current time if file not found
                $url .= '?v=' . time();
            }
        }
        
        return $url;
    }
    
    /**
     * Generira apsolutni URL s današnjim domenom
     *
     * @param string $path Putanja
     * @return string Potpuni URL
     */
    public function absoluteUrl(string $path): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        return $protocol . '://' . $host . '/' . ltrim($path, '/');
    }
    
    /**
     * Generira URL za trenutnu stranicu s izmijenjenim parametrima
     *
     * @param array $newParams Novi GET parametri
     * @param array $removeParams Parametri za ukloniti
     * @return string URL
     */
    public function current(array $newParams = [], array $removeParams = []): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        
        // Split URI into path and query
        $parts = explode('?', $uri, 2);
        $path = $parts[0];
        $query = $parts[1] ?? '';
        
        // Parse query string
        parse_str($query, $params);
        
        // Remove specified parameters
        foreach ($removeParams as $param) {
            unset($params[$param]);
        }
        
        // Add/replace new parameters
        $params = array_merge($params, $newParams);
        
        // Build new query string
        $newQuery = !empty($params) ? '?' . http_build_query($params) : '';
        
        return $protocol . '://' . $host . $path . $newQuery;
    }
    
    /**
     * Generira URL s parametrima kao segmentima putanje
     *
     * @param string $basePath Osnovna putanja
     * @param array $segments Segmenti putanje
     * @return string URL
     */
    public function segments(string $basePath, array $segments): string
    {
        $basePath = rtrim($basePath, '/');
        
        // Filter out null or empty segments
        $segments = array_filter($segments, function($segment) {
            return $segment !== null && $segment !== '';
        });
        
        // Encode each segment for URL
        $encodedSegments = array_map('urlencode', $segments);
        
        // Combine path
        $path = $basePath . '/' . implode('/', $encodedSegments);
        
        return $this->baseUrl . '/' . ltrim($path, '/');
    }
}