<?php
// app/baseKRIZAN/Security/ApiAuth.php

namespace baseKRIZAN\Security;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Cache\CacheInterface;
use baseKRIZAN\Cache\CacheManager;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Service for API authentication with token management
 */
class ApiAuth
{
    /**
     * @var DatabaseConnection
     */
    private DatabaseConnection $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * @var CacheInterface|null
     */
    private ?CacheInterface $cache;
    
    /**
     * @var SessionManager|null
     */
    private ?SessionManager $sessionManager;
    
    /**
     * Database table for storing tokens
     * 
     * @var string
     */
    private string $tokenTable = 'api_tokens';
    
    /**
     * Token expiration time in seconds (default: 7 days)
     * 
     * @var int
     */
    private int $tokenExpiration = 604800;
    
    /**
     * Default scopes for tokens
     * 
     * @var array
     */
    private array $defaultScopes = ['read'];
    
    /**
     * Cache enabled flag
     * 
     * @var bool
     */
    private bool $cacheEnabled = true;
    
    /**
     * Cache TTL in seconds (default: 5 minutes)
     * 
     * @var int
     */
    private int $cacheTtl = 300;
    
    /**
     * Constructor
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param Logger|null $logger Logger instance
     * @param CacheInterface|null $cache Cache interface
     * @param SessionManager|null $sessionManager Session manager
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        ?Logger $logger = null, 
        ?CacheInterface $cache = null,
        ?SessionManager $sessionManager = null
    ) {
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        $this->sessionManager = $sessionManager;
        
        // Initialize cache
        if ($cache !== null) {
            $this->cache = $cache;
        } else {
            try {
                $this->cache = CacheManager::getInstance($logger)->driver();
            } catch (\Throwable $e) {
                $this->cacheEnabled = false;
                if ($this->logger) {
                    $this->logger->security('ApiAuth operating without cache', [
                        'reason' => $e->getMessage()
                    ]);
                }
            }
        }
        
        // Ensure token table exists
        $this->ensureTokenTableExists();
        
        // Load configuration
        $this->loadConfiguration();
        
        if ($this->logger) {
            $this->logger->security('ApiAuth service initialized', [
                'cache_enabled' => $this->cacheEnabled ? 'yes' : 'no',
                'token_expiration' => $this->tokenExpiration
            ]);
        }
    }
    
    /**
     * Load configuration from config
     * 
     * @return void
     */
    private function loadConfiguration(): void
    {
        // Token table
        $configTokenTable = \baseKRIZAN\Config\Config::get('api_auth.token_table');
        if ($configTokenTable) {
            $this->tokenTable = $configTokenTable;
        }
        
        // Token expiration
        $configTokenExpiration = \baseKRIZAN\Config\Config::get('api_auth.token_expiration');
        if ($configTokenExpiration) {
            $this->tokenExpiration = (int)$configTokenExpiration;
        }
        
        // Default scopes
        $configDefaultScopes = \baseKRIZAN\Config\Config::get('api_auth.default_scopes');
        if ($configDefaultScopes) {
            if (is_string($configDefaultScopes)) {
                $this->defaultScopes = explode(',', $configDefaultScopes);
            } elseif (is_array($configDefaultScopes)) {
                $this->defaultScopes = $configDefaultScopes;
            }
        }
        
        // Cache settings
        $configCacheEnabled = \baseKRIZAN\Config\Config::get('api_auth.cache_enabled');
        if ($configCacheEnabled !== null) {
            $this->cacheEnabled = (bool)$configCacheEnabled;
        }
        
        $configCacheTtl = \baseKRIZAN\Config\Config::get('api_auth.cache_ttl');
        if ($configCacheTtl) {
            $this->cacheTtl = (int)$configCacheTtl;
        }
    }
    
    /**
     * Ensure token table exists
     * 
     * @return void
     */
    private function ensureTokenTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $this->tokenTable]
            );
            
            if ($tableExists === null) {
                // Create table with truncated index lengths
                $sql = "CREATE TABLE `{$this->tokenTable}` (
                    `id` VARCHAR(64) PRIMARY KEY,
                    `user_id` INT NOT NULL,
                    `token` VARCHAR(255) NOT NULL,
                    `scopes` TEXT,
                    `expires_at` INT UNSIGNED NOT NULL,
                    `last_used_at` INT UNSIGNED,
                    `created_at` INT UNSIGNED NOT NULL,
                    `revoked` TINYINT(1) NOT NULL DEFAULT 0,
                    `description` VARCHAR(255),
                    `client_ip` VARCHAR(45),
                    `user_agent` VARCHAR(255),
                    UNIQUE KEY `uk_token` (`token`(191)),
                    INDEX `idx_user_id` (`user_id`),
                    INDEX `idx_expires` (`expires_at`),
                    INDEX `idx_revoked` (`revoked`)
                ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->security('Created API token table', [
                        'table' => $this->tokenTable
                    ]);
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureTokenTableExists'
                ]);
            }
            // Re-throw database errors
            throw $e;
        }
    }
    
    /**
     * Generate a new API token for a user
     * 
     * @param int $userId User ID
     * @param array|null $scopes Token scopes
     * @param string|null $description Token description
     * @param int|null $expiresIn Expiration time in seconds
     * @param array|null $clientInfo Client information
     * @return array|null Token data or null on failure
     */
    public function generateToken(
        int $userId, 
        ?array $scopes = null, 
        ?string $description = null,
        ?int $expiresIn = null,
        ?array $clientInfo = null
    ): ?array {
        try {
            // Generate unique token ID and value
            $tokenId = $this->generateUniqueId();
            $tokenValue = $this->generateTokenValue();
            
            // Set token expiration
            $now = time();
            $expiresAt = $now + ($expiresIn ?? $this->tokenExpiration);
            
            // Prepare scopes
            $tokenScopes = $scopes ?? $this->defaultScopes;
            $scopesJson = json_encode($tokenScopes);
            
            // Extract client info
            $clientIp = $clientInfo['ip'] ?? $_SERVER['REMOTE_ADDR'] ?? null;
            $userAgent = $clientInfo['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? null;
            
            // Insert token into database
            $sql = "
                INSERT INTO `{$this->tokenTable}` 
                (id, user_id, token, scopes, expires_at, created_at, description, client_ip, user_agent)
                VALUES 
                (:id, :user_id, :token, :scopes, :expires_at, :created_at, :description, :client_ip, :user_agent)
            ";
            
            $this->dbConnection->execute($sql, [
                'id' => $tokenId,
                'user_id' => $userId,
                'token' => $tokenValue,
                'scopes' => $scopesJson,
                'expires_at' => $expiresAt,
                'created_at' => $now,
                'description' => $description,
                'client_ip' => $clientIp,
                'user_agent' => $userAgent
            ]);
            
            $tokenData = [
                'id' => $tokenId,
                'token' => $tokenValue,
                'user_id' => $userId,
                'scopes' => $tokenScopes,
                'expires_at' => $expiresAt,
                'created_at' => $now
            ];
            
            if ($this->logger) {
                $this->logger->security('Generated API token', [
                    'token_id' => $tokenId,
                    'user_id' => $userId,
                    'scopes' => implode(',', $tokenScopes),
                    'expires_at' => date('Y-m-d H:i:s', $expiresAt)
                ]);
            }
            
            return $tokenData;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'generateToken'
                ]);
            }
            return null;
        }
    }
    
    /**
     * Validate an API token
     * 
     * @param string $token Token to validate
     * @param array|null $requiredScopes Required scopes
     * @return array|null User data or null if invalid
     */
    public function validateToken(string $token, ?array $requiredScopes = null): ?array
    {
        // Try to get from cache first
        $cacheKey = 'api_auth:token:' . hash('sha256', $token);
        if ($this->cacheEnabled && $this->cache) {
            $cachedData = $this->cache->get($cacheKey);
            if ($cachedData !== null) {
                // Check if token is revoked or expired in cache
                if ($cachedData === 'revoked' || $cachedData === 'expired') {
                    return null;
                }
                
                // Check required scopes
                if ($requiredScopes && !$this->hasScopes($cachedData['scopes'], $requiredScopes)) {
                    return null;
                }
                
                return $cachedData;
            }
        }
        
        try {
            // Get token from database
            $sql = "
                SELECT t.*, u.id as user_id, u.username, u.email
                FROM `{$this->tokenTable}` t
                JOIN `users` u ON t.user_id = u.id
                WHERE t.token = :token
                AND t.revoked = 0
                AND t.expires_at > :now
            ";
            
            $tokenData = $this->dbConnection->queryAndFetchAssoc($sql, [
                'token' => $token,
                'now' => time()
            ]);
            
            if (!$tokenData) {
                // Check if token exists but is invalid
                $invalidTokenData = $this->dbConnection->queryAndFetchAssoc(
                    "SELECT revoked, expires_at FROM `{$this->tokenTable}` WHERE token = :token",
                    ['token' => $token]
                );
                
                if ($invalidTokenData) {
                    // Cache the reason why it's invalid
                    if ($this->cacheEnabled && $this->cache) {
                        $reason = (int)$invalidTokenData['revoked'] === 1 ? 'revoked' : 'expired';
                        $this->cache->set($cacheKey, $reason, $this->cacheTtl);
                    }
                }
                
                return null;
            }
            
            // Decode scopes
            $scopes = json_decode($tokenData['scopes'] ?? '[]', true) ?: [];
            
            // Check required scopes
            if ($requiredScopes && !$this->hasScopes($scopes, $requiredScopes)) {
                return null;
            }
            
            // Update last_used_at
            $this->updateLastUsed($tokenData['id']);
            
            // Prepare user data
            $userData = [
                'user_id' => $tokenData['user_id'],
                'username' => $tokenData['username'],
                'email' => $tokenData['email'],
                'token_id' => $tokenData['id'],
                'scopes' => $scopes,
                'expires_at' => (int)$tokenData['expires_at']
            ];
            
            // Cache the result
            if ($this->cacheEnabled && $this->cache) {
                $this->cache->set($cacheKey, $userData, $this->cacheTtl);
            }
            
            return $userData;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'validateToken'
                ]);
            }
            return null;
        }
    }
    
    /**
     * Revoke an API token
     * 
     * @param string $tokenId Token ID to revoke
     * @param int|null $userId User ID (for authorization check)
     * @return bool Success status
     */
    public function revokeToken(string $tokenId, ?int $userId = null): bool
    {
        try {
            $query = "UPDATE `{$this->tokenTable}` SET revoked = 1 WHERE id = :id";
            $params = ['id' => $tokenId];
            
            // If user ID is provided, only revoke if the token belongs to this user
            if ($userId !== null) {
                $query .= " AND user_id = :user_id";
                $params['user_id'] = $userId;
            }
            
            $affectedRows = $this->dbConnection->execute($query, $params);
            
            $success = $affectedRows > 0;
            
            if ($success && $this->logger) {
                $this->logger->security('Revoked API token', [
                    'token_id' => $tokenId,
                    'user_id' => $userId ?? 'admin'
                ]);
            }
            
            return $success;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'revokeToken'
                ]);
            }
            return false;
        }
    }
    
    /**
     * Revoke all tokens for a user
     * 
     * @param int $userId User ID
     * @return int Number of tokens revoked
     */
    public function revokeAllUserTokens(int $userId): int
    {
        try {
            $sql = "
                UPDATE `{$this->tokenTable}` 
                SET revoked = 1 
                WHERE user_id = :user_id 
                AND revoked = 0
            ";
            
            $affectedRows = $this->dbConnection->execute($sql, ['user_id' => $userId]);
            
            if ($affectedRows > 0 && $this->logger) {
                $this->logger->security('Revoked all API tokens for user', [
                    'user_id' => $userId,
                    'count' => $affectedRows
                ]);
            }
            
            return $affectedRows;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'revokeAllUserTokens'
                ]);
            }
            return 0;
        }
    }
    
    /**
     * Get all active tokens for a user
     * 
     * @param int $userId User ID
     * @return array Array of token data
     */
    public function getUserTokens(int $userId): array
    {
        try {
            $sql = "
                SELECT id, scopes, expires_at, last_used_at, created_at, description, client_ip, user_agent 
                FROM `{$this->tokenTable}` 
                WHERE user_id = :user_id 
                AND revoked = 0 
                AND expires_at > :now
                ORDER BY created_at DESC
            ";
            
            $rows = $this->dbConnection->queryAndFetchAllAssoc($sql, [
                'user_id' => $userId,
                'now' => time()
            ]);
            
            $tokens = [];
            
            foreach ($rows as $token) {
                // Decode scopes
                $token['scopes'] = json_decode($token['scopes'] ?? '[]', true) ?: [];
                
                // Format dates
                $token['expires_at'] = (int)$token['expires_at'];
                $token['created_at'] = (int)$token['created_at'];
                $token['last_used_at'] = $token['last_used_at'] ? (int)$token['last_used_at'] : null;
                
                $tokens[] = $token;
            }
            
            return $tokens;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'getUserTokens'
                ]);
            }
            return [];
        }
    }
    
    /**
     * Generate a unique token ID
     * 
     * @return string Unique ID
     */
    private function generateUniqueId(): string
    {
        return bin2hex(random_bytes(16)); // 32 characters
    }
    
    /**
     * Generate a secure token value
     * 
     * @return string Token value
     */
    private function generateTokenValue(): string
    {
        return bin2hex(random_bytes(32)); // 64 characters
    }
    
    /**
     * Update the last_used_at timestamp for a token
     * 
     * @param string $tokenId Token ID
     * @return void
     */
    private function updateLastUsed(string $tokenId): void
    {
        try {
            $sql = "
                UPDATE `{$this->tokenTable}` 
                SET last_used_at = :now 
                WHERE id = :id
            ";
            
            $this->dbConnection->execute($sql, [
                'id' => $tokenId,
                'now' => time()
            ]);
        } catch (\PDOException $e) {
            // Just log the error, don't fail the request
            if ($this->logger) {
                $this->logger->error('Failed to update token last_used_at', [
                    'error' => $e->getMessage(),
                    'token_id' => $tokenId
                ]);
            }
        }
    }
    
    /**
     * Check if a token has the required scopes
     * 
     * @param array $tokenScopes Token scopes
     * @param array $requiredScopes Required scopes
     * @return bool True if token has all required scopes
     */
    private function hasScopes(array $tokenScopes, array $requiredScopes): bool
    {
        // Check if the token has all the required scopes
        foreach ($requiredScopes as $scope) {
            if (!in_array($scope, $tokenScopes)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Clean up expired tokens
     * 
     * @return int Number of tokens cleaned up
     */
    public function cleanupExpiredTokens(): int
    {
        try {
            $sql = "DELETE FROM `{$this->tokenTable}` WHERE expires_at < :now";
            
            $affectedRows = $this->dbConnection->execute($sql, ['now' => time()]);
            
            if ($affectedRows > 0 && $this->logger) {
                $this->logger->security('Cleaned up expired API tokens', [
                    'count' => $affectedRows
                ]);
            }
            
            return $affectedRows;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'cleanupExpiredTokens'
                ]);
            }
            return 0;
        }
    }
    
    /**
     * Generate a personal access token for the currently logged in user
     * 
     * @param string $description Token description
     * @param array|null $scopes Token scopes
     * @param int|null $expiresIn Expiration time in seconds
     * @return array|null Token data or null on failure
     */
    public function generatePersonalAccessToken(
        string $description, 
        ?array $scopes = null,
        ?int $expiresIn = null
    ): ?array {
        // Check if session manager and active session exists
        if (!$this->sessionManager || !$this->sessionManager->isStarted()) {
            if ($this->logger) {
                $this->logger->error('Cannot generate personal access token: No active session');
            }
            return null;
        }
        
        // Get user ID from session
        $userId = $this->sessionManager->get('user_id');
        if (!$userId) {
            if ($this->logger) {
                $this->logger->error('Cannot generate personal access token: User not logged in');
            }
            return null;
        }
        
        // Get client info
        $clientInfo = [
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
        ];
        
        // Generate token
        return $this->generateToken(
            (int)$userId,
            $scopes,
            $description,
            $expiresIn,
            $clientInfo
        );
    }
    
    /**
     * Get API token by ID
     * 
     * @param string $tokenId Token ID
     * @param int|null $userId User ID (for authorization check)
     * @return array|null Token data or null if not found
     */
    public function getTokenById(string $tokenId, ?int $userId = null): ?array
    {
        try {
            $query = "
                SELECT * 
                FROM `{$this->tokenTable}` 
                WHERE id = :id
            ";
            $params = ['id' => $tokenId];
            
            // If user ID is provided, only get if the token belongs to this user
            if ($userId !== null) {
                $query .= " AND user_id = :user_id";
                $params['user_id'] = $userId;
            }
            
            $token = $this->dbConnection->queryAndFetchAssoc($query, $params);
            
            if (!$token) {
                return null;
            }
            
            // Decode scopes
            $token['scopes'] = json_decode($token['scopes'] ?? '[]', true) ?: [];
            
            // Format dates
            $token['expires_at'] = (int)$token['expires_at'];
            $token['created_at'] = (int)$token['created_at'];
            $token['last_used_at'] = $token['last_used_at'] ? (int)$token['last_used_at'] : null;
            
            return $token;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in ApiAuth', [
                    'error' => $e->getMessage(),
                    'function' => 'getTokenById'
                ]);
            }
            return null;
        }
    }
}