<?php
// app/baseKRIZAN/Services/Authentication.php

namespace baseKRIZAN\Security;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;

/**
 * Authentication service for handling user login/logout and session management
 */
class Authentication
{
    private DatabaseTable $userTable;
    private string $usernameColumn;
    private string $passwordColumn;
    private ?Logger $logger;
    private ?SessionManager $sessionManager;
    private ?object $user = null;
    private int $loginTimeout = 300; // 5 minutes
    private int $failedLoginLimit = 5;
    private string $sessionKey = 'user_id';
    private int $passwordMinLength = 8;

    /**
     * Constructor for Authentication service
     *
     * @param DatabaseTable $userTable Table containing user records
     * @param string $usernameColumn Column name for username/email
     * @param string $passwordColumn Column name for password
     * @param Logger|null $logger Optional logger for authentication events
     * @param SessionManager|null $sessionManager Optional session manager
     */
    public function __construct(
        DatabaseTable $userTable, 
        string $usernameColumn, 
        string $passwordColumn,
        ?Logger $logger = null,
        ?SessionManager $sessionManager = null
    ) {
        $this->userTable = $userTable;
        $this->usernameColumn = $usernameColumn;
        $this->passwordColumn = $passwordColumn;
        $this->logger = $logger;
        $this->sessionManager = $sessionManager ?? new SessionManager($logger);
        
        // Load user from session if session manager is provided
        if ($this->sessionManager?->has($this->sessionKey)) {
            $userId = $this->sessionManager->get($this->sessionKey);
            $this->user = $this->getUserById($userId);
            
            if ($this->user === null) {
                $this->logger?->warning('Invalid user ID in session', [
                    'user_id' => $userId
                ]);
                $this->sessionManager->remove($this->sessionKey);
            }
        }
        
        $this->logger?->security('Authentication service initialized', [
            'user_loaded' => $this->user !== null ? 'yes' : 'no'
        ]);
    }

    /**
     * Set custom configuration options
     *
     * @param array $config Configuration array
     * @return self
     */
    public function configure(array $config): self
    {
        // Set configuration options if provided
        if (isset($config['login_timeout'])) {
            $this->loginTimeout = (int)$config['login_timeout'];
        }
        
        if (isset($config['failed_login_limit'])) {
            $this->failedLoginLimit = (int)$config['failed_login_limit'];
        }
        
        if (isset($config['session_key'])) {
            $this->sessionKey = $config['session_key'];
        }
        
        if (isset($config['password_min_length'])) {
            $this->passwordMinLength = (int)$config['password_min_length'];
        }
        
        return $this;
    }

    /**
     * Log in a user with username/email and password
     *
     * @param string $username Username or email
     * @param string $password Plain text password
     * @return bool Success status
     */
    public function login(string $username, string $password): bool
    {
        // Get user by username/email
        $user = $this->getUserByUsername($username);
        
        if ($user === null) {
            $this->logger?->security('Login failed - user not found', [
                'username' => $username
            ]);
            return false;
        }
        
        // Check for too many failed login attempts
        if (isset($user->failed_login_attempts) && 
            isset($user->last_failed_login) && 
            $user->failed_login_attempts >= $this->failedLoginLimit && 
            (time() - strtotime($user->last_failed_login)) < $this->loginTimeout) {
            
            $this->logger?->security('Login blocked - too many failed attempts', [
                'username' => $username,
                'attempts' => $user->failed_login_attempts
            ]);
            
            return false;
        }
        
        // Verify password
        if (!$this->verifyPassword($password, $user->{$this->passwordColumn})) {
            // Update failed login attempts
            $this->updateFailedLoginAttempts($user);
            
            return false;
        }
        
        // Reset failed login attempts
        $this->resetFailedLoginAttempts($user);
        
        // Store the user object
        $this->user = $user;
        
        // Store user ID in session if session manager is available
        if ($this->sessionManager !== null) {
            $this->sessionManager->set($this->sessionKey, $user->id);
            
            // Regenerate session ID for security
            $this->sessionManager->regenerateId();
            
            $this->logger?->security('User logged in', [
                'username' => $username,
                'user_id' => $user->id
            ]);
        } else {
            $this->logger?->warning('Session manager not available for login', [
                'username' => $username
            ]);
        }
        
        return true;
    }

    /**
     * Log out the current user
     *
     * @return bool Success status
     */
    public function logout(): bool
    {
        // Clear user object
        $this->user = null;
        
        // Remove from session if session manager is available
        if ($this->sessionManager !== null) {
            $currentUserId = $this->sessionManager->get($this->sessionKey);
            
            if ($currentUserId) {
                $this->sessionManager->remove($this->sessionKey);
                $this->sessionManager->regenerateId();
                
                return true;
            }
            
            return false;
        }
        
        $this->logger?->warning('Session manager not available for logout');
        return false;
    }

    /**
     * Check if a user is logged in
     *
     * @return bool True if user is logged in
     */
    public function isLoggedIn(): bool
    {
        return $this->user !== null;
    }

    /**
     * Get the currently logged in user
     *
     * @return object|null User object or null if not logged in
     */
    public function getUser(): ?object
    {
        return $this->user;
    }

    /**
     * Get a user by ID
     *
     * @param string|int $id User ID
     * @return object|null User object or null if not found
     */
    public function getUserById(string|int $id): ?object
    {
        $users = $this->userTable->find('id', $id);
        return !empty($users) ? $users[0] : null;
    }

    /**
     * Get a user by username/email
     *
     * @param string $username Username or email
     * @return object|null User object or null if not found
     */
    public function getUserByUsername(string $username): ?object
    {
        $users = $this->userTable->find($this->usernameColumn, $username);
        return !empty($users) ? $users[0] : null;
    }

    /**
     * Hash a password
     *
     * @param string $password Plain text password
     * @return string Hashed password
     */
    public function hashPassword(string $password): string
    {
        return password_hash($password, PASSWORD_DEFAULT);
    }

    /**
     * Verify a password against a hash
     *
     * @param string $password Plain text password
     * @param string $hash Hashed password
     * @return bool True if password is valid
     */
    public function verifyPassword(string $password, string $hash): bool
    {
        return password_verify($password, $hash);
    }

    /**
     * Update failed login attempts for a user
     *
     * @param object $user User object
     * @return bool Success status
     */
    private function updateFailedLoginAttempts(object $user): bool
    {
        $failedAttempts = isset($user->failed_login_attempts) ? 
                        $user->failed_login_attempts + 1 : 1;
        
        $userData = [
            'id' => $user->id,
            'failed_login_attempts' => $failedAttempts,
            'last_failed_login' => date('Y-m-d H:i:s')
        ];
        
        $result = $this->userTable->save($userData);
        
        return $result !== null && isset($result->id) && $result->id == $user->id;
    }

    /**
     * Reset failed login attempts for a user
     *
     * @param object $user User object
     * @return bool Success status
     */
    private function resetFailedLoginAttempts(object $user): bool
    {
        $userData = [
            'id' => $user->id,
            'failed_login_attempts' => 0,
            'last_failed_login' => null
        ];
        
        $result = $this->userTable->save($userData);
        
        return $result !== null && isset($result->id) && $result->id == $user->id;
    }

    /**
     * Change a user's password (requires current password)
     *
     * @param string|int $userId User ID
     * @param string $currentPassword Current password
     * @param string $newPassword New password
     * @return bool Success status
     */
    public function changePassword(string|int $userId, string $currentPassword, string $newPassword): bool
    {
        $user = $this->getUserById($userId);
        
        if ($user === null) {
            $this->logger?->security('Password change failed - user not found', [
                'user_id' => $userId
            ]);
            return false;
        }
        
        // Verify current password
        if (!$this->verifyPassword($currentPassword, $user->{$this->passwordColumn})) {
            $this->logger?->security('Password change failed - current password invalid', [
                'user_id' => $userId
            ]);
            return false;
        }
        
        // Validate password strength
        if (strlen($newPassword) < $this->passwordMinLength) {
            $this->logger?->security('Password change failed - password too short', [
                'user_id' => $userId,
                'min_length' => $this->passwordMinLength
            ]);
            return false;
        }
        
        // Update the user's password
        $userData = [
            'id' => $userId,
            $this->passwordColumn => $this->hashPassword($newPassword)
        ];
        
        if ($this->userTable->save($userData)) {
            $this->logger?->security('Password changed successfully', [
                'user_id' => $userId
            ]);
            return true;
        }
        
        return false;
    }

    /**
     * Refresh the user's session and session data
     *
     * @return bool Success status
     */
    public function refreshSession(): bool
    {
        if ($this->sessionManager === null) {
            $this->logger?->warning('Session refresh failed - no session manager');
            return false;
        }
        
        // Check if user is loaded
        if ($this->user === null) {
            // Try to load from session
            $userId = $this->sessionManager->get($this->sessionKey);
            if ($userId) {
                $this->user = $this->getUserById($userId);
                
                if ($this->user === null) {
                    // Invalid user ID in session, remove it
                    $this->sessionManager->remove($this->sessionKey);
                    $this->logger?->warning('Session refresh cleared invalid user ID', [
                        'user_id' => $userId
                    ]);
                    return false;
                }
            } else {
                // No user in session
                return false;
            }
        }
        
        // Regenerate session ID periodically for security
        $lastRegenerated = $this->sessionManager->get('last_regenerated', 0);
        $regenerateTime = 1800; // 30 minutes
        
        if (time() - $lastRegenerated > $regenerateTime) {
            $this->sessionManager->regenerateId();
            $this->sessionManager->set('last_regenerated', time());
            
            $this->logger?->security('Session ID regenerated during refresh', [
                'user_id' => $this->user->id
            ]);
        }
        
        return true;
    }
}