<?php
// app/baseKRIZAN/Security/CsrfProtection.php

namespace baseKRIZAN\Security;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Session\SessionManager;

/**
 * CSRF Protection class for managing token generation, validation and protection
 * Provides mechanisms to validate cross-site request forgery protection tokens
 */
class CsrfProtection
{
    private string $tokenName = 'csrf_token';
    private string $oldTokenName = 'csrf_token_old';
    private int $tokenLifetime = 7200; // 2 hours
    private int $rotationInterval = 300; // 5 minutes
    private Logger $logger;
    private SessionManager $sessionManager;
    private array $excludedRoutes = [];
    private array $excludedExtensions = [];
    private array $protectedPrefixes = [];
    private array $allowedDomains = [];
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        SessionManager $sessionManager
    ) {
        $this->logger = $logger;
        $this->sessionManager = $sessionManager;
        
        // Load all security configuration from env
        $this->loadConfiguration();
        
        $logger->security('CSRF Protection initialized', [
            'excluded_routes_count' => count($this->excludedRoutes),
            'excluded_extensions_count' => count($this->excludedExtensions),
            'protected_prefixes_count' => count($this->protectedPrefixes),
            'allowed_domains_count' => count($this->allowedDomains)
        ]);
    }
    
    /**
     * Load all security-related configuration from environment variables
     */
    private function loadConfiguration(): void
    {
        // Load token configuration
        $tokenLifetime = \baseKRIZAN\Config\Config::get('CSRF_TOKEN_LIFETIME');
        if ($tokenLifetime) {
            $this->tokenLifetime = (int)$tokenLifetime;
        }
        
        $rotationInterval = \baseKRIZAN\Config\Config::get('CSRF_ROTATION_INTERVAL');
        if ($rotationInterval) {
            $this->rotationInterval = (int)$rotationInterval;
        }

        // Load excluded routes
        $excludedRoutes = \baseKRIZAN\Config\Config::get('security.csrf.excluded.routes');
        if (!empty($excludedRoutes)) {
            $this->excludedRoutes = array_map('trim', explode(',', $excludedRoutes));
        }
        
        // Load excluded extensions
        $excludedExtensions = \baseKRIZAN\Config\Config::get('CSRF_EXCLUDED_EXTENSIONS');
        if (!empty($excludedExtensions)) {
            $this->excludedExtensions = array_map('trim', explode(',', $excludedExtensions));
        } else {
            // Default excluded extensions
            $this->excludedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'css', 'js', 'ico', 'svg', 'woff', 'woff2', 'ttf'];
        }
        
        // Load protected prefixes
        $protectedPrefixes = \baseKRIZAN\Config\Config::get('CSRF_PROTECTED_PREFIXES');
        if (!empty($protectedPrefixes)) {
            $this->protectedPrefixes = array_map('trim', explode(',', $protectedPrefixes));
        } else {
            // Default protected prefixes like admin, dashboard, etc.
            $this->protectedPrefixes = ['admin', 'dashboard', 'user', 'profile', 'account', 'settings'];
        }
        
        // Load allowed domains
        $allowedDomains = \baseKRIZAN\Config\Config::get('CSRF_ALLOWED_DOMAINS');
        if (!empty($allowedDomains)) {
            $this->allowedDomains = array_map('trim', explode(',', $allowedDomains));
        }
    }
    
    /**
     * Generate a new CSRF token or return existing one if still valid
     * Implements token rotation based on configured intervals
     */
    public function generateToken(): string
    {
        // Ensure session is started
        if (!$this->sessionManager->isStarted()) {
            $this->sessionManager->start();
        }
        
        $tokenData = $this->sessionManager->get($this->tokenName);
        $needsNewToken = true;

        if (is_array($tokenData) && 
            isset($tokenData['value'], $tokenData['created_at']) && 
            is_string($tokenData['value']) && 
            is_int($tokenData['created_at'])) {
            
            $needsNewToken = $this->shouldRotateToken($tokenData['created_at']);
        }

        if ($needsNewToken) {
            // Save old token before creating new one
            if ($this->sessionManager->has($this->tokenName) && 
                is_array($this->sessionManager->get($this->tokenName))) {
                $this->sessionManager->set($this->oldTokenName, $this->sessionManager->get($this->tokenName));
            }

            // Generate new token with creation timestamp
            $this->sessionManager->set($this->tokenName, [
                'value' => bin2hex(random_bytes(32)),
                'created_at' => time()
            ]);
            
            $this->logger->security('CSRF token rotated');
        }

        return $this->sessionManager->get($this->tokenName)['value'];
    }
    
    /**
     * Determine if a token should be rotated based on age
     */
    private function shouldRotateToken(int $createdAt): bool
    {
        // Additional data type check
        if (!is_int($createdAt)) {
            return true;
        }

        $currentTime = time();
        
        // Rotation by interval
        $needsRotationByInterval = ($currentTime - $createdAt) >= $this->rotationInterval;
        
        // Token expiration
        $tokenExpired = ($currentTime - $createdAt) >= $this->tokenLifetime;

        return $needsRotationByInterval || $tokenExpired;
    }
    
    /**
     * Get the token name for form fields
     */
    public function getTokenName(): string
    {
        return $this->tokenName;
    }
    
    /**
     * Validate a CSRF token with request context
     * Handles route exclusions and domain validation
     */
    public function validateToken(?string $token, Request $request): bool
    {
        // Skip validation for excluded routes
        $route = $request->getPath();
        
        // Check for excluded routes
        foreach ($this->excludedRoutes as $excluded) {
            if ($excluded && str_starts_with($route, $excluded)) {
                $this->logger->security('CSRF validation skipped for excluded route', [
                    'route' => $route,
                    'excluded_pattern' => $excluded
                ]);
                return true;
            }
        }
        
        // Check for excluded extensions
        $extension = pathinfo($route, PATHINFO_EXTENSION);
        if (in_array(strtolower($extension), $this->excludedExtensions)) {
            $this->logger->security('CSRF validation skipped for asset file', [
                'route' => $route,
                'extension' => $extension
            ]);
            return true;
        }
        
        // Check for valid referrer from allowed domains
        $referrerHeader = $request->getHeader('Referer');

        // getHeader može vratiti string ili array, ovisno o implementaciji
        // Osigurajmo da uvijek imamo string
        $referrerStr = is_array($referrerHeader) ? (reset($referrerHeader) ?: '') : $referrerHeader;

        if (!empty($referrerStr)) {
            $referrerHost = parse_url($referrerStr, PHP_URL_HOST);
            foreach ($this->allowedDomains as $domain) {
                if ($domain && $referrerHost && str_contains($referrerHost, $domain)) {
                    $this->logger->security('CSRF validation skipped for allowed domain', [
                        'referrer' => $referrerHost,
                        'allowed_domain' => $domain
                    ]);
                    return true;
                }
            }
        }
        
        // Skip token validation for safe HTTP methods
        if (in_array($request->getMethod(), ['GET', 'HEAD', 'OPTIONS', 'TRACE'])) {
            $this->logger->security('CSRF validation skipped for safe HTTP method', [
                'method' => $request->getMethod()
            ]);
            return true;
        }
        
        // No token provided
        if (empty($token)) {
            $this->logger->security('CSRF validation failed: No token provided', [
                'route' => $request->getPath(),
                'method' => $request->getMethod()
            ]);
            return false;
        }

        // Check current token
        if ($this->sessionManager->has($this->tokenName) && 
            is_array($this->sessionManager->get($this->tokenName)) && 
            isset($this->sessionManager->get($this->tokenName)['value']) && 
            hash_equals($this->sessionManager->get($this->tokenName)['value'], $token)) {
            
            $this->logger->security('CSRF token validated successfully');
            return true;
        }
        
        // Check previous token (for double submit issues)
        if ($this->sessionManager->has($this->oldTokenName) && 
            is_array($this->sessionManager->get($this->oldTokenName)) && 
            isset($this->sessionManager->get($this->oldTokenName)['value']) && 
            hash_equals($this->sessionManager->get($this->oldTokenName)['value'], $token)) {
            
            $this->logger->security('CSRF token validated with previous token');
            return true;
        }
        
        $this->logger->security('CSRF validation failed: Invalid token', [
            'route' => $request->getPath(),
            'token_provided' => $token ? substr($token, 0, 8) . '...' : 'empty'
        ]);
        
        return false;
    }
    
    /**
     * Check if a route is protected and requires CSRF validation
     */
    public function isProtectedRoute(string $path): bool
    {
        foreach ($this->protectedPrefixes as $prefix) {
            if (!empty($prefix) && str_starts_with($path, $prefix)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Simpler token validation without request context
     */
    public function verifyToken(?string $token): bool
    {
        if (empty($token)) return false;

        // Check current token
        if ($this->sessionManager->has($this->tokenName) && 
            is_array($this->sessionManager->get($this->tokenName)) && 
            isset($this->sessionManager->get($this->tokenName)['value']) && 
            is_string($this->sessionManager->get($this->tokenName)['value']) && 
            hash_equals($this->sessionManager->get($this->tokenName)['value'], $token)) {
            return true;
        }

        // Check previous token
        if ($this->sessionManager->has($this->oldTokenName) && 
            is_array($this->sessionManager->get($this->oldTokenName)) && 
            isset($this->sessionManager->get($this->oldTokenName)['value']) && 
            is_string($this->sessionManager->get($this->oldTokenName)['value']) && 
            hash_equals($this->sessionManager->get($this->oldTokenName)['value'], $token)) {
            return true;
        }

        return false;
    }
    
    /**
     * Clean up old and expired CSRF tokens
     */
    public function cleanup(): void
    {
        // Remove old token if it exists or if it's expired
        if ($this->sessionManager->has($this->oldTokenName)) {
            $oldToken = $this->sessionManager->get($this->oldTokenName);
            
            // Check if old token is older than the token lifetime
            if (is_array($oldToken) && 
                isset($oldToken['created_at']) &&
                (time() - $oldToken['created_at'] > $this->tokenLifetime)) {
                $this->sessionManager->remove($this->oldTokenName);
                
                $this->logger->security('Removed expired old CSRF token');
            }
        }
        
        // If current token is too old (>token lifetime), remove it as well
        if ($this->sessionManager->has($this->tokenName)) {
            $currentToken = $this->sessionManager->get($this->tokenName);
            
            if (is_array($currentToken) && 
                isset($currentToken['created_at']) &&
                (time() - $currentToken['created_at'] > $this->tokenLifetime)) {
                $this->sessionManager->remove($this->tokenName);
                
                $this->logger->security('Removed expired CSRF token');
            }
        }
    }
    
    /**
     * Generate HTML for a CSRF form field
     */
    public function formField(): string
    {
        $token = $this->generateToken();
        return '<input type="hidden" name="' . $this->tokenName . '" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
    }
    
    /**
     * Generate HTML meta tag for CSRF token used in JavaScript
     */
    public function metaTag(): string
    {
        $token = $this->generateToken();
        return '<meta name="csrf-token" content="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
    }

    /**
     * Get the list of routes excluded from CSRF protection
     */
    public function getExcludedRoutes(): array
    {
        return $this->excludedRoutes;
    }
    
    /**
     * Get the list of extensions excluded from CSRF protection
     */
    public function getExcludedExtensions(): array
    {
        return $this->excludedExtensions;
    }
    
    /**
     * Get the list of protected route prefixes
     */
    public function getProtectedPrefixes(): array
    {
        return $this->protectedPrefixes;
    }
    
    /**
     * Get the list of allowed domains
     */
    public function getAllowedDomains(): array
    {
        return $this->allowedDomains;
    }
    
    /**
     * Get the current CSRF token lifetime in seconds
     */
    public function getTokenLifetime(): int
    {
        return $this->tokenLifetime;
    }
    
    /**
     * Get the current CSRF token rotation interval in seconds
     */
    public function getRotationInterval(): int
    {
        return $this->rotationInterval;
    }
}