<?php

namespace baseKRIZAN\Security;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Http\RequestClassifier;
use baseKRIZAN\Cache\CacheInterface;
use baseKRIZAN\Cache\CacheManager;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Class for rate limiting requests using database storage
 * with optional cache acceleration
 */
class RateLimiter
{
    /**
     * Array of protected routes with their rate limits
     */
    private array $protectedRoutes;

    /**
     * Array of whitelisted routes
     */
    private array $whitelistedRoutes;
    
    /**
     * Threshold for IP rotation detection
     */
    private int $ipRotationThreshold;
    
    /**
     * Timeframe for IP rotation detection (in seconds)
     */
    private int $ipRotationTimeframe;
    
    /**
     * Decay time for IP rotation blocks (in seconds)
     */
    private int $ipRotationDecay;
    
    private TemplateRenderer $templateRenderer;
    private ?RequestClassifier $requestClassifier;
    private ?CacheInterface $cache;
    private DatabaseConnection $dbConnection;
    private ?Logger $logger;
    
    // Database tables
    private string $attemptsTable = 'security_ratelimit_attempts';
    private string $blocksTable = 'security_ratelimit_blocks';
    private string $ipTrackingTable = 'security_ratelimit_iptracking';
    
    // Cache configuration
    private bool $cacheEnabled = false;
    private int $cacheExpiryTime = 300; // 5 minutes
    
    /**
     * Constructor
     */
    public function __construct(
        TemplateRenderer $templateRenderer,
        ?Logger $logger = null,
        ?RequestClassifier $requestClassifier = null,
        ?CacheInterface $cache = null,
        ?DatabaseConnection $dbConnection = null
    ) {
        $this->templateRenderer = $templateRenderer;
        $this->logger = $logger;
        $this->requestClassifier = $requestClassifier;
        
        // Cache is now optional, try to get from CacheManager if not provided
        if ($cache !== null) {
            $this->cache = $cache;
            $this->cacheEnabled = true;
        } else {
            try {
                $this->cache = CacheManager::getInstance($logger)->driver();
                $this->cacheEnabled = true;
            } catch (\Throwable $e) {
                // If cache manager isn't available, continue without caching
                if ($this->logger) {
                    $this->logger->security('RateLimiter operating without cache acceleration', [
                        'reason' => $e->getMessage()
                    ]);
                }
                $this->cacheEnabled = false;
            }
        }
        
        // Initialize DatabaseConnection
        if ($dbConnection !== null) {
            $this->dbConnection = $dbConnection;
        } else {
            try {
                $this->dbConnection = DatabaseConnection::getInstance();
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to create database connection for rate limiter', [
                        'error' => $e->getMessage()
                    ]);
                }
                throw $e;
            }
        }
        
        // Load configuration
        $this->loadConfiguration();
        
        // Ensure database tables exist
        $this->ensureTablesExist();
    }
    
    /**
     * Load rate limiter configuration
     */
    private function loadConfiguration(): void
    {
        // Define protected routes with their limits
        $this->protectedRoutes = [
            'register/user' => ['max' => 3, 'window' => 30],    // 3 attempts / 30 min
            'login' => ['max' => 5, 'window' => 30],            // 5 attempts / 30 min
            'password/reset' => ['max' => 3, 'window' => 60],   // 3 attempts / 1 hour
            'password/update' => ['max' => 3, 'window' => 60],  // 3 attempts / 1 hour
            'cms' => ['max' => 3, 'window' => 60],              // 3 attempts / 1 hour
            'user/permissions' => ['max' => 3, 'window' => 60]  // 3 attempts / 1 hour
        ];

        // Define whitelisted routes
        $this->whitelistedRoutes = [
            'notification/api/save-token'
        ];
        
        // Load IP rotation settings
        $this->ipRotationThreshold = (int)\baseKRIZAN\Config\Config::get('security.ratelimit.ip_rotation.threshold', 3);
        $this->ipRotationTimeframe = (int)\baseKRIZAN\Config\Config::get('security.ratelimit.ip_rotation.timeframe', 1800);
        $this->ipRotationDecay = (int)\baseKRIZAN\Config\Config::get('security.ratelimit.ip_rotation.decay', 3600);
        
        // Custom table names if configured
        $customAttemptsTable = \baseKRIZAN\Config\Config::get('security.ratelimit.attempts_table');
        if ($customAttemptsTable) {
            $this->attemptsTable = $customAttemptsTable;
        }
        
        $customBlocksTable = \baseKRIZAN\Config\Config::get('security.ratelimit.blocks_table');
        if ($customBlocksTable) {
            $this->blocksTable = $customBlocksTable;
        }
        
        $customIpTrackingTable = \baseKRIZAN\Config\Config::get('security.ratelimit.iptracking_table');
        if ($customIpTrackingTable) {
            $this->ipTrackingTable = $customIpTrackingTable;
        }
        
        // Load cache configuration
        $this->cacheEnabled = (bool)\baseKRIZAN\Config\Config::get('security.ratelimit.cache_enabled', $this->cacheEnabled);
        $this->cacheExpiryTime = (int)\baseKRIZAN\Config\Config::get('security.ratelimit.cache_expiry', $this->cacheExpiryTime);
    }
    
    /**
     * Ensure the necessary database tables exist
     */
    private function ensureTablesExist(): void
    {
        try {
            // Check if tables exist
            $this->checkAndCreateAttemptsTable();
            $this->checkAndCreateBlocksTable();
            $this->checkAndCreateIpTrackingTable();
            
            if ($this->logger) {
                $this->logger->security('Rate limiter database tables verified');
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error checking/creating rate limit tables', [
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
    
    /**
     * Check and create attempts table if needed
     */
    private function checkAndCreateAttemptsTable(): void
    {
        // Validate table name for safety
        $this->validateTableName($this->attemptsTable);
        
        // Check if table exists
        $tableExists = $this->dbConnection->querySingleValue(
            "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
            ['table' => $this->attemptsTable]
        );
        
        if ($tableExists === null) {
            // Create the table if it doesn't exist
            $sql = "CREATE TABLE `{$this->attemptsTable}` (
                `id` BIGINT AUTO_INCREMENT PRIMARY KEY,
                `key_hash` VARCHAR(64) NOT NULL,
                `route` VARCHAR(255) NOT NULL,
                `ip` VARCHAR(45) NOT NULL,
                `identifier` VARCHAR(255) DEFAULT NULL,
                `method` VARCHAR(10) NOT NULL,
                `timestamp` INT UNSIGNED NOT NULL,
                `expires_at` INT UNSIGNED NOT NULL,
                INDEX `idx_key_hash` (`key_hash`),
                INDEX `idx_expires` (`expires_at`),
                INDEX `idx_ip` (`ip`),
                INDEX `idx_identifier` (`identifier`),
                INDEX `idx_route` (`route`)
            )";
            
            $this->dbConnection->execute($sql);
            
            if ($this->logger) {
                $this->logger->security('Created rate limit attempts table', [
                    'table' => $this->attemptsTable
                ]);
            }
        }
    }
    
    /**
     * Check and create blocks table if needed
     */
    private function checkAndCreateBlocksTable(): void
    {
        // Validate table name for safety
        $this->validateTableName($this->blocksTable);
        
        // Check if table exists
        $tableExists = $this->dbConnection->querySingleValue(
            "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
            ['table' => $this->blocksTable]
        );
        
        if ($tableExists === null) {
            // Create the table if it doesn't exist
            $sql = "CREATE TABLE `{$this->blocksTable}` (
                `key_hash` VARCHAR(64) PRIMARY KEY,
                `route` VARCHAR(255) NOT NULL,
                `ip` VARCHAR(45) NOT NULL,
                `identifier` VARCHAR(255) DEFAULT NULL,
                `reason` VARCHAR(50) NOT NULL,
                `created_at` INT UNSIGNED NOT NULL,
                `blocked_until` INT UNSIGNED NOT NULL,
                `request_data` TEXT DEFAULT NULL,
                INDEX `idx_blocked_until` (`blocked_until`),
                INDEX `idx_ip` (`ip`),
                INDEX `idx_identifier` (`identifier`)
            )";
            
            $this->dbConnection->execute($sql);
            
            if ($this->logger) {
                $this->logger->security('Created rate limit blocks table', [
                    'table' => $this->blocksTable
                ]);
            }
        }
    }
    
    /**
     * Check and create IP tracking table if needed
     */
    private function checkAndCreateIpTrackingTable(): void
    {
        // Validate table name for safety
        $this->validateTableName($this->ipTrackingTable);
        
        // Check if table exists
        $tableExists = $this->dbConnection->querySingleValue(
            "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
            ['table' => $this->ipTrackingTable]
        );
        
        if ($tableExists === null) {
            // Create the table if it doesn't exist
            $sql = "CREATE TABLE `{$this->ipTrackingTable}` (
                `id` BIGINT AUTO_INCREMENT PRIMARY KEY,
                `identifier_hash` VARCHAR(64) NOT NULL,
                `identifier` VARCHAR(255) NOT NULL,
                `ip` VARCHAR(45) NOT NULL,
                `user_agent` TEXT DEFAULT NULL,
                `first_seen` INT UNSIGNED NOT NULL,
                `last_seen` INT UNSIGNED NOT NULL,
                UNIQUE KEY `uk_identifier_ip` (`identifier_hash`, `ip`),
                INDEX `idx_identifier` (`identifier_hash`),
                INDEX `idx_last_seen` (`last_seen`),
                INDEX `idx_ip` (`ip`)
            )";
            
            $this->dbConnection->execute($sql);
            
            if ($this->logger) {
                $this->logger->security('Created IP tracking table', [
                    'table' => $this->ipTrackingTable
                ]);
            }
        }
    }
    
    /**
     * Validate table name to prevent SQL injection
     */
    private function validateTableName(string $tableName): void
    {
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $tableName)) {
            throw new \InvalidArgumentException("Invalid table name: {$tableName}");
        }
    }
    
    /**
     * Comprehensive route resolution
     */
    public function resolveRoute(Request $request): string 
    {
        // Try to get route from request object first
        $route = $request->getQuery('route') ?? $request->getPath();
        
        // Clean up the route
        $basePath = \baseKRIZAN\Config\Config::get('paths.app_url');
        
        $basePathPattern = '^\/?(' . preg_quote(trim($basePath, '/'), '/') . '\/)?';
        $route = trim(preg_replace('/' . $basePathPattern . '/', '', $route), '/');
        
        // Handle root path specially
        if ($route === '' && $request->getPath() === '/') {
            return 'home'; // ili neki drugi identifikator za početnu stranicu
        }
        
        return $route ?: '';
    }

    /**
     * Check if rate limiting should be applied to a route
     */
    public function shouldRateLimit(string $route): bool 
    {
        // Use request classifier if available
        $request = Request::getCurrent();
        if ($request && $this->requestClassifier && $this->requestClassifier->isAssetRequest($request)) {
            return false; // Never rate limit asset requests
        }
        
        // Normalize route by removing leading/trailing slashes and base paths
        $normalizedRoute = trim($route, '/');
        
        // Special case for root path which becomes empty after trim
        if ($normalizedRoute === '' && $request && $request->getPath() === '/') {
            $normalizedRoute = 'home'; // ili isti identifikator kao u resolveRoute
        }
        
        $result = isset($this->protectedRoutes[$normalizedRoute]);
        
        if ($this->logger) {
            $this->logger->security('RateLimiter: Route matching check', [
                'input_route' => $route,
                'normalized_route' => $normalizedRoute ?: '/', // Prikaži '/' umjesto praznog stringa
                'protected_routes' => array_keys($this->protectedRoutes),
                'matched' => $result
            ]);
        }
        
        return $result;
    }

    /**
     * Generate a unique signature for the request
     */
    public function getRequestSignature(Request $request, string $route): string 
    {
        return sha1(
            $request->getIp() . '|' .
            $route . '|' .
            $request->getMethod() . '|' .
            ($request->getPost('email') ?? $request->getPost('username') ?? '')
        );
    }

    /**
     * Check if too many attempts were made within the sliding window
     */
    public function tooManyAttempts(string $key, int $maxAttempts, int $windowMinutes, Request $request, string $route): bool 
    {
        // First check cache for faster lookups
        if ($this->cacheEnabled && $this->cache) {
            $cacheKey = 'ratelimit:blocked:' . $key;
            $isBlocked = $this->cache->get($cacheKey);
            
            if ($isBlocked) {
                // If blocked in cache, return immediately
                return true;
            }
        }
        
        // Then check if we have an active block in database
        $blockData = $this->getBlockData($key);
        if ($blockData && time() < $blockData['blocked_until']) {
            // Cache the block if caching is enabled
            if ($this->cacheEnabled && $this->cache) {
                $cacheKey = 'ratelimit:blocked:' . $key;
                $cacheTime = $blockData['blocked_until'] - time();
                $this->cache->set($cacheKey, true, $cacheTime);
            }
            
            return true;
        }
        
        // Calculate sliding window start time
        $windowStart = time() - ($windowMinutes * 60);
        
        // Check cache for attempt count
        if ($this->cacheEnabled && $this->cache) {
            $cacheKey = 'ratelimit:attempts:' . $key;
            $attemptCount = $this->cache->get($cacheKey);
            
            if ($attemptCount !== null && $attemptCount >= $maxAttempts) {
                // Create a block record (we only get here if we had cached attempts but no cached block)
                $this->createBlock($key, $request, $route, 'too_many_attempts', time() + ($windowMinutes * 60));
                return true;
            }
        }
        
        // Count recent attempts from database
        try {
            $attemptCount = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(*) FROM `{$this->attemptsTable}` WHERE `key_hash` = :key_hash AND `timestamp` >= :window_start",
                [
                    'key_hash' => $key,
                    'window_start' => $windowStart
                ]
            );
            
            // Update cache
            if ($this->cacheEnabled && $this->cache) {
                $cacheKey = 'ratelimit:attempts:' . $key;
                $this->cache->set($cacheKey, $attemptCount, $this->cacheExpiryTime);
            }
            
            // Check if we're over the limit
            if ($attemptCount >= $maxAttempts) {
                // Create a block record
                $this->createBlock($key, $request, $route, 'too_many_attempts', time() + ($windowMinutes * 60));
                return true;
            }
            
            return false;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in rate limiter', [
                    'error' => $e->getMessage(),
                    'function' => 'tooManyAttempts'
                ]);
            }
            // On error, allow the request to proceed (fail open for legitimate users)
            return false;
        }
    }

    /**
     * Add a new attempt to the sliding window
     */
    public function addAttempt(string $key, int $windowMinutes, Request $request, string $route): void 
    {
        try {
            $now = time();
            $expires = $now + ($windowMinutes * 60);
            $identifier = $this->getIdentifier($request);
            
            $this->dbConnection->execute(
                "INSERT INTO `{$this->attemptsTable}` (`key_hash`, `route`, `ip`, `identifier`, `method`, `timestamp`, `expires_at`) VALUES (:key_hash, :route, :ip, :identifier, :method, :timestamp, :expires_at)",
                [
                    'key_hash' => $key,
                    'route' => $route,
                    'ip' => $request->getIp(),
                    'identifier' => $identifier,
                    'method' => $request->getMethod(),
                    'timestamp' => $now,
                    'expires_at' => $expires
                ]
            );
            
            // Update cache
            if ($this->cacheEnabled && $this->cache) {
                $cacheKey = 'ratelimit:attempts:' . $key;
                $currentCount = $this->cache->get($cacheKey) ?? 0;
                $this->cache->set($cacheKey, $currentCount + 1, $this->cacheExpiryTime);
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in rate limiter', [
                    'error' => $e->getMessage(),
                    'function' => 'addAttempt'
                ]);
            }
        }
    }

    /**
     * Get block data for a key
     */
    private function getBlockData(string $key): ?array
    {
        try {
            $block = $this->dbConnection->queryAndFetchAssoc(
                "SELECT * FROM `{$this->blocksTable}` WHERE `key_hash` = :key_hash AND `blocked_until` > :now",
                [
                    'key_hash' => $key,
                    'now' => time()
                ]
            );
            
            return $block ?: null;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in rate limiter', [
                    'error' => $e->getMessage(),
                    'function' => 'getBlockData'
                ]);
            }
            return null;
        }
    }
    
    /**
     * Create a block record
     */
    private function createBlock(string $key, Request $request, string $route, string $reason, int $blockedUntil): void
    {
        try {
            $now = time();
            $identifier = $this->getIdentifier($request);
            
            // Store some safe request data for analysis
            $requestData = json_encode([
                'headers' => $request->getHeaders(),
                'path' => $request->getPath(),
                'query' => $request->getQueryParams()
            ]);
            
            // Use REPLACE to handle key conflicts
            $this->dbConnection->execute(
                "REPLACE INTO `{$this->blocksTable}` (`key_hash`, `route`, `ip`, `identifier`, `reason`, `created_at`, `blocked_until`, `request_data`) VALUES (:key_hash, :route, :ip, :identifier, :reason, :created_at, :blocked_until, :request_data)",
                [
                    'key_hash' => $key,
                    'route' => $route,
                    'ip' => $request->getIp(),
                    'identifier' => $identifier,
                    'reason' => $reason,
                    'created_at' => $now,
                    'blocked_until' => $blockedUntil,
                    'request_data' => $requestData
                ]
            );
            
            // Cache the block if caching is enabled
            if ($this->cacheEnabled && $this->cache) {
                $cacheKey = 'ratelimit:blocked:' . $key;
                $cacheTime = $blockedUntil - time();
                $this->cache->set($cacheKey, true, $cacheTime);
            }
            
            if ($this->logger) {
                $this->logger->security('Rate limit block created', [
                    'key' => substr($key, 0, 8) . '...',
                    'ip' => $request->getIp(),
                    'route' => $route,
                    'reason' => $reason,
                    'blocked_until' => date('Y-m-d H:i:s', $blockedUntil)
                ]);
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in rate limiter', [
                    'error' => $e->getMessage(),
                    'function' => 'createBlock'
                ]);
            }
        }
    }

    /**
     * Calculate retry-after time in seconds
     */
    public function getRetryAfterTime(string $key): int
    {
        // Try cache first
        if ($this->cacheEnabled && $this->cache) {
            $cacheKey = 'ratelimit:blocked:' . $key . ':time';
            $blockedUntil = $this->cache->get($cacheKey);
            
            if ($blockedUntil !== null) {
                $retryAfter = max(0, $blockedUntil - time());
                return $retryAfter;
            }
        }
        
        // Fallback to database lookup
        try {
            $blockedUntil = $this->dbConnection->querySingleValue(
                "SELECT `blocked_until` FROM `{$this->blocksTable}` WHERE `key_hash` = :key_hash",
                ['key_hash' => $key]
            );
            
            if ($blockedUntil === null) {
                return 60; // Default 1 minute if no data
            }
            
            $retryAfter = max(0, $blockedUntil - time());
            
            // Cache the value
            if ($this->cacheEnabled && $this->cache) {
                $cacheKey = 'ratelimit:blocked:' . $key . ':time';
                $this->cache->set($cacheKey, $blockedUntil, $retryAfter);
            }
            
            return $retryAfter;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in rate limiter', [
                    'error' => $e->getMessage(),
                    'function' => 'getRetryAfterTime'
                ]);
            }
            return 60; // Default 1 minute if error
        }
    }

    /**
     * Get identifier from request (email/username)
     */
    public function getIdentifier(Request $request): ?string
    {
        // Try to extract an identifier from the request
        $identifier = $request->getPost('email') ?? $request->getPost('username') ?? null;
        
        // Only use valid identifiers
        if ($identifier && strlen($identifier) > 3 && str_contains($identifier, '@')) {
            return strtolower($identifier); // Normalize emails
        }
        
        return null;
    }
    
    /**
     * Track IP address for a given identifier
     */
    public function trackIpForIdentifier(string $identifier, Request $request): void
    {
        try {
            $now = time();
            $ip = $request->getIp();
            $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
            $identifierHash = hash('sha256', $identifier);
            
            // Check if this IP is already tracked for this identifier
            $existingRecord = $this->dbConnection->queryAndFetchAssoc(
                "SELECT `id`, `first_seen` FROM `{$this->ipTrackingTable}` WHERE `identifier_hash` = :identifier_hash AND `ip` = :ip",
                [
                    'identifier_hash' => $identifierHash,
                    'ip' => $ip
                ]
            );
            
            if ($existingRecord) {
                // Update the existing record
                $this->dbConnection->execute(
                    "UPDATE `{$this->ipTrackingTable}` SET `last_seen` = :last_seen, `user_agent` = :user_agent WHERE `id` = :id",
                    [
                        'last_seen' => $now,
                        'user_agent' => $userAgent,
                        'id' => $existingRecord['id']
                    ]
                );
            } else {
                // Insert a new record
                $this->dbConnection->execute(
                    "INSERT INTO `{$this->ipTrackingTable}` (`identifier_hash`, `identifier`, `ip`, `user_agent`, `first_seen`, `last_seen`) VALUES (:identifier_hash, :identifier, :ip, :user_agent, :first_seen, :last_seen)",
                    [
                        'identifier_hash' => $identifierHash,
                        'identifier' => $identifier,
                        'ip' => $ip,
                        'user_agent' => $userAgent,
                        'first_seen' => $now,
                        'last_seen' => $now
                    ]
                );
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in IP tracking', [
                    'error' => $e->getMessage(),
                    'identifier' => substr($identifier, 0, 3) . '...',
                    'ip' => $ip
                ]);
            }
        }
    }
    
    /**
     * Detect suspicious IP rotation patterns
     */
    public function detectIpRotation(Request $request): bool
    {
        $identifier = $this->getIdentifier($request);
        if (!$identifier) {
            return false;
        }
        
        $identifierHash = hash('sha256', $identifier);
        
        // Check cache first for faster lookups
        if ($this->cacheEnabled && $this->cache) {
            $cacheKey = 'ratelimit:iprot:' . $identifierHash;
            $isBlocked = $this->cache->get($cacheKey);
            
            if ($isBlocked) {
                return true;
            }
        }
        
        try {
            // Check if the identifier is already blocked for IP rotation
            $isBlocked = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM `{$this->blocksTable}` WHERE `identifier` = :identifier AND `reason` = 'ip_rotation' AND `blocked_until` > :now",
                [
                    'identifier' => $identifier,
                    'now' => time()
                ]
            );
            
            if ($isBlocked !== null) {
                // Cache the block if caching is enabled
                if ($this->cacheEnabled && $this->cache) {
                    $cacheKey = 'ratelimit:iprot:' . $identifierHash;
                    $this->cache->set($cacheKey, true, $this->ipRotationDecay);
                }
                
                return true; // Already blocked
            }
            
            // Calculate timeframe
            $timeframeStart = time() - $this->ipRotationTimeframe;
            
            // Count distinct IPs used with this identifier in the timeframe
            $ipCount = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(DISTINCT `ip`) FROM `{$this->ipTrackingTable}` WHERE `identifier_hash` = :identifier_hash AND `first_seen` >= :timeframe_start",
                [
                    'identifier_hash' => $identifierHash,
                    'timeframe_start' => $timeframeStart
                ]
            );
            
            // Check if threshold exceeded
            if ($ipCount >= $this->ipRotationThreshold) {
                // Get the IPs for logging
                $ips = $this->dbConnection->queryAndFetchAllAssoc(
                    "SELECT `ip` FROM `{$this->ipTrackingTable}` WHERE `identifier_hash` = :identifier_hash AND `first_seen` >= :timeframe_start",
                    [
                        'identifier_hash' => $identifierHash,
                        'timeframe_start' => $timeframeStart
                    ]
                );
                
                $ips = array_column($ips, 'ip');
                
                // Create a block for IP rotation
                $key = sha1($identifier . '_ip_rotation');
                $blockedUntil = time() + $this->ipRotationDecay;
                
                // Store the list of IPs in request_data
                $requestData = json_encode([
                    'ips' => $ips,
                    'ip_count' => $ipCount,
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                    'timeframe' => $this->ipRotationTimeframe
                ]);
                
                // Use REPLACE to handle key conflicts
                $this->dbConnection->execute(
                    "REPLACE INTO `{$this->blocksTable}` (`key_hash`, `route`, `ip`, `identifier`, `reason`, `created_at`, `blocked_until`, `request_data`) VALUES (:key_hash, :route, :ip, :identifier, :reason, :created_at, :blocked_until, :request_data)",
                    [
                        'key_hash' => $key,
                        'route' => $request->getPath(),
                        'ip' => $request->getIp(),
                        'identifier' => $identifier,
                        'reason' => 'ip_rotation',
                        'created_at' => time(),
                        'blocked_until' => $blockedUntil,
                        'request_data' => $requestData
                    ]
                );
                
                // Cache the block
                if ($this->cacheEnabled && $this->cache) {
                    $cacheKey = 'ratelimit:iprot:' . $identifierHash;
                    $this->cache->set($cacheKey, true, $this->ipRotationDecay);
                }
                
                return true;
            }
            
            return false;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in IP rotation detection', [
                    'error' => $e->getMessage(),
                    'identifier' => substr($identifier, 0, 3) . '...'
                ]);
            }
            return false; // On error, allow the request
        }
    }
    
    /**
     * Clean up old data
     */
    public function cleanOldData(): void 
    {
        $now = time();
        
        try {
            // Clean expired attempts
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->attemptsTable}` WHERE `expires_at` < :now",
                ['now' => $now]
            );
            $attemptsDeleted = $affectedRows;
            
            // Clean expired blocks
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->blocksTable}` WHERE `blocked_until` < :now",
                ['now' => $now]
            );
            $blocksDeleted = $affectedRows;
            
            // Clean old IP tracking (older than 30 days)
            $oldTrackingTimestamp = $now - (30 * 86400);
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->ipTrackingTable}` WHERE `last_seen` < :timestamp",
                ['timestamp' => $oldTrackingTimestamp]
            );
            $trackingDeleted = $affectedRows;
            
            if (($attemptsDeleted + $blocksDeleted + $trackingDeleted) > 0 && $this->logger) {
                $this->logger->security('Rate limiter cleanup performed', [
                    'attempts_deleted' => $attemptsDeleted,
                    'blocks_deleted' => $blocksDeleted,
                    'tracking_deleted' => $trackingDeleted
                ]);
            }
            
            // Clear any expired cache entries if caching enabled
            if ($this->cacheEnabled && $this->cache && method_exists($this->cache, 'gc')) {
                $this->cache->gc();
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error cleaning rate limit data', [
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Log blocked attempt
     */
    public function logBlockedAttempt(Request $request, string $route): void 
    {
        if ($this->logger) {
            $this->logger->security('RateLimiter: Rate limit exceeded - BLOCKED', [
                'ip' => $request->getIp(),
                'route' => $route,
                'email' => $request->getPost('email') ?? 'N/A',
                'username' => $request->getPost('username') ?? 'N/A'
            ]);
        }
    }
    
    /**
     * Log IP rotation blocked
     */
    public function logIpRotationBlocked(Request $request): void 
    {
        if ($this->logger) {
            $this->logger->security('RateLimiter: IP rotation detected and blocked', [
                'ip' => $request->getIp(),
                'identifier' => $this->getIdentifier($request),
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        }
    }

    /**
     * Build rate limit response
     */
    public function buildRateLimitResponse(float $decayMinutes): Response 
    {
        // Ensure we have an integer value of minutes
        $decayMinutesInt = (int)ceil($decayMinutes);
        
        date_default_timezone_set(\baseKRIZAN\Config\Config::get('defaulttimezone'));
        $unlockTime = date('H:i', time() + ($decayMinutesInt * 60));
        
        $message = sprintf(
            'For security reasons, further attempts have been blocked until %s. Please try again later.',
            $unlockTime
        );
        
        $response = (new Response())->renderError(
            $message,
            'Too Many Attempts',
            429
        );
        
        // Ensure integer value for the header
        $retryAfterSeconds = (int)ceil($decayMinutesInt * 60);
        $response->setHeader('Retry-After', (string)$retryAfterSeconds);
        
        return $response;
    }
    
    /**
     * Get statistics for dashboard
     * 
     * @param int $timePeriod Time period in seconds to look back
     * @return array Statistics array
     */
    public function getStatistics(int $timePeriod = 86400): array
    {
        $stats = [];
        $now = time();
        $startTime = $now - $timePeriod;
        
        try {
            // Get total attempts
            $stats['total_attempts'] = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(*) FROM `{$this->attemptsTable}` WHERE `timestamp` >= :start_time",
                ['start_time' => $startTime]
            );
            
            // Get active blocks
            $stats['active_blocks'] = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(*) FROM `{$this->blocksTable}` WHERE `blocked_until` > :now",
                ['now' => $now]
            );
            
            // Get total blocks created in period
            $stats['blocks_created'] = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(*) FROM `{$this->blocksTable}` WHERE `created_at` >= :start_time",
                ['start_time' => $startTime]
            );
            
            // Get top routes
            $stats['top_routes'] = $this->dbConnection->queryAndFetchAllAssoc(
                "SELECT `route`, COUNT(*) as attempt_count FROM `{$this->attemptsTable}` WHERE `timestamp` >= :start_time GROUP BY `route` ORDER BY attempt_count DESC LIMIT 10",
                ['start_time' => $startTime]
            );
            
            // Get top IPs
            $stats['top_ips'] = $this->dbConnection->queryAndFetchAllAssoc(
                "SELECT `ip`, COUNT(*) as attempt_count FROM `{$this->attemptsTable}` WHERE `timestamp` >= :start_time GROUP BY `ip` ORDER BY attempt_count DESC LIMIT 10",
                ['start_time' => $startTime]
            );
            
            // Get IP rotation blocks
            $stats['ip_rotation_blocks'] = (int)$this->dbConnection->querySingleValue(
                "SELECT COUNT(*) FROM `{$this->blocksTable}` WHERE `reason` = 'ip_rotation' AND `created_at` >= :start_time",
                ['start_time' => $startTime]
            );
            
            // Add cache status if available
            if ($this->cacheEnabled && $this->cache) {
                $stats['cache_enabled'] = true;
                if (method_exists($this->cache, 'getStats')) {
                    $cacheStats = $this->cache->getStats();
                    $stats['cache_stats'] = $cacheStats;
                }
            } else {
                $stats['cache_enabled'] = false;
            }
            
            return $stats;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting rate limit statistics', [
                    'error' => $e->getMessage()
                ]);
            }
            return ['error' => 'Database error'];
        }
    }
    
    /**
     * Get the list of whitelisted routes
     */
    public function getWhitelistedRoutes(): array
    {
        return $this->whitelistedRoutes;
    }
    
    /**
     * Get the list of protected routes with their configurations
     */
    public function getProtectedRoutes(): array
    {
        return $this->protectedRoutes;
    }
    
    /**
     * Get the configuration for a specific protected route
     */
    public function getProtectedRouteConfig(string $route): ?array
    {
        return $this->protectedRoutes[$route] ?? null;
    }
    
    /**
     * Get the IP rotation threshold
     */
    public function getIpRotationThreshold(): int
    {
        return $this->ipRotationThreshold;
    }
    
    /**
     * Get the IP rotation timeframe
     */
    public function getIpRotationTimeframe(): int
    {
        return $this->ipRotationTimeframe;
    }
    
    /**
     * Get the IP rotation decay time
     */
    public function getIpRotationDecay(): int
    {
        return $this->ipRotationDecay;
    }
}