<?php

namespace baseKRIZAN\Services;

/**
 * Pojednostavljena Container klasa za dependency injection.
 */
class Container
{
    /**
     * Registrirani servisi
     * 
     * @var array
     */
    private array $services = [];
    
    /**
     * Instancirani servisi (singleton)
     * 
     * @var array
     */
    private array $instances = [];
    
    /**
     * Zastavica za izbjegavanje rekurzije tijekom logiranja
     * 
     * @var bool
     */
    private bool $isLogging = false;
    
    /**
     * Definira podršku za tipove servisa
     * 
     * @var array
     */
    private array $serviceScopes = [
        'singleton', // Jedna instanca za cijelu aplikaciju (default)
        'prototype', // Nova instanca za svaki dohvat
        'request'    // Jedna instanca po HTTP zahtjevu
    ];

    /**
     * Registrira servis u container.
     * 
     * @param string $id Identifikator servisa
     * @param mixed $service Servis koji se registrira (klasa, instanca ili factory)
     * @param string $scope Opseg servisa ('singleton', 'prototype', 'request')
     * @return void
     */
    public function register(string $id, $service, string $scope = 'singleton'): void
    {
        if (!in_array($scope, $this->serviceScopes)) {
            throw new \InvalidArgumentException("Nevažeći scope: $scope. Dozvoljene vrijednosti: " . implode(', ', $this->serviceScopes));
        }
        
        $this->services[$id] = [
            'definition' => $service,
            'scope' => $scope
        ];
        
        // Logiraj samo ako ne logiramo već i imamo instancu
        if (!$this->isLogging && isset($this->instances['logger'])) {
            $logger = $this->instances['logger'];
            if (method_exists($logger, 'services')) {
                $this->isLogging = true; // Postavimo zastavicu
                $logger->services('Service registered', [
                    'id' => $id, 
                    'type' => is_object($service) ? get_class($service) : gettype($service),
                    'scope' => $scope
                ]);
                $this->isLogging = false; // Resetiramo zastavicu
            }
        }
    }

    /**
     * Registrira klasu pod više ključeva (puno ime klase, kratko ime i toLowerCase).
     * 
     * @param string $class Naziv klase koja se registrira
     * @param mixed $instance Instanca klase ili null za auto-instanciranje
     * @param string $scope Opseg servisa ('singleton', 'prototype', 'request')
     * @return void
     */
    public function registerClass(string $class, $instance = null, string $scope = 'singleton'): void
    {
        // Extract class name without namespace
        $parts = explode('\\', $class);
        $className = end($parts);
        
        // Register as original key (full namespace)
        $this->register($class, $instance ?? $class, $scope);
        
        // Register as lowercase key
        $this->register(strtolower($className), $instance ?? $class, $scope);
        
        // Also register with original capitalization
        $this->register($className, $instance ?? $class, $scope);
    }

    /**
     * Dodaje više servisa odjednom.
     * 
     * @param array $services Asocijativni niz servisa (id => definicija)
     * @param string $scope Opseg za sve servise u nizu
     * @return void
     */
    public function addServices(array $services, string $scope = 'singleton'): void
    {
        foreach ($services as $id => $service) {
            $this->register($id, $service, $scope);
        }
    }

    /**
     * Dohvaća servis iz containera.
     * 
     * @param string $id Identifikator servisa
     * @return mixed Instanca servisa
     * @throws \RuntimeException Ako servis nije pronađen
     */
    public function get(string $id)
    {
        if (!isset($this->services[$id])) {
            // Logiraj grešku bez poziva getLogger() da izbjegnemo rekurziju
            if (!$this->isLogging && isset($this->instances['logger'])) {
                $logger = $this->instances['logger'];
                if (method_exists($logger, 'error')) {
                    $this->isLogging = true;
                    $logger->error('Service not found in container', [
                        'requested_id' => $id,
                        'available_services' => array_keys($this->services)
                    ]);
                    $this->isLogging = false;
                }
            }
            
            // Pokušaj autowiring ako klasa postoji
            if (class_exists($id)) {
                return $this->autowire($id);
            }
            
            throw new \RuntimeException("Service '$id' not found in container");
        }

        $service = $this->services[$id];
        $scope = $service['scope'] ?? 'singleton';
        $definition = $service['definition'];

        // Return existing instance for singleton scope
        if ($scope === 'singleton' && isset($this->instances[$id])) {
            return $this->instances[$id];
        }
        
        // Za request scope, provjeri postoji li instanca u trenutnom requestu
        if ($scope === 'request') {
            $requestId = $this->getRequestId();
            if (isset($this->instances["{$requestId}_{$id}"])) {
                return $this->instances["{$requestId}_{$id}"];
            }
        }

        // If service is a class name string, instantiate it
        if (is_string($definition) && class_exists($definition)) {
            $instance = $this->autowire($definition);
            
            if ($scope === 'singleton') {
                $this->instances[$id] = $instance;
            } elseif ($scope === 'request') {
                $requestId = $this->getRequestId();
                $this->instances["{$requestId}_{$id}"] = $instance;
            }
            
            return $instance;
        }

        // If service is a factory function, execute it
        if ($definition instanceof \Closure) {
            $instance = $definition($this);
            
            if ($scope === 'singleton') {
                $this->instances[$id] = $instance;
            } elseif ($scope === 'request') {
                $requestId = $this->getRequestId();
                $this->instances["{$requestId}_{$id}"] = $instance;
            }
            
            return $instance;
        }

        // Otherwise return the service as is (already an instance)
        return $definition;
    }

    /**
     * Vraća jedinstveni ID za trenutni zahtjev.
     * Koristi se za upravljanje Request-scoped servisima.
     * 
     * @return string
     */
    private function getRequestId(): string
    {
        static $requestId = null;
        
        if ($requestId === null) {
            $requestId = uniqid('req_');
        }
        
        return $requestId;
    }

    /**
     * Provjerava postoji li servis u containeru.
     * 
     * @param string $id Identifikator servisa
     * @return bool
     */
    public function has(string $id): bool
    {
        return isset($this->services[$id]);
    }

    /**
     * Provjerava postoji li instanca servisa.
     * 
     * @param string $id Identifikator servisa
     * @return bool
     */
    public function hasInstance(string $id): bool
    {
        return isset($this->instances[$id]);
    }

    /**
     * Vraća sve registrirane servise.
     * 
     * @return array Asocijativni niz [id => servis]
     */
    public function getServices(): array
    {
        $services = [];
        foreach ($this->services as $id => $service) {
            $services[$id] = $service['definition'];
        }
        return $services;
    }

    /**
     * Spaja sadržaj drugog containera u ovaj.
     * 
     * @param Container $otherContainer Container čiji sadržaj treba spojiti
     * @return void
     */
    public function merge(Container $otherContainer): void
    {
        $services = $otherContainer->getServices();
        foreach ($services as $id => $service) {
            if (!$this->has($id)) {
                $this->register($id, $service);
            }
        }
    }

    /**
     * Registrira servis kao singleton i odmah ga instancira.
     * 
     * @param string $id Identifikator servisa
     * @param callable $factory Factory funkcija koja stvara instancu
     * @return void
     */
    public function singleton(string $id, callable $factory): void
    {
        if (!isset($this->instances[$id])) {
            $this->instances[$id] = $factory($this);
        }
        
        $this->register($id, function() use ($id) {
            return $this->instances[$id];
        }, 'singleton');
    }
    
    /**
     * Automatski instancira klasu rješavajući sve njene ovisnosti.
     * 
     * @param string $className Puno ime klase za autowiring
     * @return object Instancirana klasa s riješenim ovisnostima
     * @throws \RuntimeException Ako nije moguće riješiti ovisnosti
     */
    public function autowire(string $className): object
    {
        // Ako već imamo instancu, vratimo je
        if (isset($this->instances[$className])) {
            return $this->instances[$className];
        }
        
        // Specijalno rukovanje za poznate singleton klase
        if (method_exists($className, 'getInstance')) {
            $reflection = new \ReflectionMethod($className, 'getInstance');
            
            // Ako je getInstance statička metoda, vjerojatno se radi o sigletonu
            if ($reflection->isStatic()) {
                return $className::getInstance();
            }
        }
        
        try {
            // Koristi reflection za analizu konstruktora
            $reflection = new \ReflectionClass($className);
            
            // Ako nema konstruktora, jednostavno instanciraj
            if (!$constructor = $reflection->getConstructor()) {
                return new $className();
            }
            
            // Pripremi parametre za konstruktor
            $parameters = [];
            
            foreach ($constructor->getParameters() as $param) {
                // Ako parametar ima tip koji nije primitivan, pokušaj autowire
                if ($param->getType() && !$param->getType()->isBuiltin()) {
                    $paramType = $param->getType()->getName();
                    
                    // Posebno rukovanje za container
                    if ($paramType === Container::class) {
                        $parameters[] = $this;
                        continue;
                    }
                    
                    // Provjeri postoji li servis u containeru
                    if ($this->has($paramType)) {
                        $parameters[] = $this->get($paramType);
                        continue;
                    }
                    
                    // Provjeri postoji li servis pod imenom parametra
                    $paramName = $param->getName();
                    if ($this->has($paramName)) {
                        $parameters[] = $this->get($paramName);
                        continue;
                    }
                    
                    // Rekurzivno autowire parametar ako je klasa
                    if (class_exists($paramType)) {
                        $parameters[] = $this->autowire($paramType);
                        continue;
                    }
                }
                
                // Ako parametar ima default vrijednost
                if ($param->isDefaultValueAvailable()) {
                    $parameters[] = $param->getDefaultValue();
                    continue;
                }
                
                // Ako parametar dozvoljava null
                if ($param->allowsNull()) {
                    $parameters[] = null;
                    continue;
                }
                
                throw new \RuntimeException("Cannot autowire parameter '{$param->getName()}' for class '$className'");
            }
            
            // Stvori novu instancu s pripremljenim parametrima
            return $reflection->newInstanceArgs($parameters);
            
        } catch (\ReflectionException $e) {
            throw new \RuntimeException("Autowiring failed for class '$className': " . $e->getMessage());
        }
    }
    
    /**
     * Registrira servis sa 'prototype' scope-om (nova instanca za svaki get).
     * 
     * @param string $id Identifikator servisa
     * @param mixed $service Servis koji se registrira
     * @return void
     */
    public function prototype(string $id, $service): void
    {
        $this->register($id, $service, 'prototype');
    }
    
    /**
     * Registrira servis sa 'request' scope-om (jedna instanca po zahtjevu).
     * 
     * @param string $id Identifikator servisa
     * @param mixed $service Servis koji se registrira
     * @return void
     */
    public function request(string $id, $service): void
    {
        $this->register($id, $service, 'request');
    }
    
    /**
     * Čisti instance za određeni scope.
     * 
     * @param string $scope Scope za čišćenje ('singleton', 'request')
     * @return void
     */
    public function clearScope(string $scope): void
    {
        if ($scope === 'singleton') {
            $this->instances = [];
        } elseif ($scope === 'request') {
            $requestId = $this->getRequestId();
            foreach (array_keys($this->instances) as $key) {
                if (strpos($key, "{$requestId}_") === 0) {
                    unset($this->instances[$key]);
                }
            }
        }
    }
}