<?php

namespace baseKRIZAN\Services;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use baseKRIZAN\Error\Logger;

class EmailService {
    private Logger $logger;
    private array $config;
    
    public function __construct(Logger $logger) {
        $this->logger = $logger;
        $this->config = [
            'smtp_host' => \baseKRIZAN\Config\Config::get('mail.host'),
            'smtp_port' => \baseKRIZAN\Config\Config::get('mail.port'),
            'smtp_secure' => \baseKRIZAN\Config\Config::get('mail.encryption'),
            'smtp_username' => \baseKRIZAN\Config\Config::get('mail.username'),
            'smtp_password' => \baseKRIZAN\Config\Config::get('mail.password'),
            'from_address' => \baseKRIZAN\Config\Config::get('mail.fromadress'),
            'from_name' => \baseKRIZAN\Config\Config::get('mail.fromname')
        ];
    }

    public function send(
        string $to,
        string $subject,
        string $body,
        array $attachments = []
    ): bool {
        try {
            // Logiraj početak slanja
            $this->logger->services('Starting email send', [
                'to' => $to,
                'subject' => $subject,
                'has_attachments' => !empty($attachments)
            ]);
            
            $mail = new PHPMailer(true);
            
            // Server settings
            $mail->isSMTP();
            $mail->Host = $this->config['smtp_host'];
            $mail->Port = $this->config['smtp_port'];
            $mail->SMTPSecure = $this->config['smtp_secure'];
            $mail->SMTPAuth = true;
            $mail->Username = $this->config['smtp_username'];
            $mail->Password = $this->config['smtp_password'];
    
            // Recipients
            $mail->setFrom(
                $this->config['from_address'], 
                $this->config['from_name']
            );
            $mail->addAddress($this->sanitizeEmail($to));
    
            // Attachments
            foreach ($attachments as $attachment) {
                if (isset($attachment['path']) && file_exists($attachment['path'])) {
                    $mail->addAttachment(
                        $attachment['path'],
                        $attachment['name'] ?? basename($attachment['path'])
                    );
                }
            }
    
            // Content
            $mail->isHTML(true);
            $mail->Subject = $this->sanitizeString($subject);
            $mail->Body = $body;
            $mail->AltBody = strip_tags($body);
            
            $mail->CharSet = 'UTF-8';
            $mail->Encoding = 'base64';
    
            $result = $mail->send();
            
            $this->logger->services('Email sent successfully', [
                'to' => $to,
                'subject' => $subject
            ]);
            
            return $result;
    
        } catch (Exception $e) {
            $this->logger->error('Email sending failed', [
                'error' => $e->getMessage(),
                'to' => $to,
                'subject' => $subject
            ]);
            
            return false;
        }
    }
    
    private function sanitizeEmail(string $email): string {
        $email = filter_var(trim($email), FILTER_SANITIZE_EMAIL);
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new \InvalidArgumentException("Invalid email address: {$email}");
        }
        return $email;
    }
    
    private function sanitizeString(string $str): string {
        return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
    }
}