<?php

namespace baseKRIZAN\Services;

use GuzzleHttp\Client;
use baseKRIZAN\Error\Logger;
use Firebase\JWT\JWT;
use Models\FCMTokenModel;

/**
 * FCMService - Handles Firebase Cloud Messaging operations
 * 
 * This service is responsible for sending push notifications via Firebase Cloud Messaging (FCM).
 * It handles token generation, device-specific notification formatting, and notification delivery.
 */
class FCMService
{
    private const FCM_ENDPOINT = 'https://fcm.googleapis.com/v1/projects/%s/messages:send';
    private const OAUTH_TOKEN_ENDPOINT = 'https://oauth2.googleapis.com/token';
    
    private Logger $logger;
    private FCMTokenModel $fcmTokenModel;
    private string $projectId;
    private Client $client;
    private string $basePath;
    private string $accessToken = '';
    private int $tokenExpiry = 0;
    
    /**
     * Constructor
     * 
     * @param Logger $logger System logger
     * @param FCMTokenModel $fcmTokenModel FCM token model
     */
    public function __construct(Logger $logger, FCMTokenModel $fcmTokenModel)
    {
        $this->logger = $logger;
        $this->fcmTokenModel = $fcmTokenModel;
        $this->projectId = \baseKRIZAN\Config\Config::get('google.firebaseprojid') ?? '';
        $this->client = new Client();
        $this->basePath = \baseKRIZAN\Config\Config::get('paths.app_url');
    }
    
    /**
     * Save user token for push notifications
     */
    public function saveUserToken(int $userId, string $email, string $token, string $deviceType = 'unknown'): bool
    {
        try {
            // Validate userId
            if ($userId <= 0) {
                $this->logger->notification('Invalid user ID when saving FCM token', [
                    'user_id' => $userId
                ]);
                return false;
            }

            $this->logger->notification('Saving user FCM token', [
                'user_id' => $userId,
                'email' => $email,
                'device_type' => $deviceType,
                'token_length' => strlen($token)
            ]);

            // Check if the token is already saved for this user
            $existingToken = $this->fcmTokenModel->getUserToken($userId);
            
            // Only save if token is different
            if ($existingToken !== $token) {
                $result = $this->fcmTokenModel->saveUserToken($userId, $email, $token, $deviceType);
                
                if ($result) {
                    $this->logger->notification('FCM token saved successfully', [
                        'user_id' => $userId
                    ]);
                } else {
                    $this->logger->notification('Failed to save FCM token', [
                        'user_id' => $userId
                    ]);
                }
                
                return $result;
            }
            
            $this->logger->notification('FCM token unchanged, no save needed', [
                'user_id' => $userId
            ]);
            
            // Token unchanged, consider it a success
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Failed to save user FCM token', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return false;
        }
    }
    
    /**
     * Send FCM notification to a user
     * 
     * @param int $userId User ID
     * @param string $title Notification title
     * @param string $body Notification body
     * @param array $data Additional data for the notification
     * @return bool Success status
     */
    public function sendUserNotification(
        int $userId,
        string $title,
        string $body,
        array $data = []
    ): bool {
        try {
            // Get all user device tokens
            $userTokens = $this->fcmTokenModel->getAllUserTokens($userId);
            
            if (empty($userTokens)) {
                $this->logger->notification('No device tokens for user', [
                    'user_id' => $userId
                ]);
                return false;
            }
            
            $successCount = 0;
            
            // Send to each device token
            foreach ($userTokens as $tokenInfo) {
                $token = $tokenInfo['device_token'];
                $deviceType = $tokenInfo['device_type'];
                
                $result = $this->sendNotification(
                    $token,
                    $title,
                    $body,
                    $data,
                    $deviceType
                );
                
                if ($result) {
                    $successCount++;
                }
            }
            
            $this->logger->notification('Push notifications sent to user', [
                'user_id' => $userId,
                'success_count' => $successCount,
                'total_tokens' => count($userTokens)
            ]);
            
            return $successCount > 0;
        } catch (\Exception $e) {
            $this->logger->error('Failed to send push notifications to user', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return false;
        }
    }

    /**
     * Send a push notification via FCM
     * 
     * @param string $token Device token
     * @param string $title Notification title
     * @param string $body Notification body
     * @param array $data Additional data for the notification
     * @param string $deviceType Device type (mobile, desktop, etc.)
     * @return bool Success status
     */
    public function sendNotification(
        string $token,
        string $title,
        string $body,
        array $data = [],
        string $deviceType = 'unknown'
    ): bool {
        try {
            $this->logger->notification('FCM preparing notification', [
                'device_type' => $deviceType,
                'title' => $title
            ]);
            
            $accessToken = $this->getAccessToken();
            $url = sprintf(self::FCM_ENDPOINT, $this->projectId);
            
            // Get route from data or use default
            $route = isset($data['route']) ? $data['route'] : '';
            $clickAction = $this->basePath . '/' . ltrim($route, '/');
            
            $isMobile = ($deviceType === 'mobile');
            
            $this->logger->notification('FCM notification details', [
                'device_type' => $deviceType,
                'is_mobile' => $isMobile,
                'token_length' => strlen($token)
            ]);
            
            // Build message based on device type
            $message = $this->buildNotificationMessage($token, $title, $body, $data, $clickAction, $isMobile);
    
            $clientConfig = $this->getClientConfig();
            $clientConfig['headers'] = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $accessToken
            ];
    
            // For debugging, log the JSON payload
            $this->logger->notification('FCM payload', [
                'payload' => json_encode($message, JSON_PRETTY_PRINT)
            ]);
    
            $response = $this->client->post($url, ['json' => $message] + $clientConfig);
            $result = json_decode($response->getBody(), true);
            
            $this->logger->notification('FCM notification result', [
                'status_code' => $response->getStatusCode(),
                'result' => $result,
                'device_type' => $deviceType
            ]);
            
            return $response->getStatusCode() === 200;
    
        } catch (\Exception $e) {
            $this->logger->error('Push notification failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'device_type' => $deviceType
            ]);
            return false;
        }
    }

    /**
     * Get access token for FCM
     * 
     * @return string Access token
     * @throws \Exception If unable to get token
     */
    private function getAccessToken(): string
    {
        // Return cached token if not expired
        if (!empty($this->accessToken) && time() < $this->tokenExpiry) {
            return $this->accessToken;
        }
        
        try {
            $this->logger->notification('Generating FCM access token');
            
            // Get service account json content
            $serviceAccountJson = $this->getServiceAccountContent();
            
            $now = time();
            $payload = [
                'iss' => $serviceAccountJson['client_email'],
                'sub' => $serviceAccountJson['client_email'],
                'aud' => self::OAUTH_TOKEN_ENDPOINT,
                'iat' => $now,
                'exp' => $now + 3600,
                'scope' => 'https://www.googleapis.com/auth/firebase.messaging'
            ];

            $jwt = JWT::encode(
                $payload, 
                $serviceAccountJson['private_key'], 
                'RS256'
            );

            $clientConfig = $this->getClientConfig();
            $response = (new Client($clientConfig))->post(self::OAUTH_TOKEN_ENDPOINT, [
                'form_params' => [
                    'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                    'assertion' => $jwt
                ]
            ]);

            $tokenData = json_decode($response->getBody(), true);
            
            if (!isset($tokenData['access_token'])) {
                throw new \Exception('No access token in response');
            }

            $this->logger->notification('Successfully generated FCM access token', [
                'expires_in' => $tokenData['expires_in'] ?? 3600
            ]);
            
            // Cache token and expiry
            $this->accessToken = $tokenData['access_token'];
            $this->tokenExpiry = time() + ($tokenData['expires_in'] ?? 3600) - 300; // 5 minute buffer
            
            return $this->accessToken;
            
        } catch (\Exception $e) {
            $this->logger->error('Access token generation failed', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Build notification message based on device type
     * 
     * @param string $token Device token
     * @param string $title Notification title
     * @param string $body Notification body
     * @param array $data Additional data
     * @param string $clickAction URL to open when notification is clicked
     * @param bool $isMobile Whether the device is mobile
     * @return array Message payload for FCM
     */
    private function buildNotificationMessage(
        string $token, 
        string $title, 
        string $body, 
        array $data, 
        string $clickAction, 
        bool $isMobile
    ): array {
        // Common message structure
        $message = [
            'message' => [
                'token' => $token,
                'notification' => [
                    'title' => $title,
                    'body' => $body
                ],
                'data' => $data ?: new \stdClass()
            ]
        ];
        
        // Add web-specific configuration
        $message['message']['webpush'] = [
            'headers' => [
                'Urgency' => 'high'
            ],
            'notification' => [
                'title' => $title,
                'body' => $body,
                'icon' => $this->basePath . NotificationTypeRegistry::getIconForType($data['type'] ?? 'general'),
                'click_action' => $clickAction,
                'requireInteraction' => true
            ],
            'fcm_options' => [
                'link' => $clickAction
            ]
        ];
        
        // Add Android config
        if ($isMobile) {
            $message['message']['android'] = [
                'notification' => [
                    'title' => $title,
                    'body' => $body,
                    'icon' => 'notification_icon',
                    'color' => '#4285F4',
                    'sound' => 'default',
                    'click_action' => $clickAction,
                    'notification_priority' => 'PRIORITY_HIGH'
                ],
                'priority' => 'high'
            ];
            
            // Add iOS config
            $message['message']['apns'] = [
                'headers' => [
                    'apns-priority' => '10'
                ],
                'payload' => [
                    'aps' => [
                        'alert' => [
                            'title' => $title,
                            'body' => $body
                        ],
                        'sound' => 'default',
                        'badge' => 1,
                        'content-available' => 1,
                        'mutable-content' => 1
                    ]
                ]
            ];
        }
        
        return $message;
    }
    
    /**
     * Get client configuration based on environment
     * 
     * @return array Client configuration
     */
    private function getClientConfig(): array
    {
        $config = [];
        
        // For development, disable SSL verification
        if (\baseKRIZAN\Config\Config::get('environment') === 'development') {
            $config['verify'] = false;
        } else {
            // For production, use CA bundle if available
            $caBundlePath = \baseKRIZAN\Config\Config::get('paths.config') . '/ssl/cacert.pem';
            if (file_exists($caBundlePath)) {
                $config['verify'] = $caBundlePath;
            }
            // Else use default system CA bundle
        }
        
        return $config;
    }
    
    /**
     * Get service account content from file
     * 
     * @return array Service account JSON data
     * @throws \Exception If service account file not found
     */
    private function getServiceAccountContent(): array
    {
        $credentialsPaths = [
            \baseKRIZAN\Config\Config::get('paths.config') . '/firebase/service-account.json',
            \baseKRIZAN\Config\Config::get('google.firebaseappcred') ?? ''
        ];
        
        $serviceAccountJson = null;
        
        // Try each path until we find a valid file
        foreach ($credentialsPaths as $path) {
            if (file_exists($path) && is_readable($path)) {
                $serviceAccountJson = json_decode(file_get_contents($path), true);
                if ($serviceAccountJson) {
                    break;
                }
            }
        }
        
        // If still no service account file, throw an error
        if (!$serviceAccountJson) {
            throw new \Exception('Service account file not found');
        }
        
        return $serviceAccountJson;
    }

    /**
     * Send FCM notifications to multiple users
     * 
     * @param array $userIds Array of user IDs
     * @param string $title Notification title
     * @param string $body Notification body
     * @param array $data Additional data for the notification
     * @return int Number of users that received notifications
     */
    public function sendBulkNotifications(
        array $userIds,
        string $title,
        string $body,
        array $data = []
    ): int {
        $successCount = 0;
        
        foreach ($userIds as $userId) {
            $result = $this->sendUserNotification($userId, $title, $body, $data);
            
            if ($result) {
                $successCount++;
            }
        }
        
        return $successCount;
    }
}