<?php

namespace baseKRIZAN\Services;

use baseKRIZAN\Error\Logger;
use Firebase\JWT\JWT;

/**
 * FirebaseAuthService - Firebase authentication service
 * 
 * This service handles Firebase authentication, including token generation
 * for both FCM and Realtime Database.
 */
class FirebaseAuthService
{
    private Logger $logger;
    
    /**
     * Constructor
     * 
     * @param Logger $logger System logger
     */
    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }
    
    /**
     * Generate a Firebase authentication token for a user
     * 
     * @param int $userId User ID
     * @param string $userEmail User email
     * @param bool $isAdmin Whether the user is an admin
     * @return string|null Firebase auth token or null on failure
     */
    public function generateAuthToken(int $userId, string $userEmail, bool $isAdmin = false): ?string
    {
        try {
            $this->logger->notification('Generating Firebase auth token', [
                'user_id' => $userId,
                'is_admin' => $isAdmin
            ]);
            
            // Get service account path
            $serviceAccountPath = \baseKRIZAN\Config\Config::get('google.firebaseappcred');
            
            if (!file_exists($serviceAccountPath)) {
                throw new \Exception('Service account file not found at: ' . $serviceAccountPath);
            }
            
            $serviceAccount = json_decode(file_get_contents($serviceAccountPath), true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \Exception('Invalid service account JSON: ' . json_last_error_msg());
            }
            
            // Prepare custom claims for token
            $claims = [
                'admin' => $isAdmin,
                'email' => $userEmail,
                'email_verified' => true,  // Assumption: email is verified
                'database_access' => true,  // General flag for database access
                'notification_path' => "/notifications/{$userId}"  // Specific path this user can access
            ];
            
            // Generate token
            $token = $this->createCustomToken($serviceAccount, (string)$userId, $claims);
            
            return $token;
        } catch (\Exception $e) {
            $this->logger->error('Error generating Firebase auth token', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId
            ]);
            
            return null;
        }
    }
    
    /**
     * Create Firebase custom token for authentication
     * 
     * @param array $serviceAccount Service account data
     * @param string $uid User ID
     * @param array $claims Custom claims
     * @return string Firebase custom token
     * @throws \Exception If service account is invalid
     */
    private function createCustomToken(array $serviceAccount, string $uid, array $claims = []): string
    {
        if (empty($serviceAccount['client_email']) || empty($serviceAccount['private_key'])) {
            throw new \Exception('Service account missing required fields');
        }
        
        $now = time();
        
        $payload = [
            'iss' => $serviceAccount['client_email'],
            'sub' => $serviceAccount['client_email'],
            'aud' => 'https://identitytoolkit.googleapis.com/google.identity.identitytoolkit.v1.IdentityToolkit',
            'iat' => $now,
            'exp' => $now + 3600, // Token valid for 1 hour
            'uid' => $uid,
            'claims' => $claims
        ];
        
        // Ensure private_key is properly formatted
        $privateKey = str_replace("\\n", "\n", $serviceAccount['private_key']);
        
        return JWT::encode($payload, $privateKey, 'RS256');
    }
}