<?php
namespace baseKRIZAN\Services;

use baseKRIZAN\Error\Logger;
use GuzzleHttp\Client;

/**
 * Firebase Realtime Database Service
 * 
 * This service handles synchronization of notifications with Firebase Realtime Database.
 */
class FirebaseRealtimeService
{
    private const FIREBASE_DB_URL = 'https://%s.firebasedatabase.app';
    
    private Logger $logger;
    private FirebaseAuthService $firebaseAuthService;
    private string $projectId;
    private string $databaseUrl;
    private Client $client;
    private ?string $adminToken = null;
    private int $tokenExpiry = 0;
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, FirebaseAuthService $firebaseAuthService)
    {
        $this->logger = $logger;
        $this->firebaseAuthService = $firebaseAuthService;
        $this->projectId = \baseKRIZAN\Config\Config::get('google.firebaseprojid') ?? '';
        
        // Use URL from config with fallback to constructed URL
        $this->databaseUrl = \baseKRIZAN\Config\Config::get('google.firebasedatabaseurl') ??
            sprintf(self::FIREBASE_DB_URL, $this->projectId);
        
        $this->client = new Client();
        
        $this->logger->notification('Firebase Realtime Database URL', [
            'url' => $this->databaseUrl
        ]);
    }
    
    /**
     * Save a notification to Firebase Realtime Database
     * 
     * @param int $userId User ID
     * @param string $notificationId Notification ID
     * @param array $data Notification data
     * @return bool Success status
     */
    public function saveNotification(int $userId, string $notificationId, array $data): bool
    {
        try {
            // Get admin database token
            $token = $this->getAdminDatabaseToken();
            if (!$token) {
                throw new \Exception('Failed to obtain Firebase admin database token');
            }
            
            // Construct URL with auth token as query parameter
            $url = $this->databaseUrl . "/notifications/{$userId}/items/{$notificationId}.json?auth=" . urlencode($token);
            
            $this->logger->notification('Sending notification to Firebase', [
                'user_id' => $userId,
                'notification_id' => $notificationId,
                'url' => preg_replace('/auth=([^&]+)/', 'auth=REDACTED', $url)
            ]);
            
            // Process data
            $processedData = $this->processNotificationData($data);
            
            // Send PUT request
            $response = $this->client->request('PUT', $url, [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'json' => $processedData,
                'verify' => $this->shouldVerifySSL()
            ]);
            
            $statusCode = $response->getStatusCode();
            
            if ($statusCode >= 200 && $statusCode < 300) {
                // Update parent node directly
                return $this->updateNotificationCount($userId);
            } else {
                $this->logger->notification('Failed to save notification to Firebase', [
                    'status_code' => $statusCode,
                    'user_id' => $userId
                ]);
                return false;
            }
        } catch (\Exception $e) {
            $this->logger->error('Error saving notification to Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId
            ]);
            return false;
        }
    }
    
    /**
     * Get admin token for Firebase Realtime Database
     * 
     * @return string|null Admin token or null on failure
     */
    private function getAdminDatabaseToken(): ?string
    {
        // Return cached token if not expired
        if ($this->adminToken !== null && time() < $this->tokenExpiry) {
            return $this->adminToken;
        }
        
        // Try database secret first (legacy method)
        $databaseSecret = \baseKRIZAN\Config\Config::get('google.firebasedatabasesecret') ?? '';
        
        if (!empty($databaseSecret)) {
            $this->logger->notification('Using Firebase database secret for auth');
            $this->adminToken = $databaseSecret;
            $this->tokenExpiry = time() + 3600; // 1 hour
            return $databaseSecret;
        }
        
        // Get a special admin token with full database access
        $adminToken = $this->firebaseAuthService->generateAuthToken(
            0, // Special admin user ID
            'firebase-admin@example.com', 
            true // is admin
        );
        
        if ($adminToken) {
            $this->adminToken = $adminToken;
            $this->tokenExpiry = time() + 3600 - 300; // 1 hour minus 5 min buffer
            return $adminToken;
        }
        
        $this->logger->error('Failed to generate Firebase admin database token');
        return null;
    }
    
    /**
     * Process notification data before sending to Firebase
     * 
     * @param array $data Notification data
     * @return array Processed data
     */
    private function processNotificationData(array $data): array
    {
        // Ensure timestamp is integer
        if (!isset($data['timestamp'])) {
            $data['timestamp'] = (int)round(microtime(true) * 1000); // Milliseconds
        } else if (!is_int($data['timestamp'])) {
            $data['timestamp'] = (int)$data['timestamp'];
        }
        
        // Add ID to data to be available in client
        $data['id'] = $data['id'] ?? '';
        
        // Normalize boolean fields
        if (isset($data['is_read'])) {
            $data['is_read'] = (bool)$data['is_read'];
        }
        if (isset($data['is_solved'])) {
            $data['is_solved'] = (bool)$data['is_solved'];
        }
        if (isset($data['is_sticky'])) {
            $data['is_sticky'] = (bool)$data['is_sticky'];
        }
        
        return $data;
    }
    
    /**
     * Determine whether to verify SSL in requests
     * 
     * @return bool|string Whether to verify SSL or path to CA bundle
     */
    private function shouldVerifySSL()
    {
        // For development, disable SSL verification
        if (\baseKRIZAN\Config\Config::get('environment') === 'development') {
            return false;
        }
        
        // For production, use CA bundle if available
        $caBundlePath = \baseKRIZAN\Config\Config::get('paths.config') . '/ssl/cacert.pem';
        if (file_exists($caBundlePath)) {
            return $caBundlePath;
        }
        
        // Default: use system CA bundle
        return true;
    }
    
    /**
     * Update parent notification data (count, hasNewNotifications, updatedAt)
     * 
     * @param int $userId User ID
     * @return bool Success status
     */
    private function updateNotificationCount(int $userId): bool
    {
        try {
            // Get admin token
            $token = $this->getAdminDatabaseToken();
            if (!$token) {
                throw new \Exception('Failed to obtain Firebase admin database token');
            }
            
            // Construct URL with auth token as query parameter
            $url = $this->databaseUrl . "/notifications/{$userId}.json?auth=" . urlencode($token);
            
            // Send request 
            $updateResponse = $this->client->request('PATCH', $url, [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'json' => [
                    'hasNewNotifications' => true,
                    'updatedAt' => (int)round(microtime(true) * 1000)
                ],
                'verify' => $this->shouldVerifySSL()
            ]);
            
            $this->logger->notification('Updated parent notification data', [
                'user_id' => $userId,
                'status_code' => $updateResponse->getStatusCode()
            ]);
            
            return $updateResponse->getStatusCode() >= 200 && $updateResponse->getStatusCode() < 300;
            
        } catch (\Exception $e) {
            $this->logger->error('Error updating parent notification data', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            
            return false;
        }
    }
    
    /**
     * Sync notifications with Firebase Realtime Database
     * 
     * @param int $userId User ID
     * @param array $notifications Notifications from database
     * @param bool $cleanUp Whether to remove notifications from Firebase that don't exist in the database
     * @return bool Success status
     */
    public function syncNotifications(int $userId, array $notifications, bool $cleanUp = false): bool
    {
        try {
            $successCount = 0;
            $totalCount = count($notifications);
            $deletedCount = 0;
            
            // Get admin database token
            $token = $this->getAdminDatabaseToken();
            if (!$token) {
                throw new \Exception('Failed to obtain Firebase admin database token');
            }
            
            // If we want to clean up, first get all notifications from Firebase
            $firebaseNotifications = [];
            if ($cleanUp) {
                $url = $this->databaseUrl . "/notifications/{$userId}/items.json?auth=" . urlencode($token);
                
                $response = $this->client->request('GET', $url, [
                    'verify' => $this->shouldVerifySSL()
                ]);
                
                if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
                    $firebaseNotifications = json_decode($response->getBody(), true) ?: [];
                }
            }
            
            // Create a map of notifications from database
            $databaseNotificationsMap = [];
            foreach ($notifications as $notification) {
                $databaseNotificationsMap[(string)$notification['id']] = true;
            }
            
            // Sync notifications from database
            foreach ($notifications as $notification) {
                // Add timestamp if missing
                if (!isset($notification['timestamp']) && isset($notification['created_at'])) {
                    $date = new \DateTime($notification['created_at'], 
                        new \DateTimeZone(\baseKRIZAN\Config\Config::get('defaulttimezone') ?: 'UTC'));
                    $notification['timestamp'] = (int)($date->getTimestamp() * 1000);
                }
                
                // Construct URL with auth token
                $url = $this->databaseUrl . "/notifications/{$userId}/items/{$notification['id']}.json?auth=" . urlencode($token);
                
                // Normalize boolean fields
                if (isset($notification['is_read'])) {
                    $notification['is_read'] = (bool)$notification['is_read'];
                }
                if (isset($notification['is_solved'])) {
                    $notification['is_solved'] = (bool)$notification['is_solved'];
                }
                if (isset($notification['is_sticky'])) {
                    $notification['is_sticky'] = (bool)$notification['is_sticky'];
                }
                
                $response = $this->client->request('PUT', $url, [
                    'headers' => [
                        'Content-Type' => 'application/json'
                    ],
                    'json' => $notification,
                    'verify' => $this->shouldVerifySSL()
                ]);
                
                if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
                    $successCount++;
                }
            }
            
            // Delete notifications from Firebase that don't exist in database
            if ($cleanUp && !empty($firebaseNotifications)) {
                foreach ($firebaseNotifications as $id => $data) {
                    if (!isset($databaseNotificationsMap[$id])) {
                        $deleteUrl = $this->databaseUrl . "/notifications/{$userId}/items/{$id}.json?auth=" . urlencode($token);
                        $deleteResponse = $this->client->request('DELETE', $deleteUrl, [
                            'verify' => $this->shouldVerifySSL()
                        ]);
                        
                        if ($deleteResponse->getStatusCode() >= 200 && $deleteResponse->getStatusCode() < 300) {
                            $deletedCount++;
                        }
                    }
                }
            }
            
            // Update parent node directly
            if ($successCount > 0) {
                $parentUrl = $this->databaseUrl . "/notifications/{$userId}.json?auth=" . urlencode($token);
                
                $this->client->request('PATCH', $parentUrl, [
                    'headers' => [
                        'Content-Type' => 'application/json'
                    ],
                    'json' => [
                        'hasNewNotifications' => true,
                        'updatedAt' => (int)round(microtime(true) * 1000)
                    ],
                    'verify' => $this->shouldVerifySSL()
                ]);
            }
            
            return $successCount > 0 || $deletedCount > 0;
        } catch (\Exception $e) {
            $this->logger->error('Failed to sync notifications with Firebase', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return false;
        }
    }
    
    /**
     * Sync read status to Firebase
     * 
     * @param int $userId User ID
     * @param int|null $notificationId Specific notification ID or null for all
     * @param array $notifications Notifications to mark as read
     * @return bool Success status
     */
    public function syncReadStatus(int $userId, ?int $notificationId = null, array $notifications = []): bool
    {
        try {
            // Get admin database token
            $token = $this->getAdminDatabaseToken();
            if (!$token) {
                throw new \Exception('Failed to obtain Firebase admin database token');
            }
            
            // Update single notification
            if ($notificationId !== null) {
                $url = $this->databaseUrl . "/notifications/{$userId}/items/{$notificationId}.json?auth=" . urlencode($token);
                
                $response = $this->client->request('PATCH', $url, [
                    'headers' => [
                        'Content-Type' => 'application/json'
                    ],
                    'json' => [
                        'is_read' => true,
                        'updated_at' => (int)round(microtime(true) * 1000)
                    ],
                    'verify' => $this->shouldVerifySSL()
                ]);
                
                return $response->getStatusCode() >= 200 && $response->getStatusCode() < 300;
            }
            
            // No notifications provided and no specific ID, nothing to do
            if (empty($notifications)) {
                return true;
            }
            
            $successCount = 0;
            
            // Update multiple notifications
            foreach ($notifications as $notification) {
                // Skip already read notifications
                if ($notification['is_read']) {
                    continue;
                }
                
                // Skip sticky notifications (they don't get marked as read by markNotificationsAsRead)
                if ($notification['is_sticky']) {
                    continue;
                }
                
                // Construct URL with auth token as query parameter
                $url = $this->databaseUrl . "/notifications/{$userId}/items/{$notification['id']}.json?auth=" . urlencode($token);
                
                $response = $this->client->request('PATCH', $url, [
                    'headers' => [
                        'Content-Type' => 'application/json'
                    ],
                    'json' => [
                        'is_read' => true,
                        'updated_at' => (int)round(microtime(true) * 1000)
                    ],
                    'verify' => $this->shouldVerifySSL()
                ]);
                
                if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
                    $successCount++;
                }
            }
            
            $this->logger->notification('Synced read status to Firebase', [
                'user_id' => $userId,
                'success_count' => $successCount,
                'notification_id' => $notificationId
            ]);
            
            return $successCount > 0 || empty($notifications);
        } catch (\Exception $e) {
            $this->logger->error('Failed to sync read status to Firebase', [
                'error' => $e->getMessage(),
                'user_id' => $userId,
                'notification_id' => $notificationId
            ]);
            return false;
        }
    }
    
    /**
     * Update an existing notification in Firebase Realtime Database
     *
     * @param int $userId The recipient user ID
     * @param string $notificationId Notification ID
     * @param array $updateData Fields to update
     * @return bool Success status
     */
    public function updateNotification(int $userId, string $notificationId, array $updateData): bool
    {
        try {
            // Get admin database token
            $token = $this->getAdminDatabaseToken();
            if (!$token) {
                throw new \Exception('Failed to obtain Firebase admin database token');
            }
            
            // Construct URL with auth token as query parameter
            $url = $this->databaseUrl . "/notifications/{$userId}/items/{$notificationId}.json?auth=" . urlencode($token);
            
            // Add update timestamp if not provided
            if (!isset($updateData['updated_at'])) {
                $updateData['updated_at'] = (int)round(microtime(true) * 1000); // Milliseconds
            }
            
            // Normalize boolean fields
            if (isset($updateData['is_read'])) {
                $updateData['is_read'] = (bool)$updateData['is_read'];
            }
            if (isset($updateData['is_solved'])) {
                $updateData['is_solved'] = (bool)$updateData['is_solved'];
            }
            if (isset($updateData['is_sticky'])) {
                $updateData['is_sticky'] = (bool)$updateData['is_sticky'];
            }
            
            $response = $this->client->request('PATCH', $url, [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'json' => $updateData,
                'verify' => $this->shouldVerifySSL()
            ]);
            
            $statusCode = $response->getStatusCode();
            
            if ($statusCode >= 200 && $statusCode < 300) {
                $this->logger->notification('Notification updated in Firebase', [
                    'user_id' => $userId,
                    'notification_id' => $notificationId
                ]);
                return true;
            } else {
                $this->logger->notification('Failed to update notification in Firebase', [
                    'status_code' => $statusCode,
                    'user_id' => $userId
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error('Error updating notification in Firebase', [
                'error' => $e->getMessage(),
                'user_id' => $userId,
                'notification_id' => $notificationId
            ]);
            return false;
        }
    }
}