<?php

namespace baseKRIZAN\Services;

use baseKRIZAN\Error\Logger;

/**
 * NotificationEnhancerService - Enhances notifications with FCM and Realtime DB
 * 
 * This service integrates the standard notification system with FCM push
 * notifications and Firebase Realtime Database synchronization based on
 * environment configuration settings.
 */
class NotificationEnhancerService
{
    private Logger $logger;
    private NotificationService $notificationService;
    private ?FCMService $fcmService;
    private ?FirebaseRealtimeService $firebaseRealtimeService;
    
    // Feature flags from config
    private bool $fcmEnabled;
    private bool $realtimeDbEnabled;
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger,
        NotificationService $notificationService,
        ?FCMService $fcmService = null,
        ?FirebaseRealtimeService $firebaseRealtimeService = null
    ) {
        $this->logger = $logger;
        $this->notificationService = $notificationService;
        $this->fcmService = $fcmService;
        $this->firebaseRealtimeService = $firebaseRealtimeService;
        
        // Load feature flags from config
        $this->fcmEnabled = (bool)\baseKRIZAN\Config\Config::get('notifications.fcm_enabled');
        $this->realtimeDbEnabled = (bool)\baseKRIZAN\Config\Config::get('notifications.realtime_db_enabled');
        
        $this->logger->notification('Notification enhancer initialized', [
            'fcm_enabled' => $this->fcmEnabled,
            'realtime_db_enabled' => $this->realtimeDbEnabled
        ]);
    }
    
    /**
     * Check if FCM is enabled
     * 
     * @return bool Whether FCM is enabled
     */
    public function isFcmEnabled(): bool
    {
        return $this->fcmEnabled && $this->fcmService !== null;
    }
    
    /**
     * Check if Realtime DB is enabled
     * 
     * @return bool Whether Realtime DB is enabled
     */
    public function isRealtimeDbEnabled(): bool
    {
        return $this->realtimeDbEnabled && $this->firebaseRealtimeService !== null;
    }
    
    /**
     * Create an enhanced notification with push and realtime sync
     * 
     * @param array $data Notification data
     * @param array $recipients Recipient user IDs
     * @return int|null ID of the first created notification, or null on failure
     */
    public function createNotification(array $data, array $recipients = []): ?int
    {
        try {
            // First create the standard notification
            $notificationId = $this->notificationService->createNotification($data, $recipients);
            
            if (!$notificationId) {
                return null;
            }
            
            // If FCM is enabled and service is available and push is requested, send push notifications
            if ($this->fcmEnabled && $this->fcmService && ($data['send_push'] ?? true)) {
                $this->sendPushNotifications($data, $recipients);
            }
            
            // If Realtime DB is enabled and service is available, sync with Firebase
            if ($this->realtimeDbEnabled && $this->firebaseRealtimeService) {
                $this->syncWithFirebase($data, $recipients, $notificationId);
            }
            
            return $notificationId;
        } catch (\Exception $e) {
            $this->logger->error('Error creating enhanced notification', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }
    
    /**
     * Send push notifications via FCM
     * 
     * @param array $data Notification data
     * @param array $recipients Recipient user IDs
     * @return int Number of successful notifications
     */
    private function sendPushNotifications(array $data, array $recipients): int
    {
        // Skip if FCM is disabled or service is not available
        if (!$this->fcmEnabled || !$this->fcmService) {
            return 0;
        }
        
        $title = $data['title'] ?? NotificationTypeRegistry::getTitle($data['type'] ?? 'general');
        $body = $data['message'] ?? '';
        
        // Prepare additional data for the push notification
        $pushData = [
            'type' => $data['type'] ?? 'general',
            'related_id' => (string)($data['related_id'] ?? '0'),
            'route' => $data['route'] ?? NotificationTypeRegistry::getRoute($data['type'] ?? 'general')
        ];
        
        return $this->fcmService->sendBulkNotifications($recipients, $title, $body, $pushData);
    }
    
    /**
     * Sync notifications with Firebase Realtime Database
     * 
     * @param array $data Notification data
     * @param array $recipients Recipient user IDs
     * @param int $notificationId First notification ID
     * @return int Number of successful syncs
     */
    private function syncWithFirebase(array $data, array $recipients, int $notificationId): int
    {
        // Skip if Realtime DB is disabled or service is not available
        if (!$this->realtimeDbEnabled || !$this->firebaseRealtimeService) {
            return 0;
        }
        
        // Process notifications to include timestamp
        if (!isset($data['timestamp'])) {
            $data['timestamp'] = time() * 1000; // Milliseconds
        }
        
        $successCount = 0;
        
        // Sync for each recipient
        foreach ($recipients as $userId) {
            // Customize data for this recipient
            $notificationData = $data;
            $notificationData['for_id'] = $userId;
            $notificationData['id'] = $notificationId;
            
            // Save to Firebase Realtime Database
            $success = $this->firebaseRealtimeService->saveNotification(
                $userId,
                (string)$notificationId,
                $notificationData
            );
            
            if ($success) {
                $successCount++;
            }
        }
        
        return $successCount;
    }
    
    /**
     * Mark a single notification as read with realtime sync
     * 
     * @param int $userId User ID
     * @param int $notificationId Notification ID
     * @return bool Success status
     */
    public function markNotificationAsRead(int $userId, int $notificationId): bool
    {
        try {
            // First mark as read in standard system
            $result = $this->notificationService->markNotificationAsRead($userId, $notificationId);
            
            // If Realtime DB is enabled and service is available, sync read status
            if ($result && $this->realtimeDbEnabled && $this->firebaseRealtimeService) {
                $this->firebaseRealtimeService->syncReadStatus($userId, $notificationId);
            }
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Error marking notification as read with sync', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId,
                'notification_id' => $notificationId
            ]);
            return false;
        }
    }
    
    /**
     * Mark notifications as read with realtime sync
     * 
     * @param int $userId User ID
     * @return bool Success status
     */
    public function markNotificationsAsRead(int $userId): bool
    {
        try {
            // First mark as read in standard system
            $result = $this->notificationService->markNotificationsAsRead($userId);
            
            // If Realtime DB is enabled and service is available, sync read status
            if ($result && $this->realtimeDbEnabled && $this->firebaseRealtimeService) {
                // Get notifications to sync
                $notifications = $this->notificationService->getUserNotifications($userId, 100);
                
                // Sync read status to Firebase
                $this->firebaseRealtimeService->syncReadStatus($userId, null, $notifications);
            }
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Error marking notifications as read with sync', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId
            ]);
            return false;
        }
    }
    
    /**
     * Toggle sticky status with realtime sync
     * 
     * @param int $notificationId Notification ID
     * @return bool Success status
     */
    public function toggleStickyStatus(int $notificationId): bool
    {
        try {
            // First toggle in standard system
            $result = $this->notificationService->toggleStickyStatus($notificationId);
            
            // If Realtime DB is enabled and service is available, sync sticky status
            if ($result && $this->realtimeDbEnabled && $this->firebaseRealtimeService) {
                // Get the notification to check user ID and new sticky status
                $notification = $this->notificationService->getNotificationById($notificationId);
                
                if ($notification) {
                    $this->firebaseRealtimeService->updateNotification(
                        (int)$notification['for_id'],
                        (string)$notificationId,
                        [
                            'is_sticky' => (bool)$notification['is_sticky'],
                            'updated_at' => round(microtime(true) * 1000)
                        ]
                    );
                }
            }
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Error toggling sticky status with sync', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'notification_id' => $notificationId
            ]);
            return false;
        }
    }
}