<?php

namespace baseKRIZAN\Services;

use Models\NotificationModel;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use DateTime;
use DateTimeZone;
use Models\Entity\Korisnik;

/**
 * NotificationService - Core service for standard notification management
 * 
 * This service manages core notification functionality without
 * dependencies on external services.
 */
class NotificationService
{
    private Logger $logger;
    private NotificationModel $notificationModel;
    private DatabaseTable $korisniciTable;
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger,
        NotificationModel $notificationModel,
        DatabaseTable $korisniciTable
    ) {
        $this->logger = $logger;
        $this->notificationModel = $notificationModel;
        $this->korisniciTable = $korisniciTable;
    }

    /**
     * Create notification in database
     */
    public function createNotification(array $data, array $recipients = []): ?int
    {
        try {
            $this->logger->notification('Creating notification', [
                'data' => array_diff_key($data, ['message' => true]), // Skip message content in logs
                'recipients_count' => count($recipients)
            ]);
            
            // Process notification data
            $notificationData = $this->prepareNotificationData($data);
            
            // Process recipients
            $validRecipients = $this->prepareRecipients($recipients);
            
            if (empty($validRecipients)) {
                $this->logger->notification('No valid recipients');
                return null;
            }
            
            // Create database notifications for each recipient
            $notificationIds = $this->createDatabaseNotifications($notificationData, $validRecipients);
            
            if (empty($notificationIds)) {
                $this->logger->notification('Failed to create database notifications');
                return null;
            }
            
            $this->logger->notification('Notifications created', [
                'first_id' => $notificationIds[0] ?? null,
                'recipients' => count($validRecipients)
            ]);
            
            // Return the first notification ID
            return $notificationIds[0] ?? null;
            
        } catch (\Exception $e) {
            $this->logger->error('Error creating notifications', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }
    
    /**
     * Prepare notification data with default values
     */
    private function prepareNotificationData(array $data): array
    {
        // Set default values if not provided
        $now = new DateTime('now', new DateTimeZone(\baseKRIZAN\Config\Config::get('defaulttimezone') ?: 'UTC'));
        $data['created_at'] = $data['created_at'] ?? $now->format('Y-m-d H:i:s');
        
        return $data;
    }
    
    /**
     * Prepare and validate recipients list
     */
    private function prepareRecipients(array $recipients): array
    {
        // If no recipients provided, use admins as default
        if (empty($recipients)) {
            $recipients = $this->getAdminUsers();
            $this->logger->notification('Using admin users as recipients', [
                'admins_count' => count($recipients)
            ]);
        }
        
        // Ensure all recipients are valid integers
        $validRecipients = [];
        foreach ($recipients as $userId) {
            $userId = (int)$userId;
            if ($userId > 0) {
                $validRecipients[] = $userId;
            }
        }
        
        if (empty($validRecipients)) {
            $this->logger->notification('No valid recipients after filtering');
        }
        
        return $validRecipients;
    }
    
    /**
     * Create notifications in the database
     */
    private function createDatabaseNotifications(array $data, array $recipients): array
    {
        $notificationIds = [];
        
        foreach ($recipients as $userId) {
            $notificationData = $data;
            $notificationData['for_id'] = $userId;
            
            $notification = $this->notificationModel->save($notificationData);
            
            if ($notification) {
                $notificationIds[] = (int)$notification->id;
            }
        }
        
        $this->logger->notification('Database notifications created', [
            'count' => count($notificationIds)
        ]);
        
        return $notificationIds;
    }
    
    /**
     * Get admin users
     */
    private function getAdminUsers(): array
    {
        try {
            // Get users with KORISNIK_MASTER permission
            $users = $this->korisniciTable->findAll();
            $adminUsers = [];
            
            foreach ($users as $korisnik) {
                if ($korisnik->hasPermission(Korisnik::KORISNIK_MASTER)) {
                    $adminUsers[] = $korisnik->id;
                }
            }
            
            return $adminUsers;
        } catch (\Exception $e) {
            $this->logger->error('Error getting admin users', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Get user notifications
     */
    public function getUserNotifications(int $userId, int $limit = 10, string $filter = 'default'): array
    {
        try {
            // Validate userId
            if ($userId <= 0) {
                $this->logger->notification('Invalid user ID when getting notifications', [
                    'user_id' => $userId
                ]);
                return [];
            }
            
            // Validate limit
            $limit = max(1, min($limit, 100));
            
            $this->logger->notification('Getting user notifications', [
                'user_id' => $userId,
                'limit' => $limit,
                'filter' => $filter
            ]);
            
            $notifications = $this->notificationModel->getUserNotifications($userId, $limit, $filter);
            
            $this->logger->notification('Retrieved notifications', [
                'user_id' => $userId,
                'count' => count($notifications),
                'filter' => $filter
            ]);
            
            return $notifications;
        } catch (\Exception $e) {
            $this->logger->error('Failed to get user notifications', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId,
                'filter' => $filter
            ]);
            return [];
        }
    }
    
    /**
     * Mark a single notification as read
     */
    public function markNotificationAsRead(int $userId, int $notificationId): bool
    {
        try {
            // Get the notification to check if it belongs to the user
            $notification = $this->notificationModel->getNotificationById($notificationId);
            
            if (!$notification || $notification['for_id'] != $userId) {
                return false;
            }
            
            // Check if it's already read
            if ($notification['is_read']) {
                return true; // Already read, consider it a success
            }
            
            // Update notification
            $result = $this->notificationModel->updateReadStatus($notificationId, true);
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Error marking notification as read', [
                'error' => $e->getMessage(),
                'notification_id' => $notificationId,
                'user_id' => $userId
            ]);
            return false;
        }
    }

    /**
     * Mark notifications as read
     */
    public function markNotificationsAsRead(int $userId): bool
    {
        try {
            // Validate userId
            if ($userId <= 0) {
                $this->logger->notification('Invalid user ID when marking notifications as read', [
                    'user_id' => $userId
                ]);
                return false;
            }
            
            $this->logger->notification('Marking notifications as read', [
                'user_id' => $userId
            ]);
            
            $result = $this->notificationModel->markNotificationsAsRead($userId);
            
            $this->logger->notification('Notifications marked as read', [
                'user_id' => $userId,
                'success' => $result
            ]);
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Failed to mark notifications as read', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId
            ]);
            return false;
        }
    }

    /**
     * Get notification by ID
     */
    public function getNotificationById(int $notificationId): ?array
    {
        try {
            if ($notificationId <= 0) {
                return null;
            }
            
            return $this->notificationModel->getNotificationById($notificationId);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get notification by ID', [
                'error' => $e->getMessage(),
                'notification_id' => $notificationId
            ]);
            return null;
        }
    }

    /**
     * Toggle sticky status for notification
     */
    public function toggleStickyStatus(int $notificationId): bool
    {
        try {
            if ($notificationId <= 0) {
                return false;
            }
            
            $this->logger->notification('Toggling notification sticky status', [
                'notification_id' => $notificationId
            ]);
            
            $notification = $this->notificationModel->getNotificationById($notificationId);
            
            if (!$notification) {
                $this->logger->notification('Notification not found', [
                    'notification_id' => $notificationId
                ]);
                return false;
            }
            
            $newStickyState = !($notification['is_sticky'] ?? false);
            
            $result = $this->notificationModel->updateStickyStatus($notificationId, $newStickyState);
            
            $this->logger->notification('Notification sticky status updated', [
                'notification_id' => $notificationId,
                'is_sticky' => $newStickyState,
                'success' => $result
            ]);
            
            return $result;
        } catch (\Exception $e) {
            $this->logger->error('Failed to toggle sticky status', [
                'error' => $e->getMessage(),
                'notification_id' => $notificationId
            ]);
            return false;
        }
    }
}