<?php

namespace baseKRIZAN\Services;

/**
 * NotificationTypeRegistry - Registry for notification types
 * 
 * This service allows registering notification types with their metadata,
 * including titles, routes, and formatter callbacks.
 */
class NotificationTypeRegistry
{
    /**
     * Registry of notification types
     */
    private static array $notificationTypes = [];
    
    /**
     * Registry of notification type icons
     */
    private static array $notificationTypeIcons = [];

    /**
     * Initialize notification icons with default values
     */
    public static function initDefaultIcons(): void
    {
        self::$notificationTypeIcons = [
            'general' => asset('notification/images/notification-icon.png'),
            'system' => asset('notification/images/system.png'),
            'message' => asset('notification/images/message.png'),
            'task' => asset('notification/images/task.png')
        ];
    }

    /**
     * Get notification types and icons as array for JavaScript
     */
    public static function getNotificationTypesData(): array
    {
        if (empty(self::$notificationTypeIcons)) {
            self::initDefaultIcons();
        }

        $types = [];
        
        foreach (self::$notificationTypes as $type => $data) {
            $types[$type] = [
                'title' => $data['title'] ?? 'Notification',
                'route' => $data['route'] ?? '',
                'icon' => (self::$notificationTypeIcons[$type] ?? self::$notificationTypeIcons['general'])
            ];
        }
        
        // Add types that may not have complete configuration but have icons
        foreach (self::$notificationTypeIcons as $type => $icon) {
            if (!isset($types[$type])) {
                $types[$type] = [
                    'title' => ucfirst(str_replace('_', ' ', $type)),
                    'route' => '',
                    'icon' => $icon
                ];
            }
        }
        
        return [
            'types' => $types,
            'defaultIcon' => self::$notificationTypeIcons['general']
        ];
    }
    
    /**
     * Get title for notification type
     */
    public static function getTitle(string $type): string
    {
        return self::$notificationTypes[$type]['title'] ?? 'Notification';
    }
    
    /**
     * Get route for notification type
     */
    public static function getRoute(string $type): string
    {
        return self::$notificationTypes[$type]['route'] ?? '';
    }

    /**
     * Get icon path for notification type
     */
    public static function getIconForType(string $type): string
    {
        if (empty(self::$notificationTypeIcons)) {
            self::initDefaultIcons();
        }
        return self::$notificationTypeIcons[$type] ?? self::$notificationTypeIcons['general'];
    }
    
    /**
     * Register a notification type with title and route
     */
    public static function register(string $type, string $title, string $route = ''): void
    {
        if (!isset(self::$notificationTypes[$type])) {
            self::$notificationTypes[$type] = [];
        }
        
        self::$notificationTypes[$type]['title'] = $title;
        self::$notificationTypes[$type]['route'] = $route;
    }

    /**
     * Register a notification message formatter for a type
     */
    public static function registerFormatter(string $type, callable $formatter): void
    {
        if (!isset(self::$notificationTypes[$type])) {
            self::$notificationTypes[$type] = [];
        }
        
        self::$notificationTypes[$type]['formatter'] = $formatter;
    }
}