<?php
// app/baseKRIZAN/Session/DatabaseSessionHandler.php

namespace baseKRIZAN\Session;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Session handler that stores session data in database
 * 
 * Implements PHP's session handler interfaces for database storage
 */
class DatabaseSessionHandler implements \SessionHandlerInterface, \SessionUpdateTimestampHandlerInterface, SessionHandlerInterface
{
    private DatabaseConnection $dbConnection;
    private Logger $logger;
    private string $table = 'korisnici_sessions';
    private int $lifetime;

    /**
     * Constructor for the DatabaseSessionHandler
     *
     * @param Logger $logger Logger instance for session-related logging
     * @param DatabaseConnection|null $dbConnection Optional DatabaseConnection instance (will be auto-created if null)
     * @param string|null $table Optional custom table name
     */
    public function __construct(Logger $logger, ?DatabaseConnection $dbConnection = null, ?string $table = null)
    {
        $this->logger = $logger;
        
        if ($dbConnection !== null) {
            $this->dbConnection = $dbConnection;
        } else {
            // Auto-create DatabaseConnection connection
            try {
                $this->dbConnection = DatabaseConnection::getInstance();
            } catch (\Throwable $e) {
                $this->logger->error('Failed to create database connection for session handler', [
                    'error' => $e->getMessage()
                ]);
                throw $e;
            }
        }
        
        // Set custom table name if provided
        if ($table !== null) {
            $this->table = $table;
        }
        
        // Get session lifetime from config or PHP settings
        $this->lifetime = (int)ini_get('session.gc_maxlifetime');
        
        $this->logger->session('DatabaseSessionHandler initialized', [
            'table' => $this->table,
            'lifetime' => $this->lifetime
        ]);
        
        // Ensure the session table exists
        $this->ensureTableExists();
    }

    /**
     * Ensure the session table exists
     */
    private function ensureTableExists(): void
    {
        try {
            // Check if table exists using a safe method
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $this->table]
            );
            
            if ($tableExists === null) {
                // Create the table if it doesn't exist
                $this->createSessionsTable();
                $this->logger->session('Sessions table created');
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error checking/creating sessions table', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Create the sessions table in the database
     */
    private function createSessionsTable(): void
    {
        // Using a prepared statement with placeholders isn't possible for table creation
        // but we sanitize the table name by validating it contains only valid chars
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $this->table)) {
            $this->logger->error('Invalid table name for sessions', [
                'table' => $this->table
            ]);
            throw new \InvalidArgumentException('Invalid table name for sessions');
        }
        
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->table}` (
            `id` VARCHAR(128) NOT NULL PRIMARY KEY,
            `user_id` VARCHAR(128) NULL DEFAULT NULL,
            `ip_address` VARCHAR(45) NULL DEFAULT NULL,
            `user_agent` TEXT NULL DEFAULT NULL,
            `payload` TEXT NOT NULL,
            `last_activity` INT UNSIGNED NOT NULL,
            INDEX `sessions_user_id_index` (`user_id`)
        )";
        
        $this->dbConnection->execute($sql);
    }

    /**
     * Open session
     *
     * @param string $savePath Session save path
     * @param string $sessionName Session name
     * @return bool Success status
     */
    public function open(string $savePath, string $sessionName): bool
    {
        return true; // Connection is already established in constructor
    }

    /**
     * Close session
     *
     * @return bool Success status
     */
    public function close(): bool
    {
        return true; // Nothing to do, DatabaseConnection manages connection
    }

    /**
     * Read session data
     *
     * @param string $id Session ID
     * @return string Session data or empty string
     */
    public function read(string $id): string
    {
        try {
            $result = $this->dbConnection->querySingleValue(
                "SELECT `payload` FROM `{$this->table}` WHERE `id` = :id AND `last_activity` > :time",
                [
                    'id' => $id,
                    'time' => time() - $this->lifetime
                ]
            );
            
            return $result !== null ? $result : '';
        } catch (\PDOException $e) {
            $this->logger->error('Session read error', [
                'error' => $e->getMessage(),
                'session_id' => substr($id, 0, 8) . '...'
            ]);
        }
        
        return '';
    }

    /**
     * Write session data
     *
     * @param string $id Session ID
     * @param string $data Session data
     * @return bool Success status
     */
    public function write(string $id, string $data): bool
    {
        try {
            // Get user ID from session data if available
            $userId = $_SESSION['user_id'] ?? null;
            
            // Get IP address and user agent
            $ipAddress = $_SERVER['REMOTE_ADDR'] ?? null;
            $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
            $time = time();
            
            // Use INSERT ... ON DUPLICATE KEY UPDATE for better performance
            $result = $this->dbConnection->execute(
                "INSERT INTO `{$this->table}` 
                (`id`, `user_id`, `ip_address`, `user_agent`, `payload`, `last_activity`) 
                VALUES 
                (:id, :user_id, :ip_address, :user_agent, :data, :time)
                ON DUPLICATE KEY UPDATE
                `payload` = VALUES(`payload`),
                `last_activity` = VALUES(`last_activity`),
                `user_id` = VALUES(`user_id`),
                `ip_address` = VALUES(`ip_address`),
                `user_agent` = VALUES(`user_agent`)",
                [
                    'id' => $id,
                    'user_id' => $userId,
                    'ip_address' => $ipAddress,
                    'user_agent' => $userAgent,
                    'data' => $data,
                    'time' => $time
                ]
            );
            
            return $result !== false;
        } catch (\PDOException $e) {
            $this->logger->error('Session write error', [
                'error' => $e->getMessage(),
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            
            return false;
        }
    }

    /**
     * Update timestamp - required for SessionUpdateTimestampHandlerInterface
     *
     * @param string $id Session ID
     * @param string $data Session data
     * @return bool Success status
     */
    public function updateTimestamp(string $id, string $data): bool
    {
        try {
            // Only update the timestamp, not the payload
            $this->dbConnection->execute(
                "UPDATE `{$this->table}` SET `last_activity` = :time WHERE `id` = :id",
                [
                    'time' => time(),
                    'id' => $id
                ]
            );
            
            // Even if no rows were updated, consider it successful
            // (session might be created in write() later)
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Session update timestamp error', [
                'error' => $e->getMessage(),
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            
            return false;
        }
    }

    /**
     * Validate session ID - required for SessionUpdateTimestampHandlerInterface
     * 
     * @param string $id Session ID
     * @return bool True if the session ID is valid
     */
    public function validateId(string $id): bool
    {
        // Stricter validation - session IDs should only contain alphanumeric characters
        if (!ctype_alnum($id)) {
            $this->logger->session('Invalid session ID format', [
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            return false;
        }
        
        return true;
    }

    /**
     * Destroy a session
     *
     * @param string $id Session ID
     * @return bool Success status
     */
    public function destroy(string $id): bool
    {
        try {
            $this->dbConnection->execute(
                "DELETE FROM `{$this->table}` WHERE `id` = :id",
                ['id' => $id]
            );
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Session destroy error', [
                'error' => $e->getMessage(),
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            
            return false;
        }
    }

    /**
     * Garbage collection for expired sessions
     *
     * @param int $maxlifetime Maximum session lifetime
     * @return int|false Number of deleted sessions or false on failure
     */
    public function gc(int $maxlifetime): int|false
    {
        try {
            $old = time() - $maxlifetime;
            $count = $this->dbConnection->execute(
                "DELETE FROM `{$this->table}` WHERE `last_activity` < :time",
                ['time' => $old]
            );
            
            if ($count > 0) {
                $this->logger->session('Session garbage collection', [
                    'deleted_sessions' => $count
                ]);
            }
            
            return $count;
        } catch (\PDOException $e) {
            $this->logger->error('Session garbage collection error', [
                'error' => $e->getMessage()
            ]);
            
            return false;
        }
    }

    /**
     * Clean expired sessions for a specific user
     *
     * @param string|int $userId User ID
     * @return bool Success status
     */
    public function cleanExpiredSessionsForUser(string|int $userId): bool
    {
        try {
            $count = $this->dbConnection->execute(
                "DELETE FROM `{$this->table}` 
                WHERE `user_id` = :user_id 
                AND `last_activity` < :time",
                [
                    'user_id' => $userId,
                    'time' => time() - $this->lifetime
                ]
            );
            
            if ($count > 0) {
                $this->logger->session('Cleaned expired sessions for user', [
                    'user_id' => $userId,
                    'deleted_sessions' => $count
                ]);
            }
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Error cleaning user sessions', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            
            return false;
        }
    }

    /**
     * Get all active sessions for a user
     *
     * @param string|int $userId User ID
     * @return array Array of active sessions
     */
    public function getUserSessions(string|int $userId): array
    {
        try {
            $stmt = $this->dbConnection->executeQuery(
                "SELECT `id`, `ip_address`, `user_agent`, `last_activity` 
                FROM `{$this->table}` 
                WHERE `user_id` = :user_id 
                AND `last_activity` > :time",
                [
                    'user_id' => $userId,
                    'time' => time() - $this->lifetime
                ]
            );
            
            return $this->dbConnection->fetchAllAssoc($stmt);
        } catch (\PDOException $e) {
            $this->logger->error('Error getting user sessions', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            
            return [];
        }
    }

    /**
     * Delete all sessions for a specific user (logout from all devices)
     *
     * @param string|int $userId User ID
     * @param string|null $exceptSessionId Optional session ID to keep
     * @return bool Success status
     */
    public function deleteUserSessions(string|int $userId, ?string $exceptSessionId = null): bool
    {
        try {
            if ($exceptSessionId !== null) {
                $count = $this->dbConnection->execute(
                    "DELETE FROM `{$this->table}` 
                    WHERE `user_id` = :user_id 
                    AND `id` != :except_id",
                    [
                        'user_id' => $userId,
                        'except_id' => $exceptSessionId
                    ]
                );
            } else {
                $count = $this->dbConnection->execute(
                    "DELETE FROM `{$this->table}` 
                    WHERE `user_id` = :user_id",
                    ['user_id' => $userId]
                );
            }
            
            $this->logger->session('Deleted user sessions', [
                'user_id' => $userId,
                'deleted_sessions' => $count,
                'except_session' => $exceptSessionId !== null ? 'yes' : 'no'
            ]);
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Error deleting user sessions', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            
            return false;
        }
    }
}