<?php
// app/baseKRIZAN/Session/EncryptedSessionHandler.php

namespace baseKRIZAN\Session;

use baseKRIZAN\Error\Logger;

/**
 * Custom session handler that provides encryption for stored session data
 * 
 * This implements the PHP SessionHandlerInterface to handle encryption/decryption
 * at the storage level rather than trying to encrypt/decrypt individual session values.
 */
class EncryptedSessionHandler implements \SessionHandlerInterface 
{
    private Logger $logger;
    private ?string $encryptionKey = null;
    private string $algorithm = 'aes-256-gcm'; // AES-256 in GCM mode (authenticated encryption)
    private string $keyVersion = 'v1'; // For key rotation support
    
    /**
     * @param string|null $appSecret Application secret key for deriving encryption keys
     * @param Logger $logger Logger instance
     */
    public function __construct(?string $appSecret, Logger $logger) 
    {
        $this->logger = $logger;
        
        if ($appSecret) {
            // Derive a consistent encryption key from the app secret with a unique salt
            // Use a unique salt that changes each month to support key rotation
            $salt = 'SESSION_SALT_' . date('Ym');
            
            $this->encryptionKey = hash_pbkdf2(
                'sha256', 
                $appSecret, 
                $salt, 
                10000, 
                32, 
                true
            );
            $this->logger->session('Session encryption initialized');
            $this->keyVersion = 'v1_' . date('Ym');
        } else {
            $this->logger->session('No APP_SECRET defined, session encryption disabled');
        }
    }
    
    /**
     * Open session data storage
     */
    public function open(string $savePath, string $sessionName): bool 
    {
        return true;
    }
    
    /**
     * Close session data storage
     */
    public function close(): bool 
    {
        return true;
    }
    
    /**
     * Read and decrypt session data
     */
    public function read(string $id): string 
    {
        // Get the raw session data from the default handler
        $data = $this->readFromStorage($id);
        
        // Session data empty? Return empty string as required by the interface
        if (empty($data)) {
            return '';
        }
        
        // If encryption is disabled, return data as-is
        if (!$this->encryptionKey) {
            return $data;
        }
        
        // Check if data is encrypted (has our marker)
        if (str_starts_with($data, 'ENC')) {
            try {
                return $this->decrypt($data);
            } catch (\Exception $e) {
                $this->logger->error('Failed to decrypt session data', [
                    'error' => $e->getMessage(),
                    'session_id' => $id
                ]);
                // Return empty data if decryption fails to prevent session corruption
                return '';
            }
        }
        
        // Not encrypted, return as-is
        return $data;
    }
    
    /**
     * Encrypt and write session data
     */
    public function write(string $id, string $data): bool 
    {
        // Don't bother with empty data
        if (empty($data)) {
            return $this->writeToStorage($id, '');
        }
        
        // Encrypt data if encryption is enabled
        if ($this->encryptionKey) {
            try {
                $data = $this->encrypt($data);
            } catch (\Exception $e) {
                $this->logger->error('Failed to encrypt session data', [
                    'error' => $e->getMessage(),
                    'session_id' => $id
                ]);
                // Continue with unencrypted data rather than losing the session
            }
        }
        
        // Write to storage using the default handler
        return $this->writeToStorage($id, $data);
    }
    
    /**
     * Destroy session data
     */
    public function destroy(string $id): bool 
    {
        return $this->destroyInStorage($id);
    }
    
    /**
     * Garbage collection
     */
    public function gc(int $maxlifetime): int|false 
    {
        // Use the built-in garbage collection
        return $this->gcInStorage($maxlifetime);
    }
    
    /**
     * Encrypt data
     * 
     * @param string $data Data to encrypt
     * @return string Encrypted data with version prefix for key rotation
     */
    private function encrypt(string $data): string 
    {
        if (!$this->encryptionKey) {
            return $data;
        }
        
        // Generate a random IV for GCM mode (12 bytes recommended for GCM)
        $iv = random_bytes(12);
        
        // Add additional authenticated data (AAD)
        $aad = session_id();
        
        // Encrypt the data using GCM mode (authenticated encryption)
        $ciphertext = openssl_encrypt(
            $data,
            $this->algorithm,
            $this->encryptionKey,
            OPENSSL_RAW_DATA,
            $iv,
            $tag,  // This will hold the authentication tag
            $aad
        );
        
        if ($ciphertext === false) {
            throw new \RuntimeException('Encryption failed: ' . openssl_error_string());
        }
        
        // Combine key version, IV, tag, and ciphertext for storage
        // Format: ENC:{key_version}:{base64_data}
        return "ENC:{$this->keyVersion}:" . base64_encode($iv . $tag . $ciphertext);
    }
    
    /**
     * Decrypt data with support for key rotation
     * 
     * @param string $encryptedData Encrypted data with version prefix
     * @return string Decrypted data
     */
    private function decrypt(string $encryptedData): string 
    {
        if (!$this->encryptionKey) {
            return $encryptedData;
        }
        
        // Format: ENC:{key_version}:{base64_data}
        $parts = explode(':', $encryptedData, 3);
        
        // Handle old format without versioning for backwards compatibility
        if (count($parts) === 3) {
            // New versioned format
            list(, $keyVersion, $encodedData) = $parts;
            
            // Check if this is from a different key version and if we need a different key
            if ($keyVersion !== $this->keyVersion) {
                // For a production system, you would derive the key based on the version
                // For now, just log that we're using current key for old data
                $this->logger->session('Decrypting session data with different key version', [
                    'stored_version' => $keyVersion,
                    'current_version' => $this->keyVersion
                ]);
                // In a real system, you would derive the appropriate key here
            }
        } elseif (count($parts) === 2 && $parts[0] === 'ENC') {
            // Old format
            $encodedData = $parts[1];
        } else {
            // Unknown format
            return $encryptedData;
        }
        
        // Decode the combined data
        $combined = base64_decode($encodedData);
        if ($combined === false) {
            throw new \RuntimeException('Failed to decode encrypted data');
        }
        
        // Extract IV (12 bytes), tag (16 bytes), and ciphertext
        $iv = substr($combined, 0, 12);
        $tag = substr($combined, 12, 16);
        $ciphertext = substr($combined, 28);
        
        // Add additional authenticated data (AAD)
        $aad = session_id();
        
        // Decrypt the data
        $decrypted = openssl_decrypt(
            $ciphertext,
            $this->algorithm,
            $this->encryptionKey,
            OPENSSL_RAW_DATA,
            $iv,
            $tag,
            $aad
        );
        
        if ($decrypted === false) {
            throw new \RuntimeException('Decryption failed: ' . openssl_error_string());
        }
        
        return $decrypted;
    }
    
    /**
     * Read from the actual storage
     */
    private function readFromStorage(string $id): string 
    {
        // Use default PHP session storage (files)
        $path = session_save_path();
        if (empty($path)) {
            $path = sys_get_temp_dir();
        }
        
        $file = "$path/sess_$id";
        if (file_exists($file)) {
            return (string)file_get_contents($file);
        }
        
        return '';
    }
    
    /**
     * Write to the actual storage
     */
    private function writeToStorage(string $id, string $data): bool 
    {
        // Use default PHP session storage (files)
        $path = session_save_path();
        if (empty($path)) {
            $path = sys_get_temp_dir();
        }
        
        $dir = dirname("$path/sess_$id");
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
                $this->logger->error("Session directory could not be created: $dir");
                return false;
            }
        }
        
        return file_put_contents("$path/sess_$id", $data, LOCK_EX) !== false;
    }
    
    /**
     * Destroy data in storage
     */
    private function destroyInStorage(string $id): bool 
    {
        // Use default PHP session storage (files)
        $path = session_save_path();
        if (empty($path)) {
            $path = sys_get_temp_dir();
        }
        
        $file = "$path/sess_$id";
        if (file_exists($file)) {
            return unlink($file);
        }
        
        return true;
    }
    
    /**
     * Garbage collection in storage
     */
    private function gcInStorage(int $maxlifetime): int|false 
    {
        $path = session_save_path();
        if (empty($path)) {
            $path = sys_get_temp_dir();
        }
        
        $count = 0;
        $pattern = "$path/sess_*";
        
        foreach (glob($pattern) as $file) {
            if (is_file($file) && filemtime($file) + $maxlifetime < time() && unlink($file)) {
                $count++;
            }
        }
        
        return $count;
    }
}