<?php
// app/baseKRIZAN/Session/FallbackSessionHandler.php

namespace baseKRIZAN\Session;

use baseKRIZAN\Error\Logger;

/**
 * Fallback session handler that uses PHP's default file-based storage
 * but adds extra logging and security features
 */
class FallbackSessionHandler implements \SessionHandlerInterface, SessionHandlerInterface
{
    private Logger $logger;
    private string $savePath;
    private int $lifetime;

    /**
     * Constructor for the FallbackSessionHandler
     *
     * @param Logger $logger Logger instance for session-related logging
     */
    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
        $this->savePath = ini_get('session.save_path');
        $this->lifetime = (int)ini_get('session.gc_maxlifetime');
        
        // If no save path is configured, use a default path
        if (empty($this->savePath)) {
            $this->savePath = sys_get_temp_dir();
        }
        
        $this->logger->session('FallbackSessionHandler initialized', [
            'save_path' => $this->savePath,
            'lifetime' => $this->lifetime
        ]);
    }

    /**
     * Open session
     *
     * @param string $savePath Session save path
     * @param string $sessionName Session name
     * @return bool Success status
     */
    public function open(string $savePath, string $sessionName): bool
    {
        if (!empty($savePath)) {
            $this->savePath = $savePath;
        }
        
        // Ensure the save path exists and is writable
        if (!is_dir($this->savePath)) {
            if (mkdir($this->savePath, 0777, true)) {
                $this->logger->session('Created session save directory', [
                    'path' => $this->savePath
                ]);
            } else {
                $this->logger->error('Failed to create session save directory', [
                    'path' => $this->savePath
                ]);
                return false;
            }
        }
        
        if (!is_writable($this->savePath)) {
            $this->logger->error('Session save path is not writable', [
                'path' => $this->savePath
            ]);
            return false;
        }
        
        return true;
    }

    /**
     * Close session
     *
     * @return bool Success status
     */
    public function close(): bool
    {
        return true; // Nothing to do for file-based sessions
    }

    /**
     * Read session data
     *
     * @param string $id Session ID
     * @return string Session data or empty string
     */
    public function read(string $id): string
    {
        // Validate session ID for security
        if (!ctype_alnum($id)) {
            $this->logger->session('Invalid session ID format in read', [
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            return '';
        }
        
        $file = $this->getSessionFile($id);
        if (!file_exists($file)) {
            return '';
        }
        
        // Check file age
        if (filemtime($file) + $this->lifetime < time()) {
            unlink($file);
            return '';
        }
        
        // Read and return data
        $data = file_get_contents($file);
        
        // Log if session file is corrupted
        if ($data === false) {
            $this->logger->error('Failed to read session file', [
                'session_id' => substr($id, 0, 8) . '...',
                'file' => $file
            ]);
            return '';
        }
        
        return $data;
    }

    /**
     * Write session data
     *
     * @param string $id Session ID
     * @param string $data Session data
     * @return bool Success status
     */
    public function write(string $id, string $data): bool
    {
        // Validate session ID for security
        if (!ctype_alnum($id)) {
            $this->logger->session('Invalid session ID format in write', [
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            return false;
        }
        
        $file = $this->getSessionFile($id);
        
        // Make sure the directory exists
        $dir = dirname($file);
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
                $this->logger->error('Failed to create session directory', [
                    'dir' => $dir
                ]);
                return false;
            }
        }
        
        // Write data with exclusive lock
        $result = file_put_contents($file, $data, LOCK_EX);
        
        if ($result === false) {
            $this->logger->error('Failed to write session file', [
                'session_id' => substr($id, 0, 8) . '...',
                'file' => $file
            ]);
            return false;
        }
        
        return true;
    }

    /**
    * Destroy a session
    *
    * @param string $id Session ID
    * @return bool Success status
    */
    public function destroy(string $id): bool
    {
        // Validate session ID for security
        if (!ctype_alnum($id)) {
            $this->logger->session('Invalid session ID format in destroy', [
                'session_id' => substr($id, 0, 8) . '...'
            ]);
            return false;
        }
        
        $file = $this->getSessionFile($id);
        
        if (file_exists($file)) {
            $result = unlink($file);
            
            if (!$result) {
                $this->logger->error('Failed to delete session file', [
                    'session_id' => substr($id, 0, 8) . '...',
                    'file' => $file
                ]);
                return false;
            }
        }
        
        return true;
    }

    /**
     * Garbage collection for expired sessions
     *
     * @param int $maxlifetime Maximum session lifetime
     * @return int|false Number of deleted sessions or false on failure
     */
    public function gc(int $maxlifetime): int|false
    {
        $pattern = $this->savePath . '/sess_*';
        $files = glob($pattern);
        
        if ($files === false) {
            $this->logger->error('Failed to list session files for garbage collection', [
                'pattern' => $pattern
            ]);
            return false;
        }
        
        $now = time();
        $count = 0;
        
        foreach ($files as $file) {
            if (is_file($file) && filemtime($file) + $maxlifetime < $now) {
                if (unlink($file)) {
                    $count++;
                }
            }
        }
        
        if ($count > 0) {
            $this->logger->session('Session garbage collection', [
                'deleted_files' => $count
            ]);
        }
        
        return $count;
    }

    /**
     * Get the session file path
     *
     * @param string $id Session ID
     * @return string Full path to session file
     */
    private function getSessionFile(string $id): string
    {
        return $this->savePath . '/sess_' . $id;
    }
    
    /**
     * Clean expired sessions for a specific user
     * 
     * Note: This method is a stub for the interface compatibility.
     * File-based sessions do not track user information.
     *
     * @param string|int $userId User ID
     * @return bool Always returns false as this operation is not supported
     */
    public function cleanExpiredSessionsForUser(string|int $userId): bool
    {
        $this->logger->session('User-based session operations not supported with file storage');
        return false;
    }

    /**
     * Get all active sessions for a user
     * 
     * Note: This method is a stub for the interface compatibility.
     * File-based sessions do not track user information.
     *
     * @param string|int $userId User ID
     * @return array Empty array as this operation is not supported
     */
    public function getUserSessions(string|int $userId): array
    {
        return [];
    }

    /**
     * Delete all sessions for a specific user
     * 
     * Note: This method is a stub for the interface compatibility.
     * File-based sessions do not track user information.
     *
     * @param string|int $userId User ID
     * @param string|null $exceptSessionId Optional session ID to keep
     * @return bool Always returns false as this operation is not supported
     */
    public function deleteUserSessions(string|int $userId, ?string $exceptSessionId = null): bool
    {
        $this->logger->session('User-based session operations not supported with file storage');
        return false;
    }
}