<?php
// app/baseKRIZAN/Session/SessionManager.php

namespace baseKRIZAN\Session;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\RequestClassifier;

/**
 * Manages session handling, security, and storage
 */
class SessionManager
{
    private Logger $logger;
    private ?RequestClassifier $requestClassifier;
    private ?SessionHandlerInterface $sessionHandler = null;
    private array $sessionConfig;
    private bool $isStarted = false;
    private string $sessionName = 'KRIZAN_SESSION';
    private int $sessionLifetime = 7200; // 2 hours default
    private int $regenerateTime = 1800; // 30 minutes default
    private string $sameSite = 'Lax';
    private bool $secure;
    private bool $httpOnly = true;
    private bool $lazyLoading = false;

    /**
     * Constructor for the SessionManager
     *
     * @param Logger $logger The logger instance for session-related logging
     * @param RequestClassifier|null $requestClassifier For detecting API/asset requests
     * @param array $config Custom session configuration
     * @param bool $lazyLoading Whether to use lazy loading for sessions (delayed start)
     */
    public function __construct(
        Logger $logger, 
        ?RequestClassifier $requestClassifier = null, 
        array $config = [],
        bool $lazyLoading = false
    ) {
        $this->logger = $logger;
        $this->requestClassifier = $requestClassifier;
        $this->lazyLoading = $lazyLoading;
        
        // Check if session is already active
        $this->isStarted = session_status() === PHP_SESSION_ACTIVE;
        
        // Log session status
        $logger->session('SessionManager initializing', [
            'session_active' => $this->isStarted ? 'yes' : 'no',
            'session_id' => $this->isStarted ? session_id() : 'none',
            'lazy_loading' => $this->lazyLoading ? 'yes' : 'no'
        ]);
        
        // Load configuration
        $this->loadConfiguration($config);
        
        // Only configure and initialize handler if session is not already active
        if (!$this->isStarted) {
            // Configure session
            $this->configureSession();
            
            // Initialize session handler
            $this->initSessionHandler();
        } else {
            // Session is already active, can't configure or set handler
            $logger->session('Session already active, skipping configuration and handler setup');
        }
        
        $logger->session('SessionManager initialized', [
            'session_name' => session_name(),
            'session_lifetime' => $this->sessionLifetime,
            'secure' => $this->secure ? 'yes' : 'no',
            'session_active' => $this->isStarted ? 'yes' : 'no'
        ]);
    }

    /**
     * Load session configuration from provided config or defaults
     *
     * @param array $config Custom session configuration
     */
    private function loadConfiguration(array $config = []): void
    {
        // Default configuration
        $defaults = [
            'name' => 'KRIZAN_SESSION',
            'lifetime' => 7200, // 2 hours
            'regenerate_time' => 1800, // 30 minutes
            'same_site' => 'Lax',
            'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off',
            'http_only' => true,
            'use_database' => true,
            'use_cookies' => true,
            'use_encryption' => false,
            'gc_maxlifetime' => 86400, // 24 hours
            'gc_probability' => 1,
            'gc_divisor' => 100
        ];

        // Merge with provided config
        $this->sessionConfig = array_merge($defaults, $config);

        // Set properties from config
        $this->sessionName = $this->sessionConfig['name'];
        $this->sessionLifetime = $this->sessionConfig['lifetime'];
        $this->regenerateTime = $this->sessionConfig['regenerate_time'];
        $this->sameSite = $this->sessionConfig['same_site'];
        $this->secure = $this->sessionConfig['secure'];
        $this->httpOnly = $this->sessionConfig['http_only'];
    }

    /**
     * Initialize the appropriate session handler
     */
    private function initSessionHandler(): void
    {
        // Only set handler if session is not active
        if (session_status() !== PHP_SESSION_NONE) {
            $this->logger->session('Cannot set session handler - session already active');
            return;
        }
        
        try {
            // Try to use database handler if configured
            if ($this->sessionConfig['use_database'] && class_exists('\\baseKRIZAN\\Session\\DatabaseSessionHandler')) {
                $this->sessionHandler = new DatabaseSessionHandler($this->logger);
                $this->logger->session('Using database session handler');
            } else {
                // Use fallback file-based handler
                $this->sessionHandler = new FallbackSessionHandler($this->logger);
                $this->logger->session('Using fallback session handler');
            }
            
            // Apply encryption wrapper if configured
            if ($this->sessionConfig['use_encryption'] && class_exists('\\baseKRIZAN\\Session\\EncryptedSessionHandler')) {
                $appSecret = $this->sessionConfig['app_secret'] ?? null;
                $encryptedHandler = new EncryptedSessionHandler($appSecret, $this->logger);
                
                // Set the encrypted handler as the session handler
                session_set_save_handler($encryptedHandler, true);
                $this->logger->session('Using encrypted session handler');
            } else {
                // Set the regular handler
                session_set_save_handler($this->sessionHandler, true);
            }
        } catch (\Throwable $e) {
            $this->logger->error('Failed to initialize session handler, falling back to default', [
                'error' => $e->getMessage()
            ]);
            
            // Fall back to default handler in case of error
            $this->sessionHandler = new FallbackSessionHandler($this->logger);
            session_set_save_handler($this->sessionHandler, true);
        }
    }

    /**
     * Configure session parameters
     */
    private function configureSession(): void
    {
        // Skip if session is already active
        if (session_status() === PHP_SESSION_ACTIVE) {
            $this->logger->session('Cannot configure session - session already active', [
                'current_session_name' => session_name()
            ]);
            return;
        }
        
        // Set session name
        session_name($this->sessionName);
        
        // Set session cookie parameters in one call
        session_set_cookie_params([
            'lifetime' => $this->sessionLifetime,
            'path' => '/',
            'domain' => '',  // Current domain
            'secure' => $this->secure,
            'httponly' => $this->httpOnly,
            'samesite' => $this->sameSite
        ]);
        
        // Set garbage collection parameters
        ini_set('session.gc_maxlifetime', $this->sessionConfig['gc_maxlifetime']);
        ini_set('session.gc_probability', $this->sessionConfig['gc_probability']);
        ini_set('session.gc_divisor', $this->sessionConfig['gc_divisor']);
        
        // Security settings
        ini_set('session.use_cookies', $this->sessionConfig['use_cookies'] ? '1' : '0');
        ini_set('session.use_only_cookies', '1');
        ini_set('session.use_strict_mode', '1');
        
        // Set cache limiter
        session_cache_limiter('nocache');
    }

    /**
     * Start a new session or resume an existing one
     *
     * @return bool True if session started successfully
     */
    public function start(): bool
    {
        if ($this->isStarted()) {
            $this->logger->session('Session already started');
            return true;
        }

        // Skip session for certain requests if RequestClassifier is available
        if ($this->lazyLoading && $this->requestClassifier) {
            // Pročitaj URL iz servera umjesto korištenja Request objekta
            $path = $_SERVER['REQUEST_URI'] ?? '';
            
            // Kreiraj privremeni Request objekt ako je metoda projektirana tako
            $tempRequest = new \baseKRIZAN\Http\Request();
            
            // Provjeri je li asset zahtjev
            if ($this->requestClassifier->isAssetRequest($tempRequest)) {
                $this->logger->session('Skipping session for asset request');
                return false;
            }
            
            // Provjeri je li API zahtjev koji ne zahtijeva autentikaciju
            if ($this->requestClassifier->isApiRequest($tempRequest) && 
                !$tempRequest->getAttribute('requires_auth', false)) {
                $this->logger->session('Skipping session for stateless API request');
                return false;
            }
        }

        try {
            $result = session_start();
            $this->isStarted = $result;
            
            if ($result) {
                $this->logger->session('Session started', [
                    'session_id' => substr(session_id(), 0, 8) . '...',
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ]);
                
                // Check if session ID regeneration is needed
                $this->checkRegenerateId();
                
                return true;
            } else {
                $this->logger->error('Failed to start session');
                return false;
            }
        } catch (\Throwable $e) {
            $this->logger->error('Session start error', [
                'message' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Check if session needs ID regeneration for security
     */
    private function checkRegenerateId(): void
    {
        // Check if we have a last_regenerated timestamp
        if (!isset($_SESSION['last_regenerated'])) {
            $_SESSION['last_regenerated'] = time();
            return;
        }

        // Check if enough time has passed to regenerate
        $regenerateTime = $_SESSION['last_regenerated'] + $this->regenerateTime;
        
        if (time() > $regenerateTime) {
            // Regenerate session ID
            if ($this->regenerateId()) {
                $_SESSION['last_regenerated'] = time();
                $this->logger->session('Session ID regenerated');
            }
        }
    }

    /**
     * Regenerate the session ID
     *
     * @param bool $deleteOldSession Whether to delete the old session
     * @return bool Success status
     */
    public function regenerateId(bool $deleteOldSession = true): bool
    {
        if (!$this->isStarted()) {
            $this->start();
        }

        if (!$this->isStarted()) {
            return false;
        }

        try {
            $result = session_regenerate_id($deleteOldSession);
            if ($result) {
                $this->logger->session('Session ID regenerated', [
                    'new_session_id' => substr(session_id(), 0, 8) . '...'
                ]);
            }
            return $result;
        } catch (\Throwable $e) {
            $this->logger->error('Session regenerate ID error', [
                'message' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Check if the session is started
     *
     * @return bool
     */
    public function isStarted(): bool
    {
        return $this->isStarted || session_status() === PHP_SESSION_ACTIVE;
    }

    /**
     * Ensure session integrity (starts session if not started)
     *
     * @return bool Success status
     */
    public function ensureSessionIntegrity(): bool
    {
        if (!$this->isStarted()) {
            return $this->start();
        }
        return true;
    }

    /**
     * Set a value in the session
     *
     * @param string $key Session key
     * @param mixed $value Session value
     * @return bool Success status
     */
    public function set(string $key, $value): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }

        $_SESSION[$key] = $value;
        return true;
    }

    /**
     * Get a value from the session
     *
     * @param string $key Session key
     * @param mixed $default Default value if key doesn't exist
     * @return mixed The session value or default
     */
    public function get(string $key, $default = null)
    {
        if (!$this->ensureSessionIntegrity()) {
            return $default;
        }

        return $_SESSION[$key] ?? $default;
    }

    /**
     * Check if a key exists in the session
     *
     * @param string $key Session key
     * @return bool True if key exists
     */
    public function has(string $key): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }

        return isset($_SESSION[$key]);
    }

    /**
     * Remove a value from the session
     *
     * @param string $key Session key
     * @return bool Success status
     */
    public function remove(string $key): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }

        if (isset($_SESSION[$key])) {
            unset($_SESSION[$key]);
            return true;
        }

        return false;
    }

    /**
     * Clear all values from the session
     *
     * @return bool Success status
     */
    public function clear(): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }

        $_SESSION = [];
        return true;
    }

    /**
     * Destroy the current session completely
     *
     * @return bool Success status
     */
    public function destroy(): bool
    {
        if (!$this->isStarted()) {
            return true;
        }

        // Clear session data
        $_SESSION = [];

        // Get session cookie parameters
        $params = session_get_cookie_params();
        
        // Delete the session cookie
        setcookie(
            session_name(),
            '',
            [
                'expires' => time() - 42000,
                'path' => $params['path'],
                'domain' => $params['domain'] ?? '',
                'secure' => $params['secure'],
                'httponly' => $params['httponly'],
                'samesite' => $params['samesite'] ?? $this->sameSite
            ]
        );

        $destroyed = session_destroy();
        if ($destroyed) {
            $this->isStarted = false;
            $this->logger->session('Session destroyed');
        }

        return $destroyed;
    }

    /**
     * Get the session handler
     *
     * @return SessionHandlerInterface|null
     */
    public function getSessionHandler(): ?SessionHandlerInterface
    {
        return $this->sessionHandler;
    }

    /**
     * Get the session ID
     *
     * @return string|null
     */
    public function getId(): ?string
    {
        if (!$this->isStarted()) {
            return null;
        }
        return session_id();
    }

    /**
     * Add a flash message that persists only for the next request
     *
     * @param string $type Message type (e.g., 'success', 'error')
     * @param string $message The message content
     * @return bool Success status
     */
    public function flash(string $type, string $message): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }

        // Initialize flash container if it doesn't exist
        if (!isset($_SESSION['_flash'])) {
            $_SESSION['_flash'] = [];
        }

        // Add message to flash container
        $_SESSION['_flash'][$type][] = $message;
        return true;
    }

    /**
     * Get flash messages and remove them
     *
     * @param string|null $type Optional type to filter by
     * @return array Flash messages
     */
    public function getFlash(?string $type = null): array
    {
        if (!$this->ensureSessionIntegrity() || !isset($_SESSION['_flash'])) {
            return [];
        }

        // Return specific type if requested
        if ($type !== null) {
            $messages = $_SESSION['_flash'][$type] ?? [];
            unset($_SESSION['_flash'][$type]);
            return $messages;
        }

        // Return all flash messages
        $messages = $_SESSION['_flash'];
        $_SESSION['_flash'] = [];
        return $messages;
    }

    /**
     * Check if there are any flash messages
     *
     * @param string|null $type Optional type to check
     * @return bool True if flash messages exist
     */
    public function hasFlash(?string $type = null): bool
    {
        if (!$this->ensureSessionIntegrity() || !isset($_SESSION['_flash'])) {
            return false;
        }

        if ($type !== null) {
            return isset($_SESSION['_flash'][$type]) && !empty($_SESSION['_flash'][$type]);
        }

        return !empty($_SESSION['_flash']);
    }
    
    /**
     * Update user information in session after login
     * 
     * @param array $userData User data to store
     * @return bool Success status
     */
    public function setUserData(array $userData): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }
        
        // Store user data
        $_SESSION['user'] = $userData;
        $_SESSION['user_id'] = $userData['id'] ?? null;
        
        // Regenerate session ID for security
        $this->regenerateId(true);
        
        return true;
    }
    
    /**
     * Clear user information on logout
     * 
     * @param bool $destroySession Whether to destroy the entire session
     * @return bool Success status
     */
    public function clearUserData(bool $destroySession = false): bool
    {
        if (!$this->ensureSessionIntegrity()) {
            return false;
        }
        
        // Clear user data
        unset($_SESSION['user'], $_SESSION['user_id']);
        
        // Regenerate session ID for security
        $this->regenerateId(true);
        
        if ($destroySession) {
            return $this->destroy();
        }
        
        return true;
    }
}