<?php

namespace baseKRIZAN\Shared;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\MetricsCollector;

/**
 * Simple WebSocket server for real-time communication
 */
class WebSocketServer
{
    private Logger $logger;
    private int $port;
    private $socket;
    private array $clients = [];
    private static array $messageQueue = [];
    private static string $socketPath;
    private static ?self $instance = null;
    private ?MetricsCollector $metricsCollector = null;

    /**
     * Constructor
     */
    public function __construct(Logger $logger, int $port)
    {
        $this->logger = $logger;
        $this->port = $port;
        
        // Koristimo stvarnu putanju aplikacije i dodajemo poddirektorij za pohranu
        $sharedPath = \baseKRIZAN\Config\Config::get('paths.logs') . '/shared';
        
        // Set static variables for inter-process communication
        self::$socketPath = $sharedPath . '/websocket.sock';
        self::$instance = $this;
        
        // Ensure directory exists
        $socketDir = dirname(self::$socketPath);
        if (!is_dir($socketDir)) {
            mkdir($socketDir, 0755, true);
        }
        
        // Try to get MetricsCollector from container if available
        $this->initializeMetricsCollector();
    }
    
    /**
     * Initialize metrics collector from container if available
     */
    private function initializeMetricsCollector(): void
    {
        // Check if Bootstrap class exists
        if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
            $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
            
            if ($container && $container->has('lukaMetricsCollector')) {
                $this->metricsCollector = $container->get('lukaMetricsCollector');
                $this->logger->services('WebSocket server connected to MetricsCollector');
            }
        }
    }

    /**
     * Start the WebSocket server
     */
    public function run(): void
    {
        // Create WebSocket
        $this->socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
        
        // Set socket options
        socket_set_option($this->socket, SOL_SOCKET, SO_REUSEADDR, 1);
        
        // Bind to port
        if (!socket_bind($this->socket, '0.0.0.0', $this->port)) {
            $this->logger->error('Failed to bind WebSocket to port', [
                'port' => $this->port,
                'error' => socket_strerror(socket_last_error())
            ]);
            
            // Record error in metrics if available
            if ($this->metricsCollector) {
                $this->metricsCollector->recordWebSocketError('Failed to bind WebSocket to port: ' . 
                    socket_strerror(socket_last_error()));
            }
            
            return;
        }
        
        // Start listening
        if (!socket_listen($this->socket)) {
            $this->logger->error('Failed to start WebSocket listening', [
                'error' => socket_strerror(socket_last_error())
            ]);
            
            // Record error in metrics if available
            if ($this->metricsCollector) {
                $this->metricsCollector->recordWebSocketError('Failed to start WebSocket listening: ' . 
                    socket_strerror(socket_last_error()));
            }
            
            return;
        }
        
        // Create IPC socket for message publishing
        $this->createIpcSocket();
        
        $this->logger->services('WebSocket server started on port ' . $this->port);
        
        // Main server loop
        while (true) {
            // Prepare socket lists
            $read = array_merge([$this->socket], $this->clients);
            $write = $except = null;
            
            // Wait for activity on any socket
            if (socket_select($read, $write, $except, 0, 200000) < 1) {
                // No activity, check for messages in queue
                $this->processMessageQueue();
                continue;
            }
            
            // Handle new connections
            if (in_array($this->socket, $read)) {
                $this->acceptNewConnection();
                unset($read[array_search($this->socket, $read)]);
            }
            
            // Handle client messages
            foreach ($read as $client) {
                $this->handleClientMessage($client);
            }
            
            // Process any pending messages
            $this->processMessageQueue();
        }
    }

    /**
     * Create IPC socket for inter-process communication
     */
    private function createIpcSocket(): void
    {
        // Remove old socket file if exists
        if (file_exists(self::$socketPath)) {
            unlink(self::$socketPath);
        }
        
        // Create Unix domain socket for IPC on Linux
        if (PHP_OS_FAMILY === 'Linux') {
            $ipcSocket = socket_create(AF_UNIX, SOCK_DGRAM, 0);
            
            if (!$ipcSocket) {
                $this->logger->error('Failed to create IPC socket', [
                    'error' => socket_strerror(socket_last_error())
                ]);
                return;
            }
            
            if (!socket_bind($ipcSocket, self::$socketPath)) {
                $this->logger->error('Failed to bind IPC socket', [
                    'error' => socket_strerror(socket_last_error())
                ]);
                return;
            }
            
            chmod(self::$socketPath, 0777); // Make writable for all users
            
            // Add to client list to monitor
            $this->clients[] = $ipcSocket;
            
            $this->logger->services('IPC socket created for WebSocket messaging', [
                'path' => self::$socketPath
            ]);
        }
    }

    /**
     * Accept a new WebSocket connection
     */
    private function acceptNewConnection(): void
    {
        $client = socket_accept($this->socket);
        
        if ($client === false) {
            $this->logger->error('Failed to accept WebSocket connection', [
                'error' => socket_strerror(socket_last_error($this->socket))
            ]);
            
            // Record error in metrics if available
            if ($this->metricsCollector) {
                $this->metricsCollector->recordWebSocketError('Failed to accept WebSocket connection: ' . 
                    socket_strerror(socket_last_error($this->socket)));
            }
            
            return;
        }
        
        // Perform WebSocket handshake
        $this->performHandshake($client);
        
        // Add to clients array
        $this->clients[] = $client;
        
        $this->logger->services('New WebSocket client connected', [
            'total_clients' => count($this->clients) - 1 // Subtract IPC socket
        ]);
    }

    /**
     * Perform WebSocket protocol handshake
     */
    private function performHandshake($client): void
    {
        $headers = [];
        $buffer = '';
        
        // Read headers
        while (($line = socket_read($client, 2048, PHP_NORMAL_READ)) !== false) {
            $buffer .= $line;
            
            if ($line === "\r\n" || $line === "\n") {
                break;
            }
        }
        
        // Parse headers
        $lines = explode("\n", $buffer);
        foreach ($lines as $line) {
            $line = chop($line);
            if (preg_match('/^([^:]+): (.+)$/', $line, $matches)) {
                $headers[$matches[1]] = $matches[2];
            }
        }
        
        // Generate WebSocket response
        $secKey = $headers['Sec-WebSocket-Key'] ?? '';
        $secAccept = base64_encode(pack('H*', sha1($secKey . '258EAFA5-E914-47DA-95CA-C5AB0DC85B11')));
        
        $response = "HTTP/1.1 101 Switching Protocols\r\n";
        $response .= "Upgrade: websocket\r\n";
        $response .= "Connection: Upgrade\r\n";
        $response .= "Sec-WebSocket-Accept: $secAccept\r\n\r\n";
        
        socket_write($client, $response, strlen($response));
    }

    /**
     * Handle incoming client message
     */
    private function handleClientMessage($client): void
    {
        $data = socket_read($client, 2048);
        
        if ($data === false || $data === '') {
            // Client disconnected
            $this->closeConnection($client);
            return;
        }
        
        // Check if this is from IPC socket (for Linux)
        if (PHP_OS_FAMILY === 'Linux' && file_exists(self::$socketPath)) {
            $this->handleIpcMessage($data);
            return;
        }
        
        // Decode WebSocket frame
        $message = $this->decodeWebSocketFrame($data);
        
        if ($message === false) {
            return; // Not a valid frame
        }
        
        // Handle ping/pong for keepalive
        if (ord($message[0]) === 0x89) { // Ping frame
            $this->sendPong($client);
            return;
        }
        
        // Handle close frame
        if (ord($message[0]) === 0x88) { // Close frame
            $this->closeConnection($client);
            return;
        }
        
        // Process text/binary message
        $this->logger->services('Received WebSocket message from client');
        
        // Record metrics
        if ($this->metricsCollector) {
            $this->metricsCollector->recordWebSocketMessageReceived($message, strlen($message));
        }
        
        // Echo message back (for testing)
        $this->sendToClient($client, $message);
    }

    /**
     * Handle IPC socket message
     */
    private function handleIpcMessage(string $data): void
    {
        // Add to message queue
        self::$messageQueue[] = $data;
    }

    /**
     * Process any messages in the queue
     */
    private function processMessageQueue(): void
    {
        if (empty(self::$messageQueue)) {
            return;
        }
        
        foreach (self::$messageQueue as $key => $message) {
            $this->sendToAllClients($message);
            unset(self::$messageQueue[$key]);
        }
    }

    /**
     * Send message to all connected clients
     */
    private function sendToAllClients(string $message): void
    {
        foreach ($this->clients as $client) {
            // Skip IPC socket
            if (PHP_OS_FAMILY === 'Linux' && file_exists(self::$socketPath)) {
                $socketInfo = socket_getpeername($client, $addr);
                if ($socketInfo === false) {
                    continue; // Skip IPC socket
                }
            }
            
            $this->sendToClient($client, $message);
        }
        
        // Record metrics for broadcast message
        if ($this->metricsCollector) {
            $totalClients = count($this->clients) - 1; // Subtract IPC socket
            $totalSize = strlen($message) * $totalClients;
            
            $this->metricsCollector->recordWebSocketMessageSent($message, $totalSize);
        }
    }

    /**
     * Send message to a specific client
     */
    private function sendToClient($client, string $message): void
    {
        $encodedMessage = $this->encodeWebSocketFrame($message);
        $result = socket_write($client, $encodedMessage, strlen($encodedMessage));
        
        // Record metrics
        if ($this->metricsCollector) {
            if ($result === false) {
                // Record error
                $this->metricsCollector->recordWebSocketError('Failed to send message to client: ' . 
                    socket_strerror(socket_last_error($client)));
            } else {
                // Record successful send
                $this->metricsCollector->recordWebSocketMessageSent($message, strlen($message));
            }
        }
    }

    /**
     * Close a client connection
     */
    private function closeConnection($client): void
    {
        $index = array_search($client, $this->clients);
        
        if ($index !== false) {
            unset($this->clients[$index]);
            socket_close($client);
            
            $this->logger->services('WebSocket client disconnected', [
                'total_clients' => count($this->clients) - 1 // Subtract IPC socket
            ]);
        }
    }

    /**
     * Get count of connected clients
     */
    public function getConnectedClientsCount(): int
    {
        // Exclude IPC socket from count
        $countIpcSocket = (PHP_OS_FAMILY === 'Linux' && file_exists(self::$socketPath)) ? 1 : 0;
        return count($this->clients) - $countIpcSocket;
    }

    /**
     * Send pong frame to client
     */
    private function sendPong($client): void
    {
        $pongFrame = "\x8A\x00"; // Pong frame with empty payload
        socket_write($client, $pongFrame, strlen($pongFrame));
    }

    /**
     * Encode data for WebSocket transmission
     */
    private function encodeWebSocketFrame(string $message): string
    {
        $length = strlen($message);
        $header = "";
        
        // Set FIN bit and text frame opcode
        $header .= chr(0x81);
        
        // Set masking bit and payload length
        if ($length <= 125) {
            $header .= chr($length);
        } elseif ($length <= 65535) {
            $header .= chr(126) . pack('n', $length);
        } else {
            $header .= chr(127) . pack('J', $length);
        }
        
        return $header . $message;
    }

    /**
     * Decode received WebSocket frame
     */
    private function decodeWebSocketFrame(string $data): string|false
    {
        if (strlen($data) < 2) {
            return false;
        }
        
        // Check if this is a valid WebSocket frame
        $firstByte = ord($data[0]);
        $secondByte = ord($data[1]);
        
        // Check for close, ping, or pong frames
        $opcode = $firstByte & 0x0F;
        if ($opcode === 0x08 || $opcode === 0x09 || $opcode === 0x0A) {
            return $data; // Return unprocessed frame for close, ping, pong
        }
        
        $isMasked = ($secondByte & 0x80) === 0x80;
        $payloadLength = $secondByte & 0x7F;
        
        $currentOffset = 2;
        
        // Get actual payload length
        if ($payloadLength === 126) {
            $payloadLength = unpack('n', substr($data, $currentOffset, 2))[1];
            $currentOffset += 2;
        } elseif ($payloadLength === 127) {
            $payloadLength = unpack('J', substr($data, $currentOffset, 8))[1];
            $currentOffset += 8;
        }
        
        // Get masking key and decode
        if ($isMasked) {
            $maskingKey = substr($data, $currentOffset, 4);
            $currentOffset += 4;
            
            $payload = substr($data, $currentOffset);
            $decodedPayload = '';
            
            for ($i = 0; $i < strlen($payload); $i++) {
                $decodedPayload .= $payload[$i] ^ $maskingKey[$i % 4];
            }
            
            return $decodedPayload;
        }
        
        // If not masked, return the raw payload
        return substr($data, $currentOffset);
    }

    /**
     * Static method to publish message to all clients
     * This is called from other processes to send messages to WebSocket clients
     */
    public static function publishToAll(string $message): bool
    {
        // Record metrics if instance is available
        if (self::$instance !== null && self::$instance->metricsCollector) {
            self::$instance->metricsCollector->recordWebSocketMessageSent($message, strlen($message));
        }
        
        // First try using the instance if available
        if (self::$instance !== null) {
            self::$messageQueue[] = $message;
            return true;
        }
        
        // If instance not available, try IPC on Linux
        if (PHP_OS_FAMILY === 'Linux' && file_exists(self::$socketPath)) {
            $ipcSocket = socket_create(AF_UNIX, SOCK_DGRAM, 0);
            
            if (!$ipcSocket) {
                return false;
            }
            
            $result = socket_sendto($ipcSocket, $message, strlen($message), 0, self::$socketPath);
            socket_close($ipcSocket);
            
            return $result !== false;
        }
        
        // No way to publish if neither instance nor IPC is available
        return false;
    }
    
    /**
     * Shutdown the server
     */
    public function shutdown(): void
    {
        // Close all client connections
        foreach ($this->clients as $client) {
            socket_close($client);
        }
        
        // Close main socket
        socket_close($this->socket);
        
        // Remove IPC socket file
        if (PHP_OS_FAMILY === 'Linux' && file_exists(self::$socketPath)) {
            unlink(self::$socketPath);
        }
        
        $this->logger->services('WebSocket server shutdown');
    }
}