<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Config\Config;
use baseKRIZAN\Assets\ResourceLoader;
use baseKRIZAN\Error\Logger;

/**
 * Provides default template directives.
 */
class DefaultDirectives
{
    private Config $config;
    private Logger $logger;
    
    /**
     * Constructor
     *
     * @param Config $config Configuration service
     * @param Logger $logger System logger
     */
    public function __construct(Config $config, Logger $logger)
    {
        $this->config = $config;
        $this->logger = $logger;
    }
    
    /**
     * Register all default directives
     */
    public function registerAll(DirectiveProcessor $processor): void
    {
        // Log the registration process for debugging
        if (method_exists($this->config, 'get') && $this->config->get('environment') === 'development') {
            $this->logger->template('Registering directives in DefaultDirectives::registerAll');
        }
        
        // Register each directive
        $processor->registerDirective('csrf', [$this, 'csrfDirective']);
        $processor->registerDirective('date', [$this, 'dateDirective']);
        $processor->registerDirective('asset', [$this, 'assetDirective']);
        $processor->registerDirective('trans', [$this, 'transDirective']);
        $processor->registerDirective('errormodalreturn', [$this, 'errorModalDirective']);
        
        // Log registered directives
        if (method_exists($this->config, 'get') && $this->config->get('environment') === 'development') {
            $this->logger->template('Registered directives: ' . implode(', ', $processor->getRegisteredDirectives()));
        }
    }
    
    /**
     * CSRF token directive
     */
    public function csrfDirective(array $variables, array $args = []): string
    {
        $token = $variables['csrfToken'] ?? '';
        if (is_array($token)) {
            $token = $token['value'] ?? ''; 
        }
        
        // Debug log for development
        if ($this->config->get('environment') === 'development') {
            $this->logger->template('CSRF token directive executed with token: ' . substr((string)$token, 0, 8) . '...');
        }
        
        return '<input type="hidden" name="csrf_token" value="' . 
            htmlspecialchars((string)$token, ENT_QUOTES, 'UTF-8') . '">';
    }
    
    /**
     * Date directive
     */
    public function dateDirective(array $variables, array $args = []): string
    {
        $format = $args['format'] ?? 'd.m.Y';
        return date($format);
    }
    
    /**
     * Asset URL directive
     */
    public function assetDirective(array $variables, array $args = []): string
    {
        $path = $args['path'] ?? '';
        $basePath = $this->config->get('paths.app_url');
        
        return rtrim($basePath, '/') . '/' . ltrim($path, '/');
    }
    
    /**
     * Translation directive
     */
    public function transDirective(array $variables, array $args = []): string
    {
        $key = $args['key'] ?? '';
        $default = $args['default'] ?? $key;
        
        // This is just a placeholder - in a real app, you'd use a translation service
        $translations = [
            'app.name' => $this->config->get('appname'),
            'app.welcome' => 'Dobrodošli u aplikaciju'
        ];
        
        return $translations[$key] ?? $default;
    }
    
    /**
     * Error modal directive
     */
    public function errorModalDirective(array $variables, array $args = []): string
    {
        // Debug log for development
        if ($this->config->get('environment') === 'development') {
            $this->logger->template('Error modal directive executed with errors: ' . json_encode($variables['errors'] ?? []));
        }
        
        // Koristite ValidationErrorRenderer za renderiranje
        return \baseKRIZAN\Validation\ValidationErrorRenderer::renderModal($variables['errors'] ?? [], $this->config);
    }
}