<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Error\Logger;

/**
 * Processes template directives and expressions.
 */
class DirectiveProcessor 
{
    private Logger $logger;
    private array $directives = [];

    /**
     * Constructor
     *
     * @param Logger $logger System logger
     */
    public function __construct(Logger $logger) 
    {
        $this->logger = $logger;
    }

    /**
     * Register a custom directive
     *
     * @param string $name Directive name
     * @param callable $processor Function to process the directive
     */
    public function registerDirective(string $name, callable $processor): void 
    {
        $this->directives[$name] = $processor;
    }

    /**
     * Process a template string, applying directives and processing expressions
     *
     * @param string $content Template content
     * @param array $variables Template variables
     * @return string Processed content
     */
    public function process(string $content, array $variables): string 
    {
        $start = microtime(true);
        
        // Process @directives
        foreach ($this->directives as $name => $processor) {
            // Critical fix: Ensure we're matching the exact directive pattern
            $pattern = '/@' . preg_quote($name, '/') . '(?:\(([^)]*)\))?/';
            
            $content = preg_replace_callback($pattern, function($matches) use ($processor, $variables, $name) {
                $args = isset($matches[1]) ? $this->parseArguments($matches[1]) : [];
                
                try {
                    $result = $processor($variables, $args);
                    return $result;
                } catch (\Throwable $e) {
                    $this->logger->error('Error processing directive', [
                        'directive' => $name,
                        'error' => $e->getMessage()
                    ]);
                    return '<!-- Error in @' . $name . ' directive: ' . $e->getMessage() . ' -->';
                }
            }, $content);
        }

        // Process expressions in a single pass for better performance
        $content = preg_replace_callback(
            '/\{\{(.*?)\}\}|\{!!(.*?)!!\}/',
            function($matches) use ($variables) {
                // Check which expression type matched
                $isEscaped = isset($matches[1]) && $matches[1] !== '';
                $expression = $isEscaped ? trim($matches[1]) : trim($matches[2]);
                $value = $this->evaluateExpression($expression, $variables);
                
                // Handle the result based on expression type
                if (is_array($value)) {
                    return $isEscaped 
                        ? htmlspecialchars(json_encode($value), ENT_QUOTES, 'UTF-8')
                        : json_encode($value);
                }
                
                return $isEscaped 
                    ? htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8')
                    : (string)$value;
            },
            $content
        );
        
        $elapsed = microtime(true) - $start;
        
        // Log processing time if it takes more than 10ms
        if ($elapsed > 0.01) {
            $this->logger->template('Template processing completed', [
                'elapsed_ms' => round($elapsed * 1000, 2),
                'directives_count' => count($this->directives),
                'content_length' => strlen($content)
            ]);
        }

        return $content;
    }

    /**
     * Evaluate a PHP expression within the given variable context
     *
     * @param string $expression PHP expression to evaluate
     * @param array $variables Variables to use in evaluation
     * @return mixed Result of expression evaluation
     */
    private function evaluateExpression(string $expression, array $variables) 
    {
        try {
            // Try to extract the variable directly first (common case)
            if (str_starts_with($expression, '$') && !str_contains($expression, '.')) {
                $varName = substr($expression, 1);
                if (isset($variables[$varName])) {
                    return $variables[$varName];
                }
            }
            
            // For more complex expressions, use eval
            extract($variables);
            $result = eval("return $expression;");
            
            // Log successful evaluation for debugging
            $this->logger->template('Expression evaluated', [
                'expression' => $expression,
                'result_type' => is_object($result) ? get_class($result) : gettype($result)
            ]);
            
            return $result;
        } catch (\Throwable $e) {
            $this->logger->error('Error evaluating template expression', [
                'expression' => $expression,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return '';
        }
    }

    /**
     * Parse directive arguments from string format
     *
     * @param string $argsString Arguments string from directive
     * @return array Parsed arguments
     */
    private function parseArguments(string $argsString): array 
    {
        $args = [];
        $argsString = trim($argsString);
        
        if (empty($argsString)) {
            return $args;
        }
        
        // Handle key=value pairs with improved pattern
        preg_match_all('/([a-zA-Z0-9_]+)\s*=\s*(["\'])(.*?)\2/', $argsString, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $args[$match[1]] = $match[3];
        }
        
        return $args;
    }
    
    /**
     * Get all registered directives
     *
     * @return array List of directive names
     */
    public function getRegisteredDirectives(): array
    {
        return array_keys($this->directives);
    }
    
    /**
     * Check if a directive is registered
     *
     * @param string $name Directive name
     * @return bool
     */
    public function hasDirective(string $name): bool
    {
        return isset($this->directives[$name]);
    }
}