<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Cache\CacheInterface;
use baseKRIZAN\Cache\CacheManager;
use Psr\SimpleCache\CacheInterface as PsrCacheInterface;

/**
 * Handles caching of rendered templates.
 * Implements PSR-16 SimpleCache interface but uses the main CacheInterface internally.
 */
class TemplateCache implements PsrCacheInterface
{
    private string $cachePath;
    private array $memoryCache = [];
    private Logger $logger;
    private bool $enabled;
    private array $stats = [
        'hits' => 0,
        'misses' => 0,
        'writes' => 0
    ];
    
    // Maximum number of items to store in memory cache
    private int $memoryCacheLimit;
    
    // Main cache service
    private CacheInterface $cache;
    
    // Cache prefix to avoid collisions
    private string $prefix = 'template:';

    /**
     * Constructor
     *
     * @param string $cachePath Path to store cache files
     * @param bool $enabled Whether caching is enabled
     * @param Logger $logger System logger
     * @param int $memoryCacheLimit Maximum number of items in memory cache
     * @param CacheInterface|null $cache Optional cache service to use
     */
    public function __construct(
        string $cachePath, 
        bool $enabled, 
        Logger $logger, 
        int $memoryCacheLimit = 50,
        ?CacheInterface $cache = null
    ) {
        $this->cachePath = $cachePath;
        $this->enabled = $enabled;
        $this->logger = $logger;
        $this->memoryCacheLimit = $memoryCacheLimit;
    
        if ($enabled && !is_dir($cachePath)) {
            mkdir($cachePath, 0777, true);
        }
        
        // Koristi dani cache ili dohvati iz CacheManager
        if ($cache !== null) {
            $this->cache = $cache;
        } else {
            $this->cache = CacheManager::getInstance($logger)->driver();
        }
    }

    /**
     * Get cached template content if available
     *
     * @param string $key Cache key
     * @param mixed $default Default value if key not found
     * @return mixed Cached content or default value
     */
    public function get(string $key, mixed $default = null): mixed
    {
        if (!$this->enabled) {
            $this->stats['misses']++;
            return $default;
        }

        // Check memory cache first (fast)
        if (isset($this->memoryCache[$key])) {
            $this->stats['hits']++;
            return $this->memoryCache[$key]['content'];
        }

        // Get from main cache
        $prefixedKey = 'template:' . $key;
        $cachedData = $this->cache->get($prefixedKey);
        
        if ($cachedData !== null) {
            if ($this->isValidCache($cachedData)) {
                // Store in memory cache for faster future access
                $this->addToMemoryCache($key, $cachedData);
                $this->stats['hits']++;
                return $cachedData['content'];
            }
            
            // If invalid or expired, delete from cache
            $this->cache->delete($prefixedKey);
        }

        $this->stats['misses']++;
        return $default;
    }

    /**
     * Store content in the cache
     *
     * @param string $key Cache key
     * @param mixed $value Content to cache
     * @param null|int|\DateInterval $ttl Cache lifetime in seconds
     * @return bool True on success
     */
    public function set($key, $value, $ttl = null): bool
    {
        if (!$this->enabled) {
            return false;
        }

        // Extract content and metadata
        $templatePath = '';
        $mtime = time();
        
        if (is_array($value) && isset($value['content'], $value['template_path'])) {
            $templatePath = $value['template_path'];
            $content = $value['content'];
            $mtime = file_exists($templatePath) ? filemtime($templatePath) : time();
        } else {
            $content = $value;
        }

        // Calculate TTL in seconds
        $lifetime = $this->parseTtl($ttl, 3600); // Default 1 hour
        
        // Prepare cache data
        $cacheData = [
            'content' => $content,
            'expires' => time() + $lifetime,
            'template_path' => $templatePath,
            'template_mtime' => $mtime,
            'created_at' => time()
        ];
        
        // Add to memory cache first
        $this->addToMemoryCache($key, $cacheData);
        
        // Store in main cache
        $prefixedKey = 'template:' . $key;
        $result = $this->cache->set($prefixedKey, $cacheData, $lifetime);
        
        if ($result) {
            $this->stats['writes']++;
            $this->logger->template('Cached template', [
                'key' => $key,
                'path' => $templatePath,
                'lifetime' => $lifetime
            ]);
        }
        
        return $result;
    }

    /**
     * Generate a cache key for a template and variables
     *
     * @param string $templatePath Template file path
     * @param array $variables Template variables
     * @return string Cache key
     */
    public function generateKey(string $templatePath, array $variables): string 
    {
        $varsHash = md5(serialize($this->filterCacheableVariables($variables)));
        $templateHash = md5_file($templatePath);
        return "{$templateHash}_{$varsHash}";
    }

    /**
     * Delete an item from the cache
     *
     * @param string $key Cache key
     * @return bool True on success
     */
    public function delete($key): bool
    {
        // Remove from memory cache
        unset($this->memoryCache[$key]);
        
        // Remove from main cache
        $prefixedKey = 'template:' . $key;
        return $this->cache->delete($prefixedKey);
    }

    /**
     * Clear all template cache items
     *
     * @return bool True on success
     */
    public function clear(): bool
    {
        // Clear memory cache
        $this->memoryCache = [];
        
        // Unfortunately we can't selectively clear only template cache entries
        // from the main cache without knowing all the keys, so we'll need to
        // use a directory-based approach still for full clearing
        
        $cacheFiles = glob($this->cachePath . '/*.cache');
        if (is_array($cacheFiles)) {
            foreach ($cacheFiles as $file) {
                @unlink($file);
            }
        }
        
        $this->logger->template('Template cache cleared');
        $this->resetStats();
        
        return true;
    }
    
    /**
     * Filter variables to only include those that are cacheable
     *
     * @param array $variables Variables to filter
     * @return array Filtered variables
     */
    private function filterCacheableVariables(array $variables): array 
    {
        $filtered = [];
        
        foreach ($variables as $key => $value) {
            // Skip objects that can't be serialized
            if (is_object($value)) {
                if ($value instanceof \Serializable || $value instanceof \stdClass) {
                    $filtered[$key] = $value;
                }
                continue;
            }
            
            // Skip resources
            if (is_resource($value)) {
                continue;
            }
            
            // Recursively filter arrays
            if (is_array($value)) {
                $filtered[$key] = $this->filterCacheableVariables($value);
                continue;
            }
            
            // Include all other values
            $filtered[$key] = $value;
        }
        
        return $filtered;
    }
    
    /**
     * Get a collection of stats about the cache
     * 
     * @return array Cache stats
     */
    public function getStats(): array
    {
        $hits = $this->stats['hits'];
        $misses = $this->stats['misses'];
        $total = $hits + $misses;
        
        $this->stats['hit_ratio'] = $total > 0 
            ? round(($hits / $total) * 100, 2) . '%'
            : '0%';
        
        $this->stats['memory_items'] = count($this->memoryCache);
        
        $cacheFiles = glob($this->cachePath . '/*.cache');
        $this->stats['disk_items'] = is_array($cacheFiles) ? count($cacheFiles) : 0;
        
        return $this->stats;
    }
    
    /**
     * Reset cache stats
     */
    public function resetStats(): void
    {
        $this->stats = [
            'hits' => 0,
            'misses' => 0,
            'writes' => 0
        ];
    }
    
    /**
     * Run garbage collection on the cache
     * 
     * @param float $probability Probability of running GC (0-1)
     * @return bool True if GC was run
     */
    public function garbageCollection(float $probability = 0.01): bool
    {
        if (!$this->enabled || mt_rand(1, 100) > ($probability * 100)) {
            return false;
        }
        
        // Let the main cache handle garbage collection
        $removed = $this->cache->gc();
        
        if ($removed > 0) {
            $this->logger->template('Template cache garbage collection', [
                'removed_items' => $removed
            ]);
        }
        
        return true;
    }
    
    /**
     * Get multiple cache items
     * 
     * @param iterable $keys Cache keys
     * @param mixed $default Default value
     * @return iterable Items
     */
    public function getMultiple($keys, $default = null): iterable
    {
        $result = [];
        foreach ($keys as $key) {
            $result[$key] = $this->get($key, $default);
        }
        return $result;
    }
    
    /**
     * Set multiple cache items
     * 
     * @param iterable $values Key-value pairs
     * @param null|int|\DateInterval $ttl TTL
     * @return bool Success
     */
    public function setMultiple($values, $ttl = null): bool
    {
        $success = true;
        foreach ($values as $key => $value) {
            $success = $this->set($key, $value, $ttl) && $success;
        }
        return $success;
    }
    
    /**
     * Delete multiple cache items
     * 
     * @param iterable $keys Keys to delete
     * @return bool Success
     */
    public function deleteMultiple($keys): bool
    {
        $success = true;
        foreach ($keys as $key) {
            $success = $this->delete($key) && $success;
        }
        return $success;
    }
    
    /**
     * Check if a key exists
     * 
     * @param string $key Cache key
     * @return bool True if exists
     */
    public function has($key): bool
    {
        if (isset($this->memoryCache[$key])) {
            // Check if expired in memory
            if ($this->memoryCache[$key]['expires'] < time()) {
                unset($this->memoryCache[$key]);
                return false;
            }
            return true;
        }
        
        // Check in main cache
        $prefixedKey = 'template:' . $key;
        return $this->cache->has($prefixedKey);
    }
    
    /**
     * Convert TTL to seconds
     * 
     * @param mixed $ttl TTL value
     * @param int $default Default value
     * @return int TTL in seconds
     */
    private function parseTtl($ttl, int $default): int
    {
        if ($ttl === null) {
            return $default;
        }
        
        if (is_int($ttl)) {
            return max(0, $ttl);
        }
        
        if ($ttl instanceof \DateInterval) {
            $reference = new \DateTimeImmutable();
            $endTime = $reference->add($ttl);
            return max(0, $endTime->getTimestamp() - $reference->getTimestamp());
        }
        
        return $default;
    }
    
    /**
     * Add data to memory cache
     */
    private function addToMemoryCache(string $key, array $cacheData): void
    {
        // Maintain memory cache size limit
        if (count($this->memoryCache) >= $this->memoryCacheLimit) {
            // Remove the oldest item
            reset($this->memoryCache);
            unset($this->memoryCache[key($this->memoryCache)]);
        }
        
        $this->memoryCache[$key] = $cacheData;
    }
    
    /**
     * Check if cache data is valid and not expired
     */
    private function isValidCache(array $cacheData): bool
    {
        return isset($cacheData['expires'], $cacheData['template_mtime'], 
                   $cacheData['template_path'], $cacheData['content']) &&
               $cacheData['expires'] > time() && 
               (empty($cacheData['template_path']) || (
                   file_exists($cacheData['template_path']) && 
                   $cacheData['template_mtime'] === filemtime($cacheData['template_path'])
               ));
    }
}