<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Error\Logger;

/**
 * Provides component-based template rendering capabilities.
 */
class TemplateComponents
{
    private TemplateRenderer $renderer;
    private Logger $logger;
    private array $components = [];
    private array $slots = [];
    
    /**
     * Constructor
     * 
     * @param TemplateRenderer $renderer Template renderer
     * @param Logger $logger System logger
     */
    public function __construct(TemplateRenderer $renderer, Logger $logger)
    {
        $this->renderer = $renderer;
        $this->logger = $logger;
        
        // Register the component directives
        $this->registerComponentDirectives();
    }
    
    /**
     * Register component directives with the template processor
     */
    private function registerComponentDirectives(): void
    {
        $processor = $this->renderer->getProcessor();
        
        // @component directive
        $processor->registerDirective('component', function(array $variables, array $args = []) {
            $name = $args['name'] ?? null;
            if (!$name) {
                return '<!-- component name missing -->';
            }
            
            return $this->renderComponent($name, $variables);
        });
        
        // @slot directive
        $processor->registerDirective('slot', function(array $variables, array $args = []) {
            $name = $args['name'] ?? 'default';
            
            // If this slot has content, return it
            if (isset($this->slots[$name])) {
                $content = $this->slots[$name];
                // Remove after use to prevent same slot being rendered twice
                unset($this->slots[$name]);
                return $content;
            }
            
            return '';
        });
        
        // @include directive for partial templates
        $processor->registerDirective('include', function(array $variables, array $args = []) {
            $template = $args['template'] ?? null;
            if (!$template) {
                return '<!-- include template missing -->';
            }
            
            // Can pass variables to the included template
            $data = [];
            foreach ($args as $key => $value) {
                if ($key !== 'template') {
                    $data[$key] = $value;
                }
            }
            
            // Merge with existing variables
            $data = array_merge($variables, $data);
            
            return $this->renderer->render($template, $data);
        });
        
        // @each directive for rendering arrays
        $processor->registerDirective('each', function(array $variables, array $args = []) {
            $items = $args['items'] ?? null;
            $template = $args['template'] ?? null;
            $varName = $args['as'] ?? 'item';
            
            if (!$items || !$template || !isset($variables[$items])) {
                return '<!-- each directive: missing or invalid parameters -->';
            }
            
            $itemsArray = $variables[$items];
            if (!is_array($itemsArray)) {
                return '<!-- each directive: items must be an array -->';
            }
            
            $output = '';
            $itemKeys = array_keys($itemsArray);
            $firstKey = reset($itemKeys);
            $lastKey = end($itemKeys);
            
            foreach ($itemsArray as $key => $value) {
                $itemVars = array_merge($variables, [
                    $varName => $value,
                    $varName . '_key' => $key,
                    $varName . '_index' => $key,
                    $varName . '_first' => $key === $firstKey,
                    $varName . '_last' => $key === $lastKey
                ]);
                
                $output .= $this->renderer->render($template, $itemVars);
            }
            
            return $output;
        });
    }
    
    /**
     * Register a new component
     * 
     * @param string $name Component name
     * @param string $template Component template path
     * @return self
     */
    public function registerComponent(string $name, string $template): self
    {
        $this->components[$name] = $template;
        $this->logger->template('Component registered', [
            'name' => $name,
            'template' => $template
        ]);
        return $this;
    }
    
    /**
     * Set a slot content
     * 
     * @param string $name Slot name
     * @param string $content Slot content
     * @return self
     */
    public function setSlot(string $name, string $content): self
    {
        $this->slots[$name] = $content;
        return $this;
    }
    
    /**
     * Render a component
     * 
     * @param string $name Component name
     * @param array $variables Variables to pass to the component
     * @return string Rendered component
     */
    public function renderComponent(string $name, array $variables = []): string
    {
        if (!isset($this->components[$name])) {
            $this->logger->template('Component not found', [
                'name' => $name
            ]);
            return '<!-- Component not found: ' . htmlspecialchars($name) . ' -->';
        }
        
        $template = $this->components[$name];
        return $this->renderer->render($template, $variables);
    }
    
    /**
     * Get all registered components
     * 
     * @return array Registered components
     */
    public function getComponents(): array
    {
        return $this->components;
    }
    
    /**
     * Clear all slots
     * 
     * @return self
     */
    public function clearSlots(): self
    {
        $this->slots = [];
        return $this;
    }
}