<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Error\Logger;

/**
 * Template loader class for finding and loading template files.
 */
class TemplateLoader 
{
    private array $templatePaths;
    private Logger $logger;
    private array $templateAliases = [];
    private string $defaultExtension = '.html.php';

    /**
     * Constructor
     *
     * @param array $templatePaths Array of paths to search for templates
     * @param Logger $logger System logger
     */
    public function __construct(array $templatePaths, Logger $logger) 
    {
        $this->templatePaths = $templatePaths;
        $this->logger = $logger;
    }

    /**
     * Find a template file by searching through the template paths
     *
     * @param string $templateName Template name or relative path
     * @return string|null Full path to the template file or null if not found
     */
    public function findTemplate(string $templateName): ?string 
    {
        // Check if this is an alias
        if (isset($this->templateAliases[$templateName])) {
            $templateName = $this->templateAliases[$templateName];
        }
        
        // Add default extension if not provided
        if (!preg_match('/\.[a-z0-9]+$/i', $templateName)) {
            $templateName .= $this->defaultExtension;
        }
        
        // First check if templateName is already an absolute path
        if (file_exists($templateName) && is_file($templateName)) {
            $this->logger->template('Template found (absolute path)', [
                'template' => $templateName
            ]);
            return $templateName;
        }
        
        // Search in all template paths
        foreach ($this->templatePaths as $basePath) {
            $path = rtrim($basePath, '/') . '/' . ltrim($templateName, '/');
            if (file_exists($path) && is_file($path)) {
                $this->logger->template('Template found', [
                    'template' => $templateName,
                    'full_path' => $path
                ]);
                return $path;
            }
        }
        
        $this->logger->template('Template not found', [
            'template' => $templateName,
            'paths_searched' => $this->templatePaths
        ], 'warning');
        
        return null;
    }

    /**
     * Load and render a template file with the given variables
     *
     * @param string $templatePath Full path to the template file
     * @param array $variables Variables to extract into the template scope
     * @return string Raw rendered template content
     */
    public function loadTemplate(string $templatePath, array $variables): string 
    {
        if (!file_exists($templatePath) || !is_file($templatePath)) {
            $this->logger->error('Template file does not exist', [
                'path' => $templatePath
            ]);
            return '';
        }
        
        try {
            // Start time to measure template loading performance
            $startTime = microtime(true);
            
            // Extract variables into local scope
            extract($variables);
            
            // Capture output buffer
            ob_start();
            include $templatePath;
            $content = ob_get_clean();
            
            if ($content === false) {
                throw new \RuntimeException("Failed to capture template output");
            }
            
            $loadTime = microtime(true) - $startTime;
            if ($loadTime > 0.01) { // Log only if more than 10ms
                $this->logger->template('Template loaded', [
                    'path' => $templatePath,
                    'time_ms' => round($loadTime * 1000, 2),
                    'size_kb' => round(strlen($content) / 1024, 2)
                ]);
            }
            
            return $content;
        } catch (\Throwable $e) {
            $this->logger->error('Error loading template', [
                'path' => $templatePath,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return '';
        }
    }

    /**
     * Get configured template paths
     *
     * @return array Array of template paths
     */
    public function getTemplatePaths(): array 
    {
        return $this->templatePaths;
    }

    /**
     * Add a new template path
     * 
     * @param string $path Path to add
     * @param bool $prepend Whether to add at the beginning of the search paths
     * @return self
     */
    public function addTemplatePath(string $path, bool $prepend = false): self 
    {
        // Ensure path exists
        if (!is_dir($path)) {
            $this->logger->template('Template path does not exist', [
                'path' => $path
            ]);
            return $this;
        }
        
        // Normalize path
        $path = rtrim($path, '/') . '/';
        
        // Check if already in list
        if (in_array($path, $this->templatePaths)) {
            return $this;
        }
        
        if ($prepend) {
            array_unshift($this->templatePaths, $path);
        } else {
            $this->templatePaths[] = $path;
        }
        
        $this->logger->template('Template path added', [
            'path' => $path,
            'prepend' => $prepend
        ]);
        
        return $this;
    }

    /**
     * Set the default template extension
     * 
     * @param string $extension Extension (with or without dot)
     * @return self
     */
    public function setDefaultExtension(string $extension): self
    {
        if ($extension[0] !== '.') {
            $extension = '.' . $extension;
        }
        
        $this->defaultExtension = $extension;
        return $this;
    }

    /**
     * Register a template alias
     * 
     * @param string $alias Alias name
     * @param string $template Actual template path
     * @return self
     */
    public function registerAlias(string $alias, string $template): self
    {
        $this->templateAliases[$alias] = $template;
        return $this;
    }

    /**
     * Get all registered template aliases
     * 
     * @return array Template aliases
     */
    public function getAliases(): array
    {
        return $this->templateAliases;
    }

    /**
     * Check if a template exists
     * 
     * @param string $templateName Template name
     * @return bool True if template exists
     */
    public function templateExists(string $templateName): bool
    {
        return $this->findTemplate($templateName) !== null;
    }
}