<?php

namespace baseKRIZAN\Template;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Config\Config;

/**
 * Core template rendering engine
 */
class TemplateRenderer 
{
    private TemplateLoader $loader;
    private DirectiveProcessor $processor;
    private TemplateCache $cache;
    private Logger $logger;
    private bool $cacheEnabled;
    private SessionManager $sessionManager;
    private Config $config;

    /**
     * Constructor
     * 
     * @param Logger $logger System logger
     * @param SessionManager $sessionManager Session manager
     * @param TemplateLoader $loader Template loader
     * @param DirectiveProcessor $processor Directive processor
     * @param TemplateCache $cache Template cache
     * @param Config $config Configuration
     */
    public function __construct(
        Logger $logger,
        SessionManager $sessionManager,
        TemplateLoader $loader,
        DirectiveProcessor $processor,
        TemplateCache $cache,
        Config $config
    ) {
        $this->logger = $logger;
        $this->sessionManager = $sessionManager;
        $this->loader = $loader;
        $this->processor = $processor;
        $this->cache = $cache;
        $this->config = $config;
        $this->cacheEnabled = $config->get('template.cache_enabled', $config->get('environment') !== 'development');
        
        $this->logger->template('Template renderer initialized', [
            'cache_enabled' => $this->cacheEnabled
        ]);
    }

    /**
     * Render a template with the provided variables
     *
     * @param string $templateFileName Template file name (including path relative to template roots)
     * @param array $variables Variables to pass to the template
     * @return string The rendered template
     */
    public function render(string $templateFileName, array $variables = []): string 
    {
        try {
            // Log start of rendering
            $this->logger->template('Begin template rendering', [
                'template' => $templateFileName,
                'variables_count' => count($variables)
            ]);
            
            // Find the template path
            $templatePath = $this->loader->findTemplate($templateFileName);
            if (!$templatePath) {
                $this->logger->template('Template not found', [
                    'template' => $templateFileName,
                    'searched_paths' => $this->loader->getTemplatePaths()
                ], 'warning');
                return 'Template not found: ' . htmlspecialchars($templateFileName);
            }

            // Process standard variables (like CSRF token)
            $variables = $this->processCommonVariables($variables);

            // Determine if we should use cache
            $useCache = $this->shouldUseCache();
            
            // Check if template is in cache
            if ($useCache) {
                $cacheKey = $this->cache->generateKey($templatePath, $variables);
                $cachedContent = $this->cache->get($cacheKey);
                
                if ($cachedContent !== null) {
                    $this->logger->template('Using cached template', [
                        'template' => $templateFileName,
                        'cache_key' => $cacheKey
                    ]);
                    return $cachedContent;
                }
            }

            // Load the raw template content
            $this->logger->template('Loading template from disk', [
                'template_path' => $templatePath
            ]);
            $rawContent = $this->loader->loadTemplate($templatePath, $variables);
            
            // Process directives and expressions
            $processedContent = $this->processor->process($rawContent, $variables);
            
            // Cache the result if enabled
            if ($useCache) {
                $this->cache->set($cacheKey, [
                    'content' => $processedContent,
                    'template_path' => $templatePath
                ]);
                
                $this->logger->template('Template cached', [
                    'template' => $templateFileName,
                    'cache_key' => $cacheKey
                ]);
            }
            
            $this->logger->template('Template rendering complete', [
                'template' => $templateFileName,
                'content_length' => strlen($processedContent)
            ]);
            
            return $processedContent;
            
        } catch (\Throwable $e) {
            $this->logger->error('Template rendering error', [
                'template' => $templateFileName,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            if ($this->config->get('environment') === 'development') {
                return 'Error rendering template: ' . $e->getMessage() . 
                    '<pre>' . htmlspecialchars($e->getTraceAsString()) . '</pre>';
            }
            
            return 'Error rendering template. Please check the logs.';
        }
    }

    /**
     * Process common template variables
     *
     * @param array $variables Variables passed to the template
     * @return array Processed variables
     */
    private function processCommonVariables(array $variables): array 
    {
        // Add CSRF token if not provided
        if (!isset($variables['csrfToken']) && $this->sessionManager->has('csrf_token')) {
            $variables['csrfToken'] = $this->sessionManager->get('csrf_token');
        }
        
        // Add other common variables as needed
        if (!isset($variables['appName'])) {
            $variables['appName'] = $this->config->get('appname');
        }
        
        return $variables;
    }
    
    /**
     * Determine if caching should be used for current request
     */
    private function shouldUseCache(): bool
    {
        // Skip cache for AJAX requests
        $isAjaxRequest = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                          $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest';
        
        return $this->cacheEnabled && !$isAjaxRequest;
    }

    /**
     * Clear the template cache
     */
    public function clearCache(): void 
    {
        $this->cache->clear();
    }

    /**
     * Register a custom directive
     *
     * @param string $name Directive name
     * @param callable $processor Directive processor function
     */
    public function registerDirective(string $name, callable $processor): void 
    {
        $this->processor->registerDirective($name, $processor);
    }
    
    /**
     * Get the template loader instance
     *
     * @return TemplateLoader
     */
    public function getLoader(): TemplateLoader 
    {
        return $this->loader;
    }
    
    /**
     * Get the directive processor instance
     *
     * @return DirectiveProcessor
     */
    public function getProcessor(): DirectiveProcessor 
    {
        return $this->processor;
    }
    
    /**
     * Get the template cache instance
     *
     * @return TemplateCache
     */
    public function getCache(): TemplateCache 
    {
        return $this->cache;
    }
    
    /**
     * Run garbage collection on the template cache
     * 
     * @param float $probability Probability of running GC (0-1)
     */
    public function runGarbageCollection(float $probability = 0.01): void
    {
        $this->cache->garbageCollection($probability);
    }

    /**
     * Get the config service
     * 
     * @return Config
     */
    public function getConfig(): Config
    {
        return $this->config;
    }
}