<?php
// app/baseKRIZAN/Validation/ValidationErrorRenderer.php

namespace baseKRIZAN\Validation;

use baseKRIZAN\Assets\ResourceLoader;
use baseKRIZAN\Config\Config;

/**
 * Pruža metode za renderiranje grešaka validacije u različitim formatima
 */
class ValidationErrorRenderer
{
    /**
     * Renderira modal prozor s greškama validacije
     *
     * @param array $errors Lista grešaka validacije
     * @param Config|null $config Konfiguracija
     * @return string HTML za modal prozor
     */
    public static function renderModal(array $errors, ?Config $config = null): string 
    {
        if (empty($errors)) {
            return '';
        }

        // Učitavanje CSS i JS datoteka
        if (class_exists('\\baseKRIZAN\\Assets\\ResourceLoader')) {
            ResourceLoader::loadResource('errormodal');
        }
        
        $errorHtml = self::formatErrors($errors);
        return self::getModalHtml($errorHtml);
    }
    
    /**
     * Formatira greške validacije u HTML
     *
     * @param array $errors Lista grešaka
     * @return string Formatirani HTML
     */
    private static function formatErrors(array $errors): string
    {
        $errorHtml = '';
        
        // Check if we have a flat array or field-organized array
        $isFieldBased = count(array_filter($errors, 'is_array')) > 0;
        
        if ($isFieldBased) {
            foreach ($errors as $field => $fieldErrors) {
                if (!empty($fieldErrors)) {
                    $errorHtml .= '<div class="error-field">';
                    $errorHtml .= '<h3 class="error-field-name">' . htmlspecialchars(ucfirst($field)) . ':</h3>';
                    $errorHtml .= '<ul class="error-field-list">';
                    foreach ((array)$fieldErrors as $error) {
                        $errorHtml .= '<li>' . htmlspecialchars($error) . '</li>';
                    }
                    $errorHtml .= '</ul></div>';
                }
            }
        } else {
            $errorHtml .= '<ul class="error-list">';
            foreach ($errors as $error) {
                $errorHtml .= '<li>' . htmlspecialchars($error) . '</li>';
            }
            $errorHtml .= '</ul>';
        }
        
        return $errorHtml;
    }
    
    /**
     * Generira HTML za modal prozor
     *
     * @param string $errorHtml HTML s greškama za ubaciti u modal
     * @return string Kompletni HTML za modal
     */
    private static function getModalHtml(string $errorHtml): string
    {
        return '<div id="errorModal" class="modal">
            <div class="modal-content">
                <h2>Greška <span class="close" id="errorModalClose">&times;</span></h2>
                <div class="modal-body">
                    ' . $errorHtml . '
                </div>
            </div>
        </div>';
    }
    
    /**
     * Renderira greške validacije kao HTML listu
     *
     * @param array $errors Lista grešaka
     * @return string HTML lista s greškama
     */
    public static function renderList(array $errors): string
    {
        return self::formatErrors($errors);
    }
}