<?php
// app/baseKRIZAN/Validation/ValidationResult.php

namespace baseKRIZAN\Validation;

use baseKRIZAN\Error\Logger;

/**
 * Validation result class with error handling and data access
 */
class ValidationResult {
    private array $errors;
    private array $validatedData;
    private ?Logger $logger;

    /**
     * Constructor
     */
    public function __construct(
        array $errors,
        array $validatedData,
        ?Logger $logger = null
    ) {
        $this->errors = $errors;
        $this->validatedData = $validatedData;
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->validation('ValidationResult created', [
                'has_errors' => !empty($errors),
                'errors_count' => count($errors),
                'fields_count' => count($validatedData)
            ]);
        }
    }

    /**
     * Set logger instance
     */
    public function setLogger(Logger $logger): self {
        $this->logger = $logger;
        return $this;
    }

    /**
     * Check if validation failed
     */
    public function fails(): bool {
        $hasErrors = !empty($this->errors);
        
        if ($this->logger) {
            $this->logger->validation('Validation result check - fails()', [
                'has_errors' => $hasErrors
            ]);
        }
        
        return $hasErrors;
    }

    /**
     * Check if validation passed
     */
    public function passes(): bool {
        $passes = empty($this->errors);
        
        if ($this->logger) {
            $this->logger->validation('Validation result check - passes()', [
                'passes' => $passes
            ]);
        }
        
        return $passes;
    }

    /**
     * Get validation errors
     */
    public function errors(): array {
        if ($this->logger && !empty($this->errors)) {
            $this->logger->validation('Retrieving validation errors', [
                'errors_count' => count($this->errors),
                'fields_with_errors' => array_keys($this->errors)
            ]);
        }
        
        return $this->errors;
    }

    /**
     * Get all validated data
     */
    public function validated(): array {
        if ($this->logger) {
            $this->logger->validation('Retrieving all validated data', [
                'fields_count' => count($this->validatedData),
                'fields' => array_keys($this->validatedData)
            ]);
        }
        
        return $this->validatedData;
    }

    /**
     * Get a specific validated field
     */
    public function get(string $field, $default = null) {
        $exists = isset($this->validatedData[$field]);
        $value = $this->validatedData[$field] ?? $default;
        
        if ($this->logger) {
            $this->logger->validation('Retrieving specific validated field', [
                'field' => $field,
                'exists' => $exists,
                'using_default' => !$exists && $default !== null
            ]);
        }
        
        return $value;
    }
    
    /**
     * Check if a field exists in validated data
     */
    public function has(string $field): bool {
        $exists = isset($this->validatedData[$field]);
        
        if ($this->logger) {
            $this->logger->validation('Checking if field exists in validated data', [
                'field' => $field,
                'exists' => $exists
            ]);
        }
        
        return $exists;
    }
    
    /**
     * Get only specific fields from validated data
     */
    public function only(array $fields): array {
        $result = array_intersect_key($this->validatedData, array_flip($fields));
        
        if ($this->logger) {
            $this->logger->validation('Retrieving specific fields from validated data', [
                'requested_fields' => $fields,
                'found_fields' => array_keys($result)
            ]);
        }
        
        return $result;
    }
    
    /**
     * Get all validated data except specified fields
     */
    public function except(array $fields): array {
        $result = array_diff_key($this->validatedData, array_flip($fields));
        
        if ($this->logger) {
            $this->logger->validation('Excluding fields from validated data', [
                'excluded_fields' => $fields,
                'remaining_fields' => array_keys($result)
            ]);
        }
        
        return $result;
    }
    
    /**
     * Get first error message for a field
     */
    public function getFirstError(string $field): ?string {
        $hasErrors = isset($this->errors[$field]) && !empty($this->errors[$field]);
        $firstError = $hasErrors ? $this->errors[$field][0] : null;
        
        if ($this->logger) {
            $this->logger->validation('Getting first error for field', [
                'field' => $field,
                'has_errors' => $hasErrors
            ]);
        }
        
        return $firstError;
    }
    
    /**
     * Check if a specific field has errors
     */
    public function hasError(string $field): bool {
        $hasErrors = isset($this->errors[$field]) && !empty($this->errors[$field]);
        
        if ($this->logger) {
            $this->logger->validation('Checking if field has errors', [
                'field' => $field,
                'has_errors' => $hasErrors
            ]);
        }
        
        return $hasErrors;
    }
    
    /**
     * Get all errors as a flat array
     */
    public function getAllErrors(): array {
        $allErrors = [];
        foreach ($this->errors as $fieldErrors) {
            $allErrors = array_merge($allErrors, $fieldErrors);
        }
        
        if ($this->logger) {
            $this->logger->validation('Getting all errors as flat array', [
                'total_error_count' => count($allErrors)
            ]);
        }
        
        return $allErrors;
    }
    
    /**
     * Format errors for display in a template
     */
    public function formatErrorsForTemplate(): array {
        $formattedErrors = [];
        
        foreach ($this->errors as $field => $messages) {
            $formattedErrors[$field] = implode('<br>', $messages);
        }
        
        if ($this->logger) {
            $this->logger->validation('Formatting errors for template', [
                'fields_with_errors' => array_keys($formattedErrors)
            ]);
        }
        
        return $formattedErrors;
    }
}