<?php
// app/baseKRIZAN/Validation/ValidationRules.php

namespace baseKRIZAN\Validation;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;

class ValidationRules {
    private static DatabaseTable $korisniciTable;
    private static ?Logger $logger = null;

    /**
     * Initialize the validation rules with required dependencies
     * 
     * @param DatabaseTable $korisniciTable Database table for users
     * @param Logger|null $logger Logger instance
     */
    public static function init(DatabaseTable $korisniciTable, ?Logger $logger = null) {
        self::$korisniciTable = $korisniciTable;
        self::$logger = $logger;
        
        if (self::$logger) {
            self::$logger->validation('ValidationRules initialized');
        }
    }

    /**
     * Set logger instance
     * 
     * @param Logger $logger Logger instance
     */
    public static function setLogger(Logger $logger): void {
        self::$logger = $logger;
    }

    /**
     * Register common validation rules to the validator
     * 
     * @param Validator $validator Validator instance
     */
    public static function registerCommonRules(Validator $validator) {
        // Try to get logger from validator if not set
        if (!self::$logger && property_exists($validator, 'logger')) {
            try {
                $reflection = new \ReflectionProperty($validator, 'logger');
                $reflection->setAccessible(true);
                self::$logger = $reflection->getValue($validator);
            } catch (\ReflectionException $e) {
                // Ignore if unable to get logger
            }
        }
        
        if (self::$logger) {
            self::$logger->validation('Registering common validation rules');
        }
        
        // Email validation
        $validator->addRule(
            'unique_email',
            function($value) {
                if (empty($value)) return true;
                $existing = self::$korisniciTable->find('user_email', strtolower($value));
                return empty($existing);
            },
            'This email is already registered'
        );

        $validator->addRule(
            'allowed_domain',
            function($value) {
                if (empty($value)) return true;
                $allowedDomains = explode(',', \baseKRIZAN\Config\Config::get('email.domains'));
                $emailParts = explode('@', $value);
                return count($emailParts) === 2 && in_array($emailParts[1], $allowedDomains);
            },
            'Email domain is not allowed'
        );

        // Password validation
        $validator->addRule(
            'strong_password',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/', $value);
            },
            'Password must contain at least one uppercase letter, one lowercase letter, one number and one special character'
        );

        $validator->addRule(
            'password_match',
            function($value, $password) {
                return $value === $password;
            },
            'Passwords do not match'
        );

        /*
        Rule comparison (from most restrictive to least restrictive):
        alpha_spaces    = Letters + spaces                                  [\p{L}\s]
        alpha_num      = Letters + numbers (no spaces)                     [\p{L}\d]
        plain_text     = Letters + numbers + spaces + .,!?:;@/-+           [\p{L}0-9\s\.,!?:;@\/-+]
        comment        = Letters + numbers + spaces + .,!?:;@/-+ + #[]()€%  [\p{L}0-9\s\.,!?:;@#\[\]\(\)€%\/-+]

        Usage guide:
        - alpha_spaces: For names, titles (e.g., "John Smith")
        - alpha_num: For usernames, codes, IDs (e.g., "john123")
        - plain_text: For descriptions, basic messages (e.g., "Hello, how are you?")
        - comment: For comments with special formatting (e.g., "Cost: 50€, Tax: 25%")
        */
        $validator->addRule(
            'alpha_spaces',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^[\p{L}\s]+$/u', $value);
            },
            'Field can only contain letters and spaces'
        );

        $validator->addRule(
            'alpha_num',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^[\p{L}\d]+$/u', $value);
            },
            'Field can only contain letters and numbers'
        );

        $validator->addRule(
            'plain_text',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^[\p{L}0-9\s.,!?:;@\/\-+]+$/u', $value);
            },
            'Text can contain letters, numbers, spaces, common punctuation, and plus sign'
        );

        $validator->addRule(
            'comment',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^[\p{L}0-9\s.,!?:;@#\[\]\(\)€%\/\-+]+$/u', $value);
            },
            'Text can contain letters, numbers, spaces, currencies, percentages, common punctuation, and plus sign'
        );

        $validator->addRule(
            'max_words',
            function($value, $max) {
                if (empty($value)) return true;
                return str_word_count($value) <= $max;
            },
            'Text contains too many words'
        );

        // Numbers and amounts
        $validator->addRule(
            'decimal',
            function($value, $places = 2) {
                if (empty($value)) return true;
                return preg_match('/^\d+(\.\d{1,' . $places . '})?$/', $value);
            },
            'Number must be in correct decimal format'
        );

        $validator->addRule(
            'number_between',
            function($value, $min, $max) {
                if (empty($value)) return true;
                $num = filter_var($value, FILTER_VALIDATE_FLOAT);
                return $num !== false && $num >= $min && $num <= $max;
            },
            'Number must be between {min} and {max}'
        );

        $validator->addRule(
            'currency',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^\d+([,\.]\d{1,2})?$/', $value);
            },
            'Invalid currency format'
        );

        // Date and time
        $validator->addRule(
            'datetime',
            function($value) {
                if (empty($value)) return true;
                $formats = ['d-m-Y H:i', 'Y-m-d H:i'];
                foreach ($formats as $format) {
                    $date = \DateTime::createFromFormat($format, $value);
                    if ($date && $date->format($format) === $value) {
                        return true;
                    }
                }
                return false;
            },
            'Invalid date format'
        );

        $validator->addRule(
            'date',
            function($value) {
                if (empty($value)) return true;
                $formats = ['d.m.Y', 'd-m-Y', 'Y-m-d', 'd/m/Y', 'j.n.Y', 'j/n/Y'];
                foreach ($formats as $format) {
                    $date = \DateTime::createFromFormat($format, $value);
                    if ($date && $date->format($format) === $value) {
                        return true;
                    }
                }
                return false;
            },
            'Invalid date format. Use DD.MM.YYYY or YYYY-MM-DD'
        );

        $validator->addRule(
            'future_date',
            function($value) {
                if (empty($value)) return true;
                $formats = ['d.m.Y', 'Y-m-d', 'd/m/Y'];
                foreach ($formats as $format) {
                    $date = \DateTime::createFromFormat($format, $value);
                    if ($date) {
                        return $date > new \DateTime();
                    }
                }
                return false;
            },
            'Date must be in the future'
        );

        $validator->addRule(
            'time_24h',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $value);
            },
            'Invalid time format. Use HH:MM (24h format)'
        );

        // Special formats
        $validator->addRule(
            'url_domain',
            function($value, $domain) {
                if (empty($value)) return true;
                if (!filter_var($value, FILTER_VALIDATE_URL)) {
                    return false;
                }
                return strpos(parse_url($value, PHP_URL_HOST), $domain) !== false;
            },
            'URL must be from allowed domain'
        );

        $validator->addRule(
            'phone',
            function($value) {
                if (empty($value)) return true;
                return preg_match('/^[+]?[\d\s-]{8,}$/', $value);
            },
            'Invalid phone number format'
        );

        $validator->addRule(
            'oib',
            function($value) {
                if (empty($value)) return true;
                
                // Provjeri da li je string od točno 11 karaktera
                if (strlen($value) !== 11) {
                    return false;
                }
                
                // Provjeri da li sadrži samo brojeve (dozvoljen početak sa 0 ili 00)
                if (!preg_match('/^[0-9]{11}$/', $value)) {
                    return false;
                }
                
                $a = 10;
                for ($i = 0; $i < 10; $i++) {
                    $a = ($a + (int)$value[$i]) % 10;
                    if ($a === 0) { $a = 10; }
                    $a *= 2;
                    $a %= 11;
                }
                $kontrolni = 11 - $a;
                if ($kontrolni === 10) { $kontrolni = 0; }
                
                return $kontrolni === (int)$value[10];
            },
            'Invalid OIB'
        );

        // mora biti minimalno jedan oznacen, ovo je za checkboxove u tasks
        $validator->addRule(
            'array',
            function($value) {
                if (!is_array($value)) return false;
                return count($value) > 0;
            },
            'At least one option must be selected'
        );
        
        if (self::$logger) {
            self::$logger->validation('Common validation rules registered successfully', [
                'rules_count' => 21 // Broj registriranih pravila
            ]);
        }
    }
}