<?php
// app/baseKRIZAN/Validation/Validator.php

namespace baseKRIZAN\Validation;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;

class Validator {
    private Logger $logger;
    private array $errors = [];
    private array $customRules = [];
    private array $rules = []; 
    private array $customMessages = [];
    private array $data = [];
    private bool $validated = false;

    /**
     * Constructor
     */
    public function __construct(Logger $logger) {
        $this->logger = $logger;
        $this->logger->validation('Validator instance created');
    }

    /**
     * Validate request data against rules
     */
    public function validate(Request $request, array $rules, array $messages = []): ValidationResult {
        $this->rules = $rules;
        $this->errors = [];
        $this->customMessages = $messages;
        $this->data = array_merge($request->getPost(), $request->getQuery());
        $this->validated = false;

        // Log start of validation
        $this->logger->validation('Starting request validation', [
            'rules_count' => count($rules),
            'fields' => array_keys($rules),
            'path' => $request->getPath(),
            'method' => $request->getMethod()
        ]);

        // Process each field against its rules
        foreach ($rules as $field => $fieldRules) {
            // Handle array field notation
            if (preg_match('/^(\w+)\[(\w+)\]$/', $field, $matches)) {
                $arrayData = $this->data[$matches[1]] ?? [];
                $value = $arrayData[$matches[2]] ?? null;
            } else {
                $value = $this->data[$field] ?? null;
            }

            // Convert string rules to array
            $rulesList = is_string($fieldRules) ? explode('|', $fieldRules) : $fieldRules;

            // Process each rule
            foreach ($rulesList as $rule) {
                // Parse rule and parameters
                [$ruleName, $ruleParams] = $this->parseRule($rule);

                // Apply validation rule
                if (!$this->validateField($field, $value, $ruleName, $ruleParams)) {
                    break; // Stop on first validation failure for this field
                }
            }
        }

        $this->validated = true;
        
        $hasErrors = !empty($this->errors);
        
        // Log validation result
        $this->logger->validation('Validation completed', [
            'success' => !$hasErrors,
            'errors_count' => count($this->errors),
            'fields_with_errors' => $hasErrors ? array_keys($this->errors) : []
        ]);
        
        // Log detailed errors if any 
        if ($hasErrors) {
            $this->logger->validation('Validation errors', [
                'errors' => $this->errors
            ]);
        }
        
        return new ValidationResult($this->errors, $this->getValidatedData(), $this->logger);
    }

    /**
     * Parse a validation rule into name and parameters
     */
    private function parseRule($rule): array {
        if (is_string($rule)) {
            if (str_contains($rule, ':')) {
                [$ruleName, $ruleParam] = explode(':', $rule, 2);
                $ruleParams = explode(',', $ruleParam);
            } else {
                $ruleName = $rule;
                $ruleParams = [];
            }
        } else {
            $ruleName = $rule[0];
            $ruleParams = array_slice($rule, 1);
        }

        return [$ruleName, $ruleParams];
    }

    /**
     * Add a custom validation rule
     */
    public function addRule(string $name, callable $callback, string $errorMessage): self {
        $this->customRules[$name] = [
            'callback' => $callback,
            'message' => $errorMessage
        ];
        
        $this->logger->validation("Custom validation rule registered", [
            'rule' => $name
        ]);
        
        return $this;
    }

    /**
     * Get all validation errors
     */
    public function getErrors(): array {
        return $this->errors;
    }

    /**
     * Get validated data, filtering out non-validated fields
     */
    public function getValidatedData(): array {
        if (!$this->validated) {
            $this->logger->error('Cannot get validated data before validation is performed');
            throw new \RuntimeException('Cannot get validated data before validation is performed');
        }

        $validated = [];
        foreach ($this->rules as $field => $rules) {
            if (preg_match('/^(\w+)\[(\w+)\]$/', $field, $matches)) {
                $arrayField = $matches[1];
                $arrayKey = $matches[2];
                
                if (!isset($validated[$arrayField])) {
                    $validated[$arrayField] = [];
                }
                
                if (isset($this->data[$arrayField][$arrayKey])) {
                    $validated[$arrayField][$arrayKey] = $this->data[$arrayField][$arrayKey];
                }
            } else {
                if (isset($this->data[$field])) {
                    $validated[$field] = $this->data[$field];
                }
            }
        }

        return $validated;
    }

    /**
     * Validate a single field with a specific rule
     */
    private function validateField(string $field, $value, string $rule, array $params = []): bool {
        // First check custom rules
        if (isset($this->customRules[$rule])) {
            $result = call_user_func($this->customRules[$rule]['callback'], $value, ...$params);
            
            if (!$result) {
                $errorMessage = $this->getErrorMessage($field, $rule, $params, $this->customRules[$rule]['message']);
                $this->addError($field, $errorMessage);
                
                $this->logger->validation('Field validation failed', [
                    'field' => $field,
                    'rule' => $rule,
                    'value' => is_scalar($value) ? $value : gettype($value),
                    'error' => $errorMessage
                ]);
                
                return false;
            }
            return true;
        }

        // Process built-in rules
        return match ($rule) {
            'required' => $this->validateRequired($field, $value, $params),
            'nullable' => $this->validateNullable($field, $value),
            'optional' => $this->validateOptional($field, $value),
            'email' => $this->validateEmail($field, $value),
            'min' => $this->validateMin($field, $value, $params),
            'max' => $this->validateMax($field, $value, $params),
            'matches' => $this->validateMatches($field, $value, $params),
            'numeric' => $this->validateNumeric($field, $value),
            'url' => $this->validateUrl($field, $value),
            'string' => $this->validateString($field, $value),
            'array' => $this->validateArray($field, $value),
            default => $this->handleUnknownRule($field, $rule)
        };
    }

    /**
     * Validate required field
     */
    private function validateRequired(string $field, $value, array $params): bool {
        if (empty($value) && $value !== '0') {
            $errorMessage = $this->getErrorMessage($field, 'required', $params, ucfirst($field) . ' is required');
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Required field missing', [
                'field' => $field
            ]);
            
            return false;
        }
        return true;
    }
    
    /**
     * Validate nullable field (always passes, allows null/empty values)
     */
    private function validateNullable(string $field, $value): bool {
        // Nullable fields always pass validation
        return true;
    }

    /**
     * Validate optional field
     */
    private function validateOptional(string $field, $value): bool {
        return empty($value) || true;
    }

    /**
     * Validate email field
     */
    private function validateEmail(string $field, $value): bool {
        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $errorMessage = $this->getErrorMessage($field, 'email', [], 'Invalid email format');
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Invalid email format', [
                'field' => $field,
                'value' => $value
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate minimum length
     */
    private function validateMin(string $field, $value, array $params): bool {
        if (!empty($value) && strlen($value) < ($params[0] ?? 0)) {
            $errorMessage = $this->getErrorMessage(
                $field, 
                'min', 
                $params, 
                ucfirst($field) . " must be at least {$params[0]} characters"
            );
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Value too short', [
                'field' => $field,
                'min_length' => $params[0] ?? 0,
                'actual_length' => strlen($value)
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate maximum length
     */
    private function validateMax(string $field, $value, array $params): bool {
        if (!empty($value) && strlen($value) > ($params[0] ?? 255)) {
            $errorMessage = $this->getErrorMessage(
                $field, 
                'max', 
                $params, 
                ucfirst($field) . " must not exceed {$params[0]} characters"
            );
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Value too long', [
                'field' => $field,
                'max_length' => $params[0] ?? 255,
                'actual_length' => strlen($value)
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate matches other field
     */
    private function validateMatches(string $field, $value, array $params): bool {
        $otherField = $params[0] ?? '';
        $otherValue = $this->data[$otherField] ?? null;
        if ($value !== $otherValue) {
            $errorMessage = $this->getErrorMessage(
                $field, 
                'matches', 
                $params, 
                ucfirst($field) . " must match " . ucfirst($otherField)
            );
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Values do not match', [
                'field' => $field,
                'other_field' => $otherField
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate numeric value
     */
    private function validateNumeric(string $field, $value): bool {
        if (!empty($value) && !is_numeric($value)) {
            $errorMessage = $this->getErrorMessage($field, 'numeric', [], ucfirst($field) . " must be numeric");
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Value not numeric', [
                'field' => $field,
                'value' => $value
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate URL
     */
    private function validateUrl(string $field, $value): bool {
        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_URL)) {
            $errorMessage = $this->getErrorMessage($field, 'url', [], ucfirst($field) . " must be a valid URL");
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Invalid URL format', [
                'field' => $field,
                'value' => $value
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate string value
     */
    private function validateString(string $field, $value): bool {
        if (!empty($value) && !is_string($value)) {
            $errorMessage = $this->getErrorMessage($field, 'string', [], ucfirst($field) . " must be a string");
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Value not a string', [
                'field' => $field,
                'type' => gettype($value)
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Validate array value
     */
    private function validateArray(string $field, $value): bool {
        if (!empty($value) && !is_array($value)) {
            $errorMessage = $this->getErrorMessage($field, 'array', [], ucfirst($field) . " must be an array");
            $this->addError($field, $errorMessage);
            
            $this->logger->validation('Value not an array', [
                'field' => $field,
                'type' => gettype($value)
            ]);
            
            return false;
        }
        return true;
    }

    /**
     * Handle unknown validation rule
     */
    private function handleUnknownRule(string $field, string $rule): bool {
        $this->logger->validation("Unknown validation rule", [
            'rule' => $rule,
            'field' => $field
        ]);
        return true;
    }

    /**
     * Add an error message for a field
     */
    private function addError(string $field, string $message): void {
        if (!isset($this->errors[$field])) {
            $this->errors[$field] = [];
        }
        $this->errors[$field][] = $message;
    }

    /**
     * Get a customized error message if available
     */
    private function getErrorMessage(string $field, string $rule, array $params, string $defaultMessage): string {
        // Check for field.rule specific message
        if (isset($this->customMessages["$field.$rule"])) {
            $message = $this->customMessages["$field.$rule"];
        } 
        // Check for field specific message
        elseif (isset($this->customMessages[$field])) {
            $message = $this->customMessages[$field];
        }
        // Check for rule specific message
        elseif (isset($this->customMessages[$rule])) {
            $message = $this->customMessages[$rule];
        }
        // Use the default message
        else {
            $message = $defaultMessage;
        }

        // Replace placeholders with parameter values
        foreach ($params as $index => $param) {
            $message = str_replace("{{$index}}", $param, $message);
        }

        // Replace field name placeholder
        $message = str_replace("{field}", ucfirst($field), $message);

        return $message;
    }
}