<?php
// app/config/app_basedependencies.php

use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;
use Models\DatabaseTable;
use Models\NotificationModel;
use Models\FCMTokenModel;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Validation\ValidationRules;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Services\EmailService;
use baseKRIZAN\Services\FCMService;
use baseKRIZAN\Services\FirebaseAuthService;
use baseKRIZAN\Services\FirebaseRealtimeService;
use baseKRIZAN\Services\NotificationService;
use baseKRIZAN\Services\NotificationEnhancerService;
use baseKRIZAN\Http\RequestClassifier;

/**
 * Registers basic dependencies in the Container.
 * 
 * @param Container $container DI container
 * @return void
 */
return function(Container $container) {
    // Database connection from container
    $dbConnection = $container->get('dbConnection');
    
    // Setup logger
    $logPath = \baseKRIZAN\Config\Config::get('paths.logs');
    if (!$container->has('logger')) {
        $logger = new Logger($logPath, \baseKRIZAN\Config\Config::get('environment') === 'development');
        $container->register('logger', $logger, 'singleton');
    } else {
        $logger = $container->get('logger');
    }
    
    // Log services initialization
    $logger->bootstrap('Initializing base services and dependencies');
    
    // Registration of database tables as services
    registerDatabaseTables($container, $dbConnection, $logger);
    
    // Registration of models
    registerModels($container, $dbConnection, $logger);
    
    // Registration of core services
    registerCoreServices($container, $logger);
    
    // Initialize validation rules
    initializeValidationRules($container, $logger);
    
    // Registration of complete services
    registerApplicationServices($container, $logger);
    
    // Registration of controllers
    registerControllers($container, $logger);
    
    $logger->bootstrap('Base dependencies initialization completed');
};

/**
 * Registers database table services
 *
 * @param Container $container Container instance
 * @param \baseKRIZAN\Database\DatabaseConnection $dbConnection DatabaseConnection instance
 * @param Logger $logger Logger instance
 * @return void
 */
function registerDatabaseTables(Container $container, \baseKRIZAN\Database\DatabaseConnection $dbConnection, Logger $logger): void
{
    $databaseTables = [
        'passwordresetTable' => ['korisnici_password_reset', 'id'],
        'sessionTable' => ['korisnici_sessions', 'id'],
        'notificationsTable' => ['notifications', 'id'],
        'notificationTokensTable' => ['notifications_user_tokens', 'id'],
    ];
    
    // Register all tables using DatabaseConnection
    foreach ($databaseTables as $tableName => [$table, $primaryKey]) {
        $container->prototype($tableName, function() use ($dbConnection, $table, $primaryKey) {
            return new DatabaseTable($dbConnection, $table, $primaryKey);
        });
    }
    
    // Special handling for korisniciTable due to dependencies
    $container->prototype('korisniciTable', function($container) use ($dbConnection) {
        $notificationsTable = $container->get('notificationsTable');
        return new DatabaseTable($dbConnection, 'korisnici', 'id', '\Models\Entity\Korisnik', [$notificationsTable]);
    });
    
    $logger->bootstrap('Database tables initialized with DatabaseConnection');
}

/**
 * Registers models
 *
 * @param Container $container Container instance
 * @param \baseKRIZAN\Database\DatabaseConnection $dbConnection DatabaseConnection instance
 * @param Logger $logger Logger instance
 * @return void
 */
function registerModels(Container $container, \baseKRIZAN\Database\DatabaseConnection $dbConnection, Logger $logger): void
{
    // Configure DatabaseConnection with eventDispatcher and logger
    if ($container->has('eventDispatcher')) {
        $dbConnection->setServices(
            $container->get('eventDispatcher'),
            $logger
        );
    }
    
    $models = [
        'notificationModel' => function($container) use ($dbConnection) {
            return new NotificationModel($dbConnection, $container->get('logger'));
        },
        'fcmTokenModel' => function($container) use ($dbConnection) {
            return new FCMTokenModel($dbConnection, $container->get('logger'));
        }
    ];
    
    // Register all models as singletons
    foreach ($models as $modelName => $factory) {
        $container->singleton($modelName, $factory);
    }
}

/**
 * Registers core services
 *
 * @param Container $container Container instance
 * @param Logger $logger Logger instance 
 * @return void
 */
function registerCoreServices(Container $container, Logger $logger): void
{
    $coreServices = [
        'requestClassifier' => function($container) {
            return new RequestClassifier($container->get('logger'));
        },
        'sessionManager' => function($container) {
            return new SessionManager(
                $container->get('logger'), 
                $container->get('requestClassifier')
            );
        },
        'validator' => function($container) {
            return new Validator($container->get('logger'));
        },
        'eventDispatcher' => function($container) {
            return new \baseKRIZAN\Events\EventDispatcher($container->get('logger'));
        }
    ];
    
    // Register all core services as singletons
    foreach ($coreServices as $serviceName => $factory) {
        $container->singleton($serviceName, $factory);
    }
    
    $logger->bootstrap('Core services initialized');
}

function initializeValidationRules(Container $container, Logger $logger): void 
{
    try {
        if ($container->has('korisniciTable') && $container->has('validator')) {
            ValidationRules::init($container->get('korisniciTable'), $logger);
            ValidationRules::registerCommonRules($container->get('validator'));
            $logger->bootstrap('ValidationRules initialized and registered');
        } else {
            $logger->bootstrap('ValidationRules initialization deferred - required dependencies not available');
        }
    } catch (\Throwable $e) {
        $logger->error('Failed to initialize ValidationRules', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    }
}

/**
 * Registers application services
 * 
 * @param Container $container Container instance
 * @param Logger $logger Logger instance
 * @return void
 */
function registerApplicationServices(Container $container, Logger $logger): void
{
    // Check notification features enabled state from config
    $fcmEnabled = (bool)\baseKRIZAN\Config\Config::get('notifications.fcm_enabled');
    $realtimeDbEnabled = (bool)\baseKRIZAN\Config\Config::get('notifications.realtime_db_enabled');
    
    $logger->bootstrap('Notification features status', [
        'fcm_enabled' => $fcmEnabled,
        'realtime_db_enabled' => $realtimeDbEnabled
    ]);
    
    $applicationServices = [
        'authentication' => function($container) {
            return new Authentication(
                $container->get('korisniciTable'), 
                'user_email', 
                'user_password',
                $container->get('logger'),
                $container->get('sessionManager')
            );
        },
        'emailService' => function($container) {
            return new EmailService($container->get('logger'));
        },
        'firebaseAuthService' => function($container) {
            return new FirebaseAuthService($container->get('logger'));
        },
        'fcmService' => function($container) use ($fcmEnabled) {
            if (!$fcmEnabled) {
                $container->get('logger')->bootstrap('FCM Service disabled via config');
                return null;
            }
            return new FCMService(
                $container->get('logger'),
                $container->get('fcmTokenModel')
            );
        },
        'firebaseRealtimeService' => function($container) use ($realtimeDbEnabled) {
            if (!$realtimeDbEnabled) {
                $container->get('logger')->bootstrap('Firebase Realtime Service disabled via config');
                return null;
            }
            return new FirebaseRealtimeService(
                $container->get('logger'),
                $container->get('firebaseAuthService')
            );
        },
        'notificationService' => function($container) {
            return new NotificationService(
                $container->get('logger'),
                $container->get('notificationModel'),
                $container->get('korisniciTable')
            );
        },
        'notificationEnhancerService' => function($container) {
            return new NotificationEnhancerService(
                $container->get('logger'),
                $container->get('notificationService'),
                $container->has('fcmService') ? $container->get('fcmService') : null,
                $container->has('firebaseRealtimeService') ? $container->get('firebaseRealtimeService') : null
            );
        }
    ];
    
    // Register all application services as singletons
    foreach ($applicationServices as $serviceName => $factory) {
        $container->singleton($serviceName, $factory);
    }
    
    $logger->bootstrap('Application services initialized');
}

/**
 * Registers application controllers
 * 
 * @param Container $container Container instance
 * @param Logger $logger Logger instance
 * @return void 
 */
function registerControllers(Container $container, Logger $logger): void
{
    $controllers = [
        'registerController' => \Controllers\Register::class,
        'loginController' => \Controllers\Login::class,
        'userController' => function($container) {
            return new \Controllers\User(
                $container->get('korisniciTable'),
                $container->get('authentication'),
                $container->get('validator'),
                $container->get('logger'),
                $container->get('sessionManager'),
                $container->has('emailService') ? $container->get('emailService') : null
            );
        },
        'passwordController' => \Controllers\Password::class,
        'cmsController' => \Controllers\CMS::class,
        'homeController' => \Controllers\Home::class,
        'dbmanagementController' => \Controllers\DB::class,
        'notificationController' => \Controllers\NotificationController::class,
        'securityController' => \Controllers\Security::class,
        'fcmController' => \Controllers\FCMController::class,
        'firebaseRealtimeController' => \Controllers\FirebaseRealtimeController::class,
        'initialDataController' => \Controllers\InitialDataController::class,
    ];
    
    // Checking if LUKA module is enabled before registering LUKA controller
    if (\baseKRIZAN\Config\Config::get('luka.enabled', false)) {
        $controllers['lukaController'] = \Controllers\LUKA::class;
        $logger->bootstrap('LUKA controller registered - module is enabled');
    } else {
        $logger->bootstrap('LUKA controller skipped - module is disabled');
    }
    
    // Checking if BORNA module is enabled before registering BORNA controller
    if (\baseKRIZAN\Config\Config::get('borna.enabled', false)) {
        $controllers['bornaController'] = \Controllers\BORNA::class;
        // Also add BORNATest controller when BORNA module is enabled
        $controllers['bornaTestController'] = \Controllers\BORNATest::class;
        $logger->bootstrap('BORNA controller registered - module is enabled');
    } else {
        $logger->bootstrap('BORNA controller skipped - module is disabled');
    }
    
    $logger->bootstrap('Registering controllers', [
        'count' => count($controllers),
        'controllers' => array_keys($controllers)
    ]);

    // Register all controllers as request scope
    foreach ($controllers as $id => $class) {
        if (is_callable($class)) {
            // Custom factory function
            $container->request($id, $class);
        } elseif (class_exists($class)) {
            // Standard class registration
            $container->request($id, $class);
        } else {
            $logger->bootstrap("Controller class not found during registration", [
                'id' => $id,
                'class' => is_string($class) ? $class : 'callable'
            ]);
        }
    }
}