/**
 * BORNA IPs Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    searchIps: `${publicUrl}/borna/security/api/search-ips`,
    ipLookup: `${publicUrl}/borna/security/api/ip-info`
  };

  // Elements
  const addBlockBtn = document.getElementById('addBlockBtn');
  const emptyAddBlockBtn = document.getElementById('emptyAddBlockBtn');
  const cancelBlockBtn = document.getElementById('cancelBlockBtn');
  const blockIpForm = document.getElementById('blockIpForm');
  const ipSearch = document.getElementById('ipSearch');
  const alertElements = document.querySelectorAll('.borna-alert');
  
  // Show the IP blocking form
  function showBlockForm() {
    blockIpForm.style.display = 'block';
    // Scroll to the form
    blockIpForm.scrollIntoView({ behavior: 'smooth' });
    // Focus on the IP input field
    document.getElementById('ip').focus();
  }
  
  // Hide the IP blocking form
  function hideBlockForm() {
    blockIpForm.style.display = 'none';
  }
  
  // Event listeners
  if (addBlockBtn) {
    addBlockBtn.addEventListener('click', showBlockForm);
  }
  
  if (emptyAddBlockBtn) {
    emptyAddBlockBtn.addEventListener('click', showBlockForm);
  }
  
  if (cancelBlockBtn) {
    cancelBlockBtn.addEventListener('click', hideBlockForm);
  }
  
  // IP search functionality
  if (ipSearch) {
    ipSearch.addEventListener('keyup', function() {
      const searchValue = this.value.toLowerCase();
      fetchFilteredIPs(searchValue);
    });
  }
  
  /**
   * Fetch and filter IPs via API
   */
  function fetchFilteredIPs(searchValue) {
    const ipTable = document.getElementById('ipTable');
    const noIpsMessage = document.getElementById('noIpsMessage');
    
    if (!ipTable) return;
    
    // Show loading state
    const tbody = ipTable.querySelector('tbody');
    if (tbody) {
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="text-center">
            <div class="loading-spinner"></div>
            <p>Searching IPs...</p>
          </td>
        </tr>
      `;
    }
    
    // Fetch IPs from API
    fetch(`${routes.searchIps}?q=${encodeURIComponent(searchValue)}`)
      .then(response => {
        if (!response.ok) {
          throw new Error('Network response was not ok');
        }
        return response.json();
      })
      .then(data => {
        // Clear loading state
        if (tbody) {
          tbody.innerHTML = '';
        }
        
        if (data.success && data.ips && data.ips.length > 0) {
          // Populate table with filtered IPs
          data.ips.forEach(ip => {
            const row = document.createElement('tr');
            row.innerHTML = `
              <td>${escapeHtml(ip.address)}</td>
              <td>${escapeHtml(ip.reason || 'No reason provided')}</td>
              <td>${escapeHtml(ip.block_date)}</td>
              <td>${escapeHtml(ip.expiration_date || 'Permanent')}</td>
              <td>
                <form method="post" class="unblock-form">
                  <input type="hidden" name="ip" value="${escapeHtml(ip.address)}">
                  <button type="submit" name="action" value="unblock" class="btn btn-sm btn-danger">
                    <i class="fas fa-unlock"></i> Unblock
                  </button>
                </form>
              </td>
            `;
            tbody.appendChild(row);
          });
          
          // Hide no IPs message
          if (noIpsMessage) {
            noIpsMessage.style.display = 'none';
          }
          
          // Show table
          if (ipTable) {
            ipTable.style.display = 'table';
          }
        } else {
          // No IPs found
          if (noIpsMessage) {
            noIpsMessage.style.display = 'block';
            noIpsMessage.textContent = searchValue 
              ? 'No matching IPs found.' 
              : 'No blocked IPs exist.';
          }
          
          // Hide table
          if (ipTable) {
            ipTable.style.display = 'none';
          }
        }
      })
      .catch(error => {
        console.error('Error searching IPs:', error);
        
        // Clear loading state and show error
        if (tbody) {
          tbody.innerHTML = `
            <tr>
              <td colspan="5" class="text-center text-danger">
                <i class="fas fa-exclamation-triangle"></i>
                Failed to search IPs. ${error.message}
              </td>
            </tr>
          `;
        }
      });
  }
  
  /**
   * Escape HTML to prevent XSS
   * @param {string} html - The string to escape
   * @returns {string} - The escaped string
   */
  function escapeHtml(html) {
    if (!html) return '';
    const div = document.createElement('div');
    div.textContent = html;
    return div.innerHTML;
  }
  
  // Check if IPs table is empty
  checkEmptyIPs();
  
  /**
   * Check if IPs table is empty and show message if needed
   */
  function checkEmptyIPs() {
    const ipTable = document.getElementById('ipTable');
    const noIpsMessage = document.getElementById('noIpsMessage');
    
    if (ipTable && noIpsMessage) {
      const rows = ipTable.querySelectorAll('tbody tr');
      
      if (rows.length === 0) {
        // No IPs, show message
        noIpsMessage.style.display = 'block';
        ipTable.style.display = 'none';
        
        // Show the "add new IP block" section
        if (emptyAddBlockBtn) {
          const emptyState = document.getElementById('emptyState');
          if (emptyState) {
            emptyState.style.display = 'block';
          }
        }
      } else {
        // IPs found, hide message
        noIpsMessage.style.display = 'none';
        ipTable.style.display = 'table';
        
        // Hide the empty state section
        const emptyState = document.getElementById('emptyState');
        if (emptyState) {
          emptyState.style.display = 'none';
        }
      }
    }
  }
  
  // Auto-dismiss alerts after 5 seconds
  if (alertElements.length > 0) {
    setTimeout(() => {
      alertElements.forEach(alert => {
        alert.style.opacity = '0';
        setTimeout(() => {
          alert.style.display = 'none';
        }, 300); // Match the CSS transition duration
      });
    }, 5000);
  }
  
  // Confirmation for unblocking IPs
  const unblockForms = document.querySelectorAll('.unblock-form');
  
  unblockForms.forEach(form => {
    form.addEventListener('submit', function(e) {
      const ipInput = this.querySelector('input[name="ip"]');
      if (!ipInput) return;
      
      const ipAddress = ipInput.value;
      if (!confirm(`Are you sure you want to unblock ${ipAddress}?`)) {
        e.preventDefault();
      }
    });
  });
  
  // IP validation for the block form
  const blockForm = document.querySelector('form[action*="ips"]');
  
  if (blockForm) {
    blockForm.addEventListener('submit', function(e) {
      const ipInput = document.getElementById('ip');
      if (!ipInput) return;
      
      const ipAddress = ipInput.value.trim();
      
      // Basic IP validation using regex
      const ipv4Regex = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
      const ipv6Regex = /^([0-9a-fA-F]{1,4}:){7}([0-9a-fA-F]{1,4})$/;
      
      if (!ipv4Regex.test(ipAddress) && !ipv6Regex.test(ipAddress)) {
        alert('Please enter a valid IP address.');
        e.preventDefault();
        return;
      }
      
      // Additional validation for IPv4
      if (ipv4Regex.test(ipAddress)) {
        const parts = ipAddress.match(ipv4Regex);
        for (let i = 1; i <= 4; i++) {
          const part = parseInt(parts[i], 10);
          if (part < 0 || part > 255) {
            alert('Please enter a valid IPv4 address. Each segment must be between 0 and 255.');
            e.preventDefault();
            return;
          }
        }
      }
    });
  }
  
  // Add expiration time calculation
  const durationInput = document.getElementById('duration');
  const expirationPreview = document.getElementById('expiration-preview');
  
  if (durationInput && expirationPreview) {
    durationInput.addEventListener('input', updateExpirationPreview);
    durationInput.addEventListener('change', updateExpirationPreview);
    
    // Initial update
    updateExpirationPreview();
  }
  
  function updateExpirationPreview() {
    const duration = parseInt(durationInput.value, 10) || 24; // Default to 24 hours
    
    const now = new Date();
    const expiration = new Date(now.getTime() + (duration * 60 * 60 * 1000)); // Convert hours to milliseconds
    
    // Format date for display
    const formattedDate = expiration.toLocaleString(undefined, {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
    
    expirationPreview.textContent = `IP will be blocked until: ${formattedDate}`;
  }
  
  // Show IP details from GeoIP lookup
  const ipInput = document.getElementById('ip');
  const lookupBtn = document.getElementById('ip-lookup-btn');
  const ipInfoContainer = document.getElementById('ip-info');
  
  if (ipInput && lookupBtn && ipInfoContainer) {
    lookupBtn.addEventListener('click', function() {
      const ip = ipInput.value.trim();
      
      if (!ip) {
        ipInfoContainer.innerHTML = `
          <div class="borna-alert warning">
            <i class="fas fa-exclamation-triangle"></i>
            <div>Please enter an IP address first.</div>
          </div>
        `;
        return;
      }
      
      // Show loading state
      ipInfoContainer.innerHTML = `
        <div class="loading-spinner"></div>
        <p class="text-center">Looking up IP information...</p>
      `;
      
      // Make API request to get IP information
      fetch(`${routes.ipLookup}?ip=${encodeURIComponent(ip)}`)
        .then(response => {
          if (!response.ok) {
            throw new Error('Network response was not ok');
          }
          return response.json();
        })
        .then(data => {
          if (data.success) {
            // Display IP information
            ipInfoContainer.innerHTML = `
              <div class="ip-info-box">
                <div class="ip-info-row">
                  <span class="ip-info-label">Country:</span>
                  <span class="ip-info-value">${data.country || 'Unknown'}</span>
                </div>
                ${data.city ? `
                <div class="ip-info-row">
                  <span class="ip-info-label">City:</span>
                  <span class="ip-info-value">${data.city}</span>
                </div>
                ` : ''}
                ${data.isp ? `
                <div class="ip-info-row">
                  <span class="ip-info-label">ISP:</span>
                  <span class="ip-info-value">${data.isp}</span>
                </div>
                ` : ''}
                ${data.threat_score ? `
                <div class="ip-info-row">
                  <span class="ip-info-label">Threat Score:</span>
                  <span class="ip-info-value">${data.threat_score} / 100</span>
                </div>
                ` : ''}
                ${data.previous_blocks ? `
                <div class="ip-info-row">
                  <span class="ip-info-label">Previous Blocks:</span>
                  <span class="ip-info-value">${data.previous_blocks}</span>
                </div>
                ` : ''}
              </div>
            `;
            
            // Auto-fill reason if it's empty and threat info is available
            const reasonInput = document.getElementById('reason');
            if (reasonInput && !reasonInput.value.trim() && data.threat_info) {
              reasonInput.value = data.threat_info;
            }
          } else {
            ipInfoContainer.innerHTML = `
              <div class="borna-alert warning">
                <i class="fas fa-exclamation-triangle"></i>
                <div>${data.message || 'Could not retrieve IP information.'}</div>
              </div>
            `;
          }
        })
        .catch(error => {
          console.error('Error looking up IP:', error);
          ipInfoContainer.innerHTML = `
            <div class="borna-alert danger">
              <i class="fas fa-exclamation-circle"></i>
              <div>Error looking up IP: ${error.message}</div>
            </div>
          `;
        });
    });
  }
});