/**
 * BORNA Rules Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    search: `${publicUrl}/borna/firewall/api/search-rules`
  };

  // Search functionality
  const searchInput = document.getElementById('rulesSearch');
  if (searchInput) {
    searchInput.addEventListener('keyup', function() {
      const searchText = this.value.toLowerCase();
      fetchFilteredRules(searchText);
    });
  }
  
  // Auto-hide alerts after 5 seconds
  const alerts = document.querySelectorAll('.borna-alert');
  alerts.forEach(alert => {
    setTimeout(() => {
      alert.style.opacity = '0';
      setTimeout(() => {
        alert.style.display = 'none';
      }, 300);
    }, 5000);
  });
  
  // Check if rules table is empty
  checkEmptyRules();
  
  /**
   * Check if rules table is empty and show message if needed
   */
  function checkEmptyRules() {
    const rulesTable = document.querySelector('.rules-table');
    const noRulesMessage = document.getElementById('noRulesMessage');
    
    if (rulesTable && noRulesMessage) {
      const rows = rulesTable.querySelectorAll('tbody tr');
      
      if (rows.length === 0) {
        // No rules, show message
        noRulesMessage.style.display = 'block';
        rulesTable.style.display = 'none';
      } else {
        // Rules found, hide message
        noRulesMessage.style.display = 'none';
        rulesTable.style.display = 'table';
      }
    }
  }
  
  /**
   * Fetch and filter rules via API
   */
  function fetchFilteredRules(searchText) {
    const rulesTable = document.querySelector('.rules-table tbody');
    const noRulesMessage = document.getElementById('noRulesMessage');
    
    if (!rulesTable) return;
    
    // Show loading state
    rulesTable.innerHTML = `
      <tr>
        <td colspan="5" class="text-center">
          <div class="loading-spinner"></div>
          <p>Searching rules...</p>
        </td>
      </tr>
    `;
    
    // Fetch rules from API
    fetch(`${routes.search}?q=${encodeURIComponent(searchText)}`)
      .then(response => {
        if (!response.ok) {
          throw new Error('Network response was not ok');
        }
        return response.json();
      })
      .then(data => {
        // Clear loading state
        rulesTable.innerHTML = '';
        
        if (data.success && data.rules && data.rules.length > 0) {
          // Populate table with filtered rules
          data.rules.forEach(rule => {
            const row = document.createElement('tr');
            row.innerHTML = `
              <td class="rule-name">${escapeHtml(rule.name)}</td>
              <td class="rule-description">${escapeHtml(rule.description)}</td>
              <td class="rule-type">${escapeHtml(rule.type)}</td>
              <td class="pattern-cell">
                <span class="pattern-display">${escapeHtml(rule.pattern)}</span>
              </td>
              <td>
                <div class="rule-actions">
                  <form method="post">
                    <input type="hidden" name="id" value="${rule.id}">
                    <button type="submit" name="action" value="edit" class="btn btn-sm btn-primary">
                      <i class="fas fa-edit"></i>
                    </button>
                    <button type="submit" name="action" value="delete" class="btn btn-sm btn-danger delete-rule-btn" 
                      data-rule-name="${escapeHtml(rule.name)}">
                      <i class="fas fa-trash"></i>
                    </button>
                  </form>
                </div>
              </td>
            `;
            rulesTable.appendChild(row);
          });
          
          // Hide no rules message
          if (noRulesMessage) {
            noRulesMessage.style.display = 'none';
          }
        } else {
          // No rules found
          if (noRulesMessage) {
            noRulesMessage.style.display = 'block';
            noRulesMessage.textContent = searchText 
              ? 'No matching rules found.' 
              : 'No rules exist.';
          }
        }
      })
      .catch(error => {
        console.error('Error searching rules:', error);
        
        // Clear loading state and show error
        rulesTable.innerHTML = `
          <tr>
            <td colspan="5" class="text-center text-danger">
              <i class="fas fa-exclamation-triangle"></i>
              Failed to search rules. ${error.message}
            </td>
          </tr>
        `;
      });
  }
  
  /**
   * Escape HTML to prevent XSS
   * @param {string} html - The string to escape
   * @returns {string} - The escaped string
   */
  function escapeHtml(html) {
    const div = document.createElement('div');
    div.textContent = html;
    return div.innerHTML;
  }
  
  // Form validation
  const ruleForm = document.querySelector('.rule-form');
  if (ruleForm) {
    ruleForm.addEventListener('submit', function(e) {
      let valid = true;
      
      // Required fields
      const requiredFields = this.querySelectorAll('[required]');
      requiredFields.forEach(field => {
        if (!field.value.trim()) {
          field.style.borderColor = 'var(--borna-danger)';
          valid = false;
          
          // Add error message if not exists
          const errorMsg = field.parentNode.querySelector('.field-error');
          if (!errorMsg) {
            const msg = document.createElement('div');
            msg.className = 'field-error';
            msg.textContent = 'This field is required';
            field.parentNode.appendChild(msg);
          }
        } else {
          field.style.borderColor = '';
          const errorMsg = field.parentNode.querySelector('.field-error');
          if (errorMsg) {
            errorMsg.remove();
          }
        }
      });
      
      // Validate pattern field for regex syntax
      const patternField = document.getElementById('pattern');
      if (patternField && patternField.value.trim()) {
        try {
          new RegExp(patternField.value);
          patternField.style.borderColor = '';
          const errorMsg = patternField.parentNode.querySelector('.field-error');
          if (errorMsg) {
            errorMsg.remove();
          }
        } catch (error) {
          patternField.style.borderColor = 'var(--borna-danger)';
          valid = false;
          
          // Add error message if not exists
          const errorMsg = patternField.parentNode.querySelector('.field-error');
          if (!errorMsg) {
            const msg = document.createElement('div');
            msg.className = 'field-error';
            msg.textContent = 'Invalid regular expression';
            patternField.parentNode.appendChild(msg);
          } else {
            errorMsg.textContent = 'Invalid regular expression';
          }
        }
      }
      
      if (!valid) {
        e.preventDefault();
      }
    });
    
    // Clear error styling on input
    const formInputs = ruleForm.querySelectorAll('input, select, textarea');
    formInputs.forEach(input => {
      input.addEventListener('input', function() {
        this.style.borderColor = '';
        const errorMsg = this.parentNode.querySelector('.field-error');
        if (errorMsg) {
          errorMsg.remove();
        }
      });
    });
  }
  
  // Toggle rule status
  const toggleSwitches = document.querySelectorAll('.rule-toggle');
  if (toggleSwitches.length > 0) {
    toggleSwitches.forEach(toggle => {
      toggle.addEventListener('change', function() {
        const ruleId = this.getAttribute('data-rule-id');
        const enabled = this.checked ? 1 : 0;
        
        if (!ruleId) return;
        
        // Create and submit form
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'rules';
        form.style.display = 'none';
        
        // Add action and rule ID
        const actionInput = document.createElement('input');
        actionInput.type = 'hidden';
        actionInput.name = 'action';
        actionInput.value = 'toggle';
        form.appendChild(actionInput);
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'id';
        idInput.value = ruleId;
        form.appendChild(idInput);
        
        // Add enabled status
        const enabledInput = document.createElement('input');
        enabledInput.type = 'hidden';
        enabledInput.name = 'enabled';
        enabledInput.value = enabled;
        form.appendChild(enabledInput);
        
        // Add to document and submit
        document.body.appendChild(form);
        form.submit();
      });
    });
  }
  
  // Confirm rule deletion
  const deleteButtons = document.querySelectorAll('.delete-rule-btn');
  if (deleteButtons.length > 0) {
    deleteButtons.forEach(button => {
      button.addEventListener('click', function(e) {
        e.preventDefault();
        
        const ruleName = this.getAttribute('data-rule-name');
        const confirmMessage = `Are you sure you want to delete the rule "${ruleName}"? This action cannot be undone.`;
        
        if (confirm(confirmMessage)) {
          // Get the form and submit it
          const form = this.closest('form');
          if (form) {
            form.submit();
          }
        }
      });
    });
  }
  
  // Toggle hover effect on rule type display
  const patternCells = document.querySelectorAll('.pattern-cell');
  patternCells.forEach(cell => {
    cell.addEventListener('mouseenter', function() {
      const pattern = this.querySelector('.pattern-display');
      if (pattern && pattern.offsetWidth < pattern.scrollWidth) {
        pattern.title = pattern.textContent;
      }
    });
  });
  
  // Rule type change handler for form fields
  const ruleTypeSelect = document.getElementById('type');
  if (ruleTypeSelect) {
    ruleTypeSelect.addEventListener('change', function() {
      updatePatternFieldHint(this.value);
    });
    
    // Initialize with current value
    updatePatternFieldHint(ruleTypeSelect.value);
  }
  
  /**
   * Update pattern field hint based on rule type
   */
  function updatePatternFieldHint(ruleType) {
    const hintElement = document.getElementById('patternHint');
    if (!hintElement) return;
    
    let hintText = '';
    
    switch (ruleType) {
      case 'path':
        hintText = 'Enter a regex pattern to match request paths. Example: /\\.git';
        break;
      case 'parameter':
        hintText = 'Enter a regex pattern to match parameter values. Example: /\'.*OR.*1=1/i';
        break;
      case 'user_agent':
        hintText = 'Enter a regex pattern to match user agents. Example: /(nmap|sqlmap)/i';
        break;
      case 'ip':
        hintText = 'Enter a regex pattern to match IP addresses. Example: /^192\\.168\\.1\\./';
        break;
      case 'method':
        hintText = 'Enter a regex pattern to match HTTP methods. Example: /^(DELETE|PUT)$/i';
        break;
      case 'header':
        hintText = 'Enter a regex pattern to match header values. Example: /(curl|wget)/i';
        break;
      default:
        hintText = 'Enter a valid regular expression pattern';
    }
    
    hintElement.textContent = hintText;
  }
  
  // Add test pattern feature if on edit page
  const testPatternBtn = document.getElementById('testPatternBtn');
  if (testPatternBtn) {
    testPatternBtn.addEventListener('click', function() {
      const patternField = document.getElementById('pattern');
      const testInputField = document.getElementById('testPatternInput');
      
      if (!patternField || !testInputField) return;
      
      const pattern = patternField.value.trim();
      const testValue = testInputField.value.trim();
      const resultElement = document.getElementById('testPatternResult');
      
      if (!pattern || !testValue) {
        if (resultElement) {
          resultElement.innerHTML = `
            <div class="borna-alert warning">
              <i class="fas fa-exclamation-triangle"></i>
              <div>Please enter both a pattern and a test value.</div>
            </div>
          `;
        }
        return;
      }
      
      try {
        const regex = new RegExp(pattern);
        const matches = regex.test(testValue);
        
        if (resultElement) {
          if (matches) {
            resultElement.innerHTML = `
              <div class="borna-alert success">
                <i class="fas fa-check-circle"></i>
                <div>Pattern matches the test value!</div>
              </div>
            `;
          } else {
            resultElement.innerHTML = `
              <div class="borna-alert warning">
                <i class="fas fa-times-circle"></i>
                <div>Pattern does not match the test value.</div>
              </div>
            `;
          }
        }
      } catch (error) {
        if (resultElement) {
          resultElement.innerHTML = `
            <div class="borna-alert danger">
              <i class="fas fa-exclamation-circle"></i>
              <div>Invalid regular expression: ${error.message}</div>
            </div>
          `;
        }
      }
    });
  }
});