/**
 * BORNA Reports Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    generateReport: `${publicUrl}/borna/reports/api/generate`,
    exportReport: `${publicUrl}/borna/reports/api/export`
  };

  // Initialize charts based on report type
  const reportType = document.getElementById('reportType')?.value;
  
  if (reportType === 'security') {
    initializeSecurityCharts();
  } else if (reportType === 'geographic') {
    initializeGeographicCharts();
  } else if (reportType === 'threats') {
    initializeThreatCharts();
  }
  
  // Event listeners
  document.getElementById('resetFiltersBtn')?.addEventListener('click', function() {
    // Reset date fields
    const fromDateInput = document.getElementById('from');
    const toDateInput = document.getElementById('to');
    
    if (fromDateInput) {
      // Default to 7 days ago
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
      fromDateInput.value = formatDate(sevenDaysAgo);
    }
    
    if (toDateInput) {
      // Default to today
      const today = new Date();
      toDateInput.value = formatDate(today);
    }
    
    // Reset report type to security
    const reportTypeSelect = document.getElementById('reportType');
    if (reportTypeSelect) {
      reportTypeSelect.value = 'security';
    }

    // Inicijaliziraj percentage bars
    initPercentageBars();
    
  });
  
  // Export report
  document.getElementById('exportReportBtn')?.addEventListener('click', function() {
    exportReport();
  });
  
  // Print report
  document.getElementById('printReportBtn')?.addEventListener('click', function() {
    window.print();
  });
  
  // Report type change
  document.getElementById('reportType')?.addEventListener('change', function() {
    // Submit form when report type changes
    document.getElementById('reportForm')?.submit();
  });
  
  /**
   * Export the current report
   */
  function exportReport() {
    // Get report type
    const reportType = document.getElementById('reportType')?.value || 'security';
    
    // Get date range
    const fromDate = document.getElementById('from')?.value || '';
    const toDate = document.getElementById('to')?.value || '';
    
    // Prepare request data
    const requestData = {
      report_type: reportType,
      from_date: fromDate,
      to_date: toDate
    };
    
    // Fetch export from API
    fetch(routes.exportReport, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(requestData)
    })
    .then(response => {
      if (!response.ok) {
        throw new Error('Failed to export report');
      }
      return response.blob();
    })
    .then(blob => {
      // Create download link
      const url = URL.createObjectURL(blob);
      const timestamp = new Date().toISOString().split('T')[0];
      const filename = `security_report_${reportType}_${timestamp}.json`;
      
      const a = document.createElement('a');
      a.style.display = 'none';
      a.href = url;
      a.download = filename;
      
      document.body.appendChild(a);
      a.click();
      
      // Clean up
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    })
    .catch(error => {
      console.error('Error exporting report:', error);
      alert(`Failed to export report: ${error.message}`);
    });
  }
  
  /**
   * Format date as YYYY-MM-DD
   */
  function formatDate(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  }
  
  /**
   * Initialize charts for security report
   */
  function initializeSecurityCharts() {
    fetchAndInitializeChart('attacksChart', 'attacks_by_day', 'line');
    fetchAndInitializeChart('blockingChart', 'blocks_by_day', 'bar');
  }
  
  /**
   * Initialize charts for geographic report
   */
  function initializeGeographicCharts() {
    fetchAndInitializeChart('countriesChart', 'countries', 'bar', true);
  }
  
  /**
   * Initialize charts for threat report
   */
  function initializeThreatCharts() {
    fetchAndInitializeChart('threatTypeChart', 'threat_types', 'doughnut');
    fetchAndInitializeChart('pathsChart', 'paths', 'bar', true);
  }
  
  /**
   * Fetch and initialize a chart dynamically
   * @param {string} chartId - ID of the chart canvas
   * @param {string} dataKey - Key in the report data to use
   * @param {string} chartType - Type of chart to create
   * @param {boolean} [isHorizontal=false] - Whether to use horizontal bar chart
   */
  function fetchAndInitializeChart(chartId, dataKey, chartType, isHorizontal = false) {
    const ctx = document.getElementById(chartId)?.getContext('2d');
    if (!ctx) return;
    
    // Fetch report data from API
    fetch(routes.generateReport, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        report_type: document.getElementById('reportType')?.value || 'security',
        from_date: document.getElementById('from')?.value || '-7 days',
        to_date: document.getElementById('to')?.value || 'now'
      })
    })
    .then(response => response.json())
    .then(data => {
      if (!data.success || !data.report_data || !data.report_data[dataKey]) {
        throw new Error(`No data found for ${dataKey}`);
      }
      
      const reportData = data.report_data[dataKey];
      
      // Prepare chart data
      const chartData = {
        labels: Object.keys(reportData),
        values: Object.values(reportData)
      };
      
      // Create chart configuration based on chart type
      const chartConfig = getChartConfig(chartType, chartData, dataKey, isHorizontal);
      
      // Create chart
      new Chart(ctx, chartConfig);
    })
    .catch(error => {
      console.error(`Error initializing ${chartId}:`, error);
      
      // Show error in chart container
      const container = document.getElementById(`${chartId}Container`);
      if (container) {
        container.innerHTML = `
          <div class="borna-alert danger">
            <i class="fas fa-exclamation-triangle"></i>
            <div>Failed to load chart data: ${error.message}</div>
          </div>
        `;
      }
    });
  }
  
  /**
   * Get chart configuration based on chart type
   */
  function getChartConfig(chartType, chartData, dataKey, isHorizontal = false) {
    const baseConfig = {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          labels: {
            color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
          }
        }
      }
    };
    
    switch (chartType) {
      case 'line':
        return {
          ...baseConfig,
          type: 'line',
          data: {
            labels: chartData.labels,
            datasets: [{
              label: dataKey === 'attacks_by_day' ? 'Attack Attempts' : 'IP Blocks',
              data: chartData.values,
              borderColor: 'rgba(231, 76, 60, 1)',
              backgroundColor: 'rgba(231, 76, 60, 0.1)',
              borderWidth: 2,
              tension: 0.3,
              fill: true
            }]
          },
          options: {
            ...baseConfig.plugins,
            scales: {
              y: {
                beginAtZero: true,
                title: {
                  display: true,
                  text: dataKey === 'attacks_by_day' ? 'Attack Count' : 'Block Count',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              }
            }
          }
        };
      
      case 'bar':
        return {
          ...baseConfig,
          type: isHorizontal ? 'horizontalBar' : 'bar',
          data: {
            labels: chartData.labels,
            datasets: [{
              label: dataKey === 'countries' ? 'Attack Count' : 'Paths',
              data: chartData.values,
              backgroundColor: 'rgba(52, 152, 219, 0.7)',
              borderColor: 'rgba(52, 152, 219, 1)',
              borderWidth: 1,
              borderRadius: 4
            }]
          },
          options: {
            ...baseConfig.plugins,
            indexAxis: isHorizontal ? 'y' : 'x',
            scales: {
              [isHorizontal ? 'x' : 'y']: {
                beginAtZero: true,
                title: {
                  display: true,
                  text: dataKey === 'countries' ? 'Attack Count' : 'Path',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              }
            }
          }
        };
      
      case 'doughnut':
        return {
          ...baseConfig,
          type: 'doughnut',
          data: {
            labels: chartData.labels,
            datasets: [{
              label: 'Threat Types',
              data: chartData.values,
              backgroundColor: [
                'rgba(231, 76, 60, 0.7)',    // Red
                'rgba(52, 152, 219, 0.7)',   // Blue
                'rgba(46, 204, 113, 0.7)',   // Green
                'rgba(155, 89, 182, 0.7)',   // Purple
                'rgba(241, 196, 15, 0.7)',   // Yellow
                'rgba(230, 126, 34, 0.7)',   // Orange
                'rgba(52, 73, 94, 0.7)',     // Dark blue
                'rgba(149, 165, 166, 0.7)'   // Gray
              ]
            }]
          },
          options: {
            ...baseConfig.plugins,
            plugins: {
              ...baseConfig.plugins,
              tooltip: {
                callbacks: {
                  label: function(context) {
                    const label = context.label || '';
                    const value = context.formattedValue;
                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                    const percentage = Math.round((context.raw / total) * 100);
                    return `${label}: ${value} (${percentage}%)`;
                  }
                }
              }
            }
          }
        };
      
      default:
        throw new Error(`Unsupported chart type: ${chartType}`);
    }
  }

  /**
   * Initialize percentage bars with CSS variables instead of inline styles
   */
  function initPercentageBars() {
    const percentageFills = document.querySelectorAll('.percentage-fill');
    percentageFills.forEach(fill => {
      const percent = fill.getAttribute('data-percent');
      fill.style.setProperty('--percent-width', percent + '%');
    });
  }
});