/**
 * BORNA Results Page JavaScript
 * Contains functionality for test results display
 */
document.addEventListener('DOMContentLoaded', function() {
  // Initialize test components like tooltips and accordions
  initTestComponents();
  
  // Add effects to test detail cards
  initTestDetailCards();
  
  // Set up progress bars
  initDataProgressBars();
  
  // Animate test results for visual appeal
  animateTestResults();
  
  // Mobile sidebar toggle
  initMobileSidebar();
  
  // Setup code copy functionality
  setupCodeCopy();
});

/**
 * Initialize basic components
 */
function initTestComponents() {
  // Initialize tooltips if Bootstrap is available
  if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
      return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Initialize popovers
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.map(function (popoverTriggerEl) {
      return new bootstrap.Popover(popoverTriggerEl);
    });
  }
  
  // Initialize accordions manually if Bootstrap is not available
  initAccordions();
}

/**
 * Initialize accordions manually for theme compatibility
 */
function initAccordions() {
  const accordionButtons = document.querySelectorAll('.accordion-button');
  
  accordionButtons.forEach(button => {
    button.addEventListener('click', function() {
      // Toggle class for this button
      this.classList.toggle('collapsed');
      
      // Find the target collapse element
      const target = document.querySelector(this.getAttribute('data-bs-target'));
      if (!target) return;
      
      // Toggle collapse
      if (target.classList.contains('show')) {
        target.classList.remove('show');
        target.style.height = '0';
      } else {
        target.classList.add('show');
        target.style.height = target.scrollHeight + 'px';
      }
    });
  });
}

/**
 * Add effects to test detail cards
 */
function initTestDetailCards() {
  // Add hover effects to test detail cards
  const testDetailCards = document.querySelectorAll('.borna-test-detail');
  testDetailCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-5px)';
      this.style.boxShadow = '0 8px 15px var(--shadow)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
      this.style.boxShadow = 'none';
    });
  });
}

/**
 * Initialize mobile sidebar
 */
function initMobileSidebar() {
  const sidebar = document.querySelector('.borna-sidebar');
  
  // Avoid creating duplicate toggle buttons
  if (!document.querySelector('.borna-mobile-toggle') && sidebar) {
    const mobileToggle = document.createElement('button');
    mobileToggle.className = 'borna-mobile-toggle';
    mobileToggle.innerHTML = '<i class="fas fa-bars"></i>';
    document.body.appendChild(mobileToggle);
    
    // Add click event
    mobileToggle.addEventListener('click', function() {
      sidebar.classList.toggle('show');
      // Change icon based on state
      const icon = this.querySelector('i');
      if (sidebar.classList.contains('show')) {
        icon.className = 'fas fa-times';
      } else {
        icon.className = 'fas fa-bars';
      }
    });
  }
  
  // Close sidebar when clicking outside on mobile
  document.addEventListener('click', function(event) {
    if (!sidebar) return;
    
    const isClickInsideSidebar = sidebar.contains(event.target);
    const isClickOnMobileToggle = event.target.closest('.borna-mobile-toggle');
    
    if (!isClickInsideSidebar && !isClickOnMobileToggle && sidebar.classList.contains('show')) {
      sidebar.classList.remove('show');
      const icon = document.querySelector('.borna-mobile-toggle i');
      if (icon) {
        icon.className = 'fas fa-bars';
      }
    }
  });
}

/**
 * Initialize progress bars with data-progress attribute
 */
function initDataProgressBars() {
  const progressBars = document.querySelectorAll('[data-progress]');
  
  progressBars.forEach(bar => {
    const progressValue = parseFloat(bar.getAttribute('data-progress') || 0);
    bar.style.width = progressValue + '%';
  });
}

/**
 * Animate test results for visual appeal
 */
function animateTestResults() {
  // Animate progress bars 
  const progressBars = document.querySelectorAll('.borna-test-progress-bar');
  progressBars.forEach(bar => {
    // Read from data-progress attribute
    const value = parseFloat(bar.getAttribute('data-progress') || 0);
    bar.style.width = '0%';
    
    setTimeout(() => {
      bar.style.transition = 'width 1s ease-in-out';
      bar.style.width = value + '%';
    }, 300);
  });
  
  // Animate numeric values
  const testScores = document.querySelectorAll('.metric-box-value, .display-4');
  testScores.forEach(score => {
    const text = score.textContent.trim();
    // Check if it contains a number or fraction
    if (/\d/.test(text)) {
      let finalValue;
      
      // Handle fractions like "5 / 10"
      if (text.includes('/')) {
        // Save the format to restore it later
        const parts = text.split('/');
        const num = parseInt(parts[0].trim(), 10);
        // Don't animate if parsing failed
        if (isNaN(num)) return;
        
        finalValue = num;
        const denom = parts[1].trim();
        
        score.setAttribute('data-denominator', denom);
        score.textContent = '0';
      } else {
        finalValue = parseInt(text, 10);
        // Don't animate if parsing failed
        if (isNaN(finalValue)) return;
        
        score.textContent = '0';
      }
      
      let currentValue = 0;
      const duration = 1000; // 1 second
      const stepTime = 20; // 20ms
      const steps = duration / stepTime;
      const increment = finalValue / steps;
      
      const counter = setInterval(() => {
        currentValue += increment;
        
        if (currentValue >= finalValue) {
          clearInterval(counter);
          
          // Check if we need to restore a fraction format
          const denominator = score.getAttribute('data-denominator');
          if (denominator) {
            score.textContent = finalValue + ' / ' + denominator;
          } else {
            score.textContent = Math.round(finalValue);
          }
        } else {
          score.textContent = Math.round(currentValue);
          // Add denominator during animation if needed
          const denominator = score.getAttribute('data-denominator');
          if (denominator) {
            score.textContent += ' / ' + denominator;
          }
        }
      }, stepTime);
    }
  });
}

/**
 * Set up code copy functionality
 */
function setupCodeCopy() {
  const codeBlocks = document.querySelectorAll('pre.code-block');
  
  codeBlocks.forEach(block => {
    // Skip if it already has a copy button
    if (block.querySelector('.copy-button')) return;
    
    // Create copy button
    const copyBtn = document.createElement('button');
    copyBtn.className = 'copy-button';
    copyBtn.innerHTML = '<i class="fas fa-copy"></i>';
    copyBtn.setAttribute('aria-label', 'Kopiraj kod');
    copyBtn.setAttribute('title', 'Kopiraj kod');
    
    // Set relative position on parent if it doesn't have it
    if (getComputedStyle(block).position === 'static') {
      block.style.position = 'relative';
    }
    
    // Add button to block
    block.appendChild(copyBtn);
    
    // Add event listener
    copyBtn.addEventListener('click', () => {
      const code = block.querySelector('code') ? 
        block.querySelector('code').textContent : 
        block.textContent;
      
      // Skip the button text when copying
      const cleanCode = code.replace(copyBtn.textContent, '');
      
      navigator.clipboard.writeText(cleanCode).then(() => {
        copyBtn.innerHTML = '<i class="fas fa-check"></i>';
        setTimeout(() => {
          copyBtn.innerHTML = '<i class="fas fa-copy"></i>';
        }, 2000);
      }).catch(err => {
        console.error('Could not copy code: ', err);
        copyBtn.innerHTML = '<i class="fas fa-times"></i>';
        setTimeout(() => {
          copyBtn.innerHTML = '<i class="fas fa-copy"></i>';
        }, 2000);
      });
    });
  });
}

/**
 * Helper function to load additional test details if needed
 * @param {string} testId ID of the test to load details for
 */
function loadTestDetails(testId) {
  // Try to get app URL from the config
  let publicUrl = '';
  const configEl = document.getElementById('app-config');
  if (configEl) {
    try {
      const config = JSON.parse(configEl.textContent);
      publicUrl = config.paths.app_url || '';
    } catch (e) {
      console.error('Error parsing app config:', e);
    }
  }
  
  const detailsContainer = document.getElementById(`details-${testId}`);
  if (!detailsContainer) return;
  
  // Show loading animation
  detailsContainer.innerHTML = '<div class="text-center p-3"><i class="fas fa-spinner fa-spin"></i> Učitavanje detalja...</div>';
  
  // Fetch details via API
  fetch(`${publicUrl}/borna/security/api/get-results?test=${testId}&format=json`)
    .then(response => {
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then(data => {
      if (data.success) {
        renderTestDetails(detailsContainer, data.results);
      } else {
        detailsContainer.innerHTML = '<div class="borna-alert danger"><i class="fas fa-times-circle"></i><div>Greška pri dohvaćanju detalja testa.</div></div>';
      }
    })
    .catch(error => {
      console.error('Error fetching test details:', error);
      detailsContainer.innerHTML = '<div class="borna-alert danger"><i class="fas fa-times-circle"></i><div>Greška pri dohvaćanju detalja testa: ' + error.message + '</div></div>';
    });
}

/**
 * Render test details in the container
 * @param {HTMLElement} container Element to render details in
 * @param {Object} data Test data
 */
function renderTestDetails(container, data) {
  let html = '';
  
  if (data.tests && data.tests.length) {
    data.tests.forEach((test, index) => {
      const isPassed = test.passed === true;
      html += `
        <div class="borna-test-detail ${isPassed ? 'passed' : 'failed'}">
          <div class="borna-test-detail-header ${isPassed ? 'success' : 'error'}">
            <i class="fas fa-${isPassed ? 'check-circle' : 'times-circle'}"></i>
            Test #${index + 1}: ${test.name || 'Unnamed Test'}
          </div>
          <div class="borna-test-detail-content">
            ${renderTestData(test)}
          </div>
        </div>
      `;
    });
  } else {
    html = '<div class="borna-alert info"><i class="fas fa-info-circle"></i><div>Nema dostupnih detalja za ovaj test.</div></div>';
  }
  
  container.innerHTML = html;
  
  // Setup code copy functionality after rendering
  setupCodeCopy();
}

/**
 * Render individual test data
 * @param {Object} test Test data object
 * @returns {string} HTML markup
 */
function renderTestData(test) {
  let html = '<div class="row">';
  
  // Left column - basic details
  html += '<div class="col-md-6">';
  html += '<h6>Details</h6>';
  html += '<table class="borna-test-detail-table">';
  
  // Add standard fields
  const standardFields = [
    { key: 'payload', label: 'Payload', format: 'code' },
    { key: 'expected_score', label: 'Expected Score', format: 'text' },
    { key: 'detected_score', label: 'Detected Score', format: 'progress' },
    { key: 'execution_time', label: 'Execution Time', format: 'time' },
    { key: 'country', label: 'Country', format: 'text' },
    { key: 'ip', label: 'IP', format: 'text' },
    { key: 'requests', label: 'Requests', format: 'text' },
    { key: 'fingerprint', label: 'Fingerprint', format: 'code' },
    { key: 'result', label: 'Result', format: 'text' }
  ];
  
  standardFields.forEach(field => {
    if (test[field.key] !== undefined) {
      html += '<tr>';
      html += `<th>${field.label}</th>`;
      html += '<td>';
      
      // Format based on field type
      switch (field.format) {
        case 'code':
          html += `<code>${escapeHtml(test[field.key])}</code>`;
          break;
        case 'progress':
          const value = parseFloat(test[field.key]);
          const progressClass = value >= 50 ? 'danger' : (value >= 25 ? 'warning' : 'success');
          html += `
            <div class="borna-test-progress">
              <div class="borna-test-progress-bar ${progressClass}" data-progress="${value}"></div>
            </div>
            <div class="mt-1">${value}</div>
          `;
          break;
        case 'time':
          html += `${test[field.key]} ms`;
          break;
        default:
          html += escapeHtml(test[field.key].toString());
      }
      
      html += '</td>';
      html += '</tr>';
    }
  });
  
  html += '</table>';
  html += '</div>';
  
  // Right column - threat details
  html += '<div class="col-md-6">';
  if (test.details && Object.keys(test.details).length > 0) {
    html += '<h6>Threat Details</h6>';
    
    // Check if details is an array or object with a type property
    if (Array.isArray(test.details)) {
      html += renderThreatDetailsArray(test.details);
    } else if (typeof test.details === 'object') {
      if (test.details.type === 'none') {
        html += `
          <div class="borna-alert info">
            <i class="fas fa-info-circle"></i>
            <div>${escapeHtml(test.details.details || 'No threats detected')}</div>
          </div>
        `;
      } else {
        html += renderThreatDetailsObject(test.details);
      }
    } else if (typeof test.details === 'string') {
      html += `<pre class="code-block"><code>${escapeHtml(test.details)}</code></pre>`;
    }
  } else {
    html += `
      <div class="borna-alert info">
        <i class="fas fa-info-circle"></i>
        <div>No threat details available</div>
      </div>
    `;
  }
  html += '</div>';
  
  html += '</div>'; // Close row
  
  return html;
}

/**
 * Render an array of threat details
 * @param {Array} details Array of threat detail objects
 * @returns {string} HTML markup
 */
function renderThreatDetailsArray(details) {
  let html = '<div class="accordion">';
  
  details.forEach((detail, i) => {
    const detailId = 'detail_' + Math.random().toString(36).substr(2, 9);
    
    html += `
      <div class="accordion-item">
        <h2 class="accordion-header">
          <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#${detailId}" aria-expanded="false" aria-controls="${detailId}">
            ${escapeHtml(detail.type || 'Unknown Threat')}
            ${detail.description ? ' - ' + escapeHtml(detail.description) : ''}
          </button>
        </h2>
        <div id="${detailId}" class="accordion-collapse collapse">
          <div class="accordion-body">
    `;
    
    if (detail.details && Object.keys(detail.details).length > 0) {
      html += `<pre class="code-block"><code>${escapeHtml(JSON.stringify(detail.details, null, 2))}</code></pre>`;
    } else {
      html += '<div class="text-muted">No additional details available</div>';
    }
    
    html += `
          </div>
        </div>
      </div>
    `;
  });
  
  html += '</div>';
  return html;
}

/**
 * Render a single threat detail object
 * @param {Object} details Threat detail object
 * @returns {string} HTML markup
 */
function renderThreatDetailsObject(details) {
  return `<pre class="code-block"><code>${escapeHtml(JSON.stringify(details, null, 2))}</code></pre>`;
}

/**
 * Utility function to escape HTML special characters
 * @param {string} text Text to escape
 * @returns {string} Escaped text
 */
function escapeHtml(text) {
  if (typeof text !== 'string') {
    return '';
  }
  
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}