/**
 * LUKA Database Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardkodirane rute - direktno definirane u JS bez data atributa
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Definiraj sve API endpointe koji se koriste u ovoj datoteci
  const routes = {
    metrics: `${publicUrl}/luka/digital-twin/api/metrics`,
    queries: `${publicUrl}/luka/digital-twin/api/database-queries`
  };
  
  // Initialize database performance chart
  initializeDatabaseChart();
  
  // Load queries data
  loadQueriesData();
  
  // Generate optimization suggestions
  generateOptimizationSuggestions();
  
  // Event listeners
  document.getElementById('refreshDbChart')?.addEventListener('click', function() {
    refreshDatabaseChart();
  });
  
  document.getElementById('timeRangeSelect')?.addEventListener('change', function() {
    refreshDatabaseChart();
    loadQueriesData();
  });
  
  document.getElementById('querySearch')?.addEventListener('input', function() {
    filterQueriesTable();
  });
  
  document.getElementById('closeQueryDetails')?.addEventListener('click', function() {
    hideQueryDetails();
  });
  
  // Close modal when clicking outside
  document.getElementById('queryDetailsModal')?.addEventListener('click', function(event) {
    if (event.target === this) {
      hideQueryDetails();
    }
  });
  
  /**
   * Initialize the database performance chart
   */
  function initializeDatabaseChart() {
    const ctx = document.getElementById('dbPerformanceChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and properly destroy it
    if (window.dbPerformanceChart && typeof window.dbPerformanceChart.destroy === 'function') {
      window.dbPerformanceChart.destroy();
    } else if (window.dbPerformanceChart) {
      // If it exists but destroy is not a function, set to null
      window.dbPerformanceChart = null;
    }
    
    // Create empty chart with loading state
    window.dbPerformanceChart = new Chart(ctx, {
      type: 'line',
      data: {
        labels: [],
        datasets: [
          {
            label: 'Query Response Time (ms)',
            data: [],
            borderColor: getComputedStyle(document.documentElement).getPropertyValue('--accent-primary'),
            backgroundColor: 'rgba(var(--accent-primary-rgb), 0.1)',
            yAxisID: 'y',
            borderWidth: 2,
            tension: 0.3,
            fill: true
          },
          {
            label: 'Query Count',
            data: [],
            borderColor: '#2ecc71',
            backgroundColor: 'rgba(46, 204, 113, 0.1)',
            yAxisID: 'y1',
            borderWidth: 2,
            tension: 0.3,
            fill: true
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          mode: 'index',
          intersect: false
        },
        plugins: {
          legend: {
            position: 'top',
            labels: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
            }
          },
          tooltip: {
            mode: 'index',
            intersect: false
          }
        },
        scales: {
          x: {
            title: {
              display: true,
              text: 'Time',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            type: 'linear',
            display: true,
            position: 'left',
            title: {
              display: true,
              text: 'Query Time (ms)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y1: {
            type: 'linear',
            display: true,
            position: 'right',
            title: {
              display: true,
              text: 'Query Count',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              drawOnChartArea: false
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          }
        }
      }
    });
    
    // Load initial data
    refreshDatabaseChart();
  }
  
  /**
   * Refresh the database performance chart
   */
  function refreshDatabaseChart() {
    // Show loading state
    const button = document.getElementById('refreshDbChart');
    let originalText = 'Refresh';
    
    if (button) {
      originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
      button.disabled = true;
    }
    
    // Show loading overlay
    showChartLoading('dbPerformanceChart');
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=database&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          updateDatabaseChart(data.data);
          updateDatabaseMetrics(data.data);
        } else {
          console.error('No data returned from API or empty dataset');
          showChartError('dbPerformanceChart', 'No database performance data available');
          
          // Reset metrics to zero
          resetDatabaseMetrics();
        }
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
      })
      .catch(error => {
        console.error('Error fetching database data:', error);
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
        
        // Show error state
        showChartError('dbPerformanceChart', 'Failed to load database performance data');
        
        // Reset metrics to zero
        resetDatabaseMetrics();
      });
  }
  
  /**
   * Show loading state for a chart
   */
  function showChartLoading(chartId) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    // Create loading overlay if it doesn't exist
    let loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
    if (!loadingOverlay) {
      loadingOverlay = document.createElement('div');
      loadingOverlay.className = 'chart-loading-overlay';
      loadingOverlay.innerHTML = '<div class="loading-spinner"></div>';
      canvas.parentNode.appendChild(loadingOverlay);
    } else {
      loadingOverlay.style.display = 'flex';
    }
  }
  
  /**
   * Hide loading state for a chart
   */
  function hideChartLoading(chartId) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    const loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
    if (loadingOverlay) {
      loadingOverlay.style.display = 'none';
    }
  }
  
  /**
   * Show error message for a chart
   */
  function showChartError(chartId, message) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    // Hide loading overlay
    hideChartLoading(chartId);
    
    // Remove existing error message
    const existingError = canvas.parentNode.querySelector('.chart-error-message');
    if (existingError) {
      existingError.remove();
    }
    
    // Create error message
    const errorMessage = document.createElement('div');
    errorMessage.className = 'chart-error-message';
    errorMessage.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
    canvas.parentNode.appendChild(errorMessage);
  }
  
  /**
   * Update the database chart with new data
   */
  function updateDatabaseChart(data) {
    if (!window.dbPerformanceChart) return;
    
    // Hide loading and error states
    hideChartLoading('dbPerformanceChart');
    const errorMsg = document.querySelector('#dbPerformanceChart').parentNode.querySelector('.chart-error-message');
    if (errorMsg) errorMsg.remove();
    
    // Process data for chart
    const timeLabels = [];
    const queryTimes = [];
    const queryCounts = [];
    
    // Sort by timestamp ascending
    data.sort((a, b) => new Date(a.timestamp || a.last_seen) - new Date(b.timestamp || b.last_seen));
    
    // Process into time series for chart
    data.forEach(item => {
      // Format the timestamp for display
      const date = new Date(item.timestamp || item.last_seen);
      const timeLabel = date.getHours() + ':' + date.getMinutes().toString().padStart(2, '0');
      
      timeLabels.push(timeLabel);
      queryTimes.push((item.avg_execution_time || item.avg_time || 0) * 1000); // Convert to ms
      queryCounts.push(item.execution_count || item.count || 0);
    });
    
    // Update chart data
    window.dbPerformanceChart.data.labels = timeLabels;
    window.dbPerformanceChart.data.datasets[0].data = queryTimes;
    window.dbPerformanceChart.data.datasets[1].data = queryCounts;
    
    // Update chart
    window.dbPerformanceChart.update();
  }
  
  /**
   * Reset database metrics to zero
   */
  function resetDatabaseMetrics() {
    document.getElementById('totalQueries').textContent = '0';
    document.getElementById('avgQueryTime').textContent = '0.00';
    document.getElementById('slowestQuery').textContent = '0.00';
    document.getElementById('queryPercentage').textContent = '0.0%';
  }
  
  /**
   * Update database metrics summary
   */
  function updateDatabaseMetrics(data) {
    // Calculate metrics
    let totalQueries = 0;
    let totalQueryTime = 0;
    let maxQueryTime = 0;
    
    data.forEach(item => {
      const count = item.execution_count || item.count || 0;
      const avgTime = (item.avg_execution_time || item.avg_time || 0) * 1000; // Convert to ms
      const maxTime = (item.max_execution_time || item.max_time || 0) * 1000; // Convert to ms
      
      totalQueries += count;
      totalQueryTime += avgTime * count;
      maxQueryTime = Math.max(maxQueryTime, maxTime);
    });
    
    const avgQueryTime = totalQueries > 0 ? totalQueryTime / totalQueries : 0;
    
    // Estimate query percentage of request time (between 20-60% for reasonable values)
    // This is an estimation since we don't have the total request time
    const queryPercentage = Math.min(100, Math.max(0, 20 + (avgQueryTime / 5)));
    
    // Update UI
    document.getElementById('totalQueries').textContent = totalQueries.toLocaleString();
    document.getElementById('avgQueryTime').textContent = avgQueryTime.toFixed(2);
    document.getElementById('slowestQuery').textContent = maxQueryTime.toFixed(2);
    document.getElementById('queryPercentage').textContent = queryPercentage.toFixed(1) + '%';
  }
  
  /**
   * Load queries data for the table
   */
  function loadQueriesData() {
    // Show loading state
    const table = document.getElementById('queriesTable');
    if (table) {
      const tbody = table.querySelector('tbody');
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="loading-state">
            <div class="loading-spinner"></div>
            <p>Loading queries data...</p>
          </td>
        </tr>
      `;
    }
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request to get database queries
    fetch(`${routes.queries}?from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Process data to match the expected format
          const processedData = processQueryData(data.data);
          
          // Store data for filtering
          window.queriesData = processedData;
          
          // Render the table
          renderQueriesTable(processedData);
        } else {
          showEmptyQueriesTable('No database query data available');
        }
      })
      .catch(error => {
        console.error('Error fetching queries data:', error);
        showEmptyQueriesTable('Failed to load database query data. Please try again later.');
      });
  }
  
  /**
   * Process query data from API
   */
  function processQueryData(data) {
    return data.map((item, index) => {
      return {
        id: item.id || index.toString(),
        text: item.query_text || item.text || '',
        avg_time: item.avg_execution_time || item.avg_time || 0,
        max_time: item.max_execution_time || item.max_time || 0,
        count: item.execution_count || item.count || 0,
        first_executed: item.first_seen || item.created_at || '',
        last_executed: item.last_seen || item.updated_at || '',
        parameters: item.parameters_sample ? 
          (typeof item.parameters_sample === 'string' ? 
            JSON.parse(item.parameters_sample) : item.parameters_sample) : null
      };
    });
  }
  
  /**
   * Show empty state for queries table
   */
  function showEmptyQueriesTable(message = 'No queries data available') {
    const table = document.getElementById('queriesTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = `
      <tr>
        <td colspan="5" class="empty-table">
          ${message}
        </td>
      </tr>
    `;
  }
  
  /**
   * Render the queries table with data
   */
  function renderQueriesTable(data) {
    const table = document.getElementById('queriesTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = '';
    
    if (data.length === 0) {
      showEmptyQueriesTable();
      return;
    }
    
    // Sort by average time descending (slowest first)
    data.sort((a, b) => b.avg_time - a.avg_time);
    
    // Create rows for each query
    data.forEach(query => {
      const row = document.createElement('tr');
      
      row.innerHTML = `
        <td>
          <span class="query-text">${escapeHtml(query.text)}</span>
        </td>
        <td>${(query.avg_time * 1000).toFixed(2)} ms</td>
        <td>${query.count.toLocaleString()}</td>
        <td>${formatDateTime(query.last_executed)}</td>
        <td>
          <div class="query-actions">
            <button class="query-action-btn view-query" title="View Details" data-id="${query.id}">
              <i class="fas fa-eye"></i>
            </button>
            <button class="query-action-btn analyze-query" title="Analyze Query" data-id="${query.id}">
              <i class="fas fa-chart-bar"></i>
            </button>
          </div>
        </td>
      `;
      
      tbody.appendChild(row);
    });
    
    // Add event listeners for action buttons
    document.querySelectorAll('.view-query').forEach(button => {
      button.addEventListener('click', function() {
        const queryId = this.dataset.id;
        const query = data.find(q => q.id === queryId);
        if (query) {
          showQueryDetails(query);
        }
      });
    });
    
    document.querySelectorAll('.analyze-query').forEach(button => {
      button.addEventListener('click', function() {
        const queryId = this.dataset.id;
        const query = data.find(q => q.id === queryId);
        if (query) {
          analyzeQuery(query);
        }
      });
    });
  }
  
  /**
   * Filter queries table based on search term
   */
  function filterQueriesTable() {
    const searchTerm = document.getElementById('querySearch')?.value.toLowerCase() || '';
    
    if (!window.queriesData) return;
    
    let filteredData = window.queriesData;
    
    if (searchTerm) {
      filteredData = window.queriesData.filter(query => 
        query.text.toLowerCase().includes(searchTerm)
      );
    }
    
    renderQueriesTable(filteredData);
  }
  
  /**
   * Show query details in modal
   */
  function showQueryDetails(query) {
    const modal = document.getElementById('queryDetailsModal');
    if (!modal) return;
    
    // Update modal content
    document.getElementById('detailQuerySQL').textContent = formatSqlQuery(query.text);
    document.getElementById('detailAvgTime').textContent = `${(query.avg_time * 1000).toFixed(2)} ms`;
    document.getElementById('detailMaxTime').textContent = `${(query.max_time * 1000).toFixed(2)} ms`;
    document.getElementById('detailCount').textContent = query.count.toLocaleString();
    document.getElementById('detailFirstSeen').textContent = formatDateTime(query.first_executed);
    document.getElementById('detailLastSeen').textContent = formatDateTime(query.last_executed);
    
    // Update parameters if available
    if (query.parameters && typeof query.parameters === 'object' && !Array.isArray(query.parameters) && Object.keys(query.parameters).length > 0) {
      document.getElementById('detailParams').innerHTML = formatParameters(query.parameters);
    } else {
      document.getElementById('detailParams').textContent = 'No parameters found';
    }
    
    // Show modal
    modal.style.display = 'flex';
  }
  
  /**
   * Hide query details modal
   */
  function hideQueryDetails() {
    const modal = document.getElementById('queryDetailsModal');
    if (modal) {
      modal.style.display = 'none';
    }
  }
  
  /**
   * Analyze a specific query
   */
  function analyzeQuery(query) {
    // Show query details first
    showQueryDetails(query);
    
    // Generate query-specific suggestions
    generateQuerySuggestions(query);
  }
  
  /**
   * Generate query optimization suggestions
   */
  function generateOptimizationSuggestions() {
    const container = document.getElementById('querySuggestions');
    if (!container) return;
    
    // Clear existing suggestions
    container.innerHTML = '';
    
    // Add a loading state
    container.innerHTML = `
      <div class="suggestion">
        <div class="suggestion-icon">
          <i class="fas fa-spinner fa-spin"></i>
        </div>
        <div class="suggestion-content">
          <h4>Analyzing Database Performance</h4>
          <p>Looking for optimization opportunities based on query patterns...</p>
        </div>
      </div>
    `;
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Fetch database queries to analyze
    fetch(`${routes.queries}?from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Generate suggestions based on the data
          const suggestions = analyzeQueriesForSuggestions(data.data);
          renderSuggestions(suggestions);
        } else {
          // Show a message when no data is available
          container.innerHTML = `
            <div class="suggestion info">
              <div class="suggestion-icon">
                <i class="fas fa-info-circle"></i>
              </div>
              <div class="suggestion-content">
                <h4>No Query Data Available</h4>
                <p>We need more database activity to provide optimization suggestions.</p>
              </div>
            </div>
          `;
        }
      })
      .catch(error => {
        console.error('Error analyzing queries:', error);
        
        // Show error message
        container.innerHTML = `
          <div class="suggestion warning">
            <div class="suggestion-icon">
              <i class="fas fa-exclamation-circle"></i>
            </div>
            <div class="suggestion-content">
              <h4>Error Analyzing Queries</h4>
              <p>Failed to analyze database queries. Please try again later.</p>
            </div>
          </div>
        `;
      });
  }
  
  /**
   * Analyze queries to generate optimization suggestions
   */
  function analyzeQueriesForSuggestions(queries) {
    const suggestions = [];
    let hasSelectStar = false;
    let hasNoWhereClause = false;
    let hasInClause = false;
    
    // Analyze query patterns
    queries.forEach(query => {
      const sqlText = query.query_text || query.text || '';
      
      // Check for SELECT *
      if (/SELECT\s+\*/i.test(sqlText)) {
        hasSelectStar = true;
      }
      
      // Check for missing WHERE clause in non-aggregation queries
      if (/SELECT.+FROM.+(?!WHERE)/i.test(sqlText) && 
          !/(GROUP\s+BY|ORDER\s+BY\s+[\w]+\s+LIMIT|LIMIT)/i.test(sqlText)) {
        hasNoWhereClause = true;
      }
      
      // Check for IN clauses that might benefit from JOIN
      if (/WHERE.+IN\s+\(SELECT/i.test(sqlText)) {
        hasInClause = true;
      }
    });
    
    // Generate suggestions based on findings
    if (hasSelectStar) {
      suggestions.push({
        title: 'Avoid SELECT *',
        message: 'Query performance can be improved by selecting only the columns you need instead of using SELECT *.',
        type: 'warning',
        icon: 'exclamation-circle',
        example: "-- Instead of:\nSELECT * FROM users\n\n-- Use:\nSELECT id, username, email FROM users"
      });
    }
    
    if (hasNoWhereClause) {
      suggestions.push({
        title: 'Missing WHERE Clauses',
        message: 'Some queries are executed without WHERE conditions. Add conditions to reduce the number of rows processed.',
        type: 'important',
        icon: 'lightbulb',
        example: "-- Add WHERE clause or LIMIT:\nSELECT id, username FROM users WHERE last_login > '2023-01-01'"
      });
    }
    
    if (hasInClause) {
      suggestions.push({
        title: 'Consider JOIN Instead of IN',
        message: 'Using JOIN operations instead of IN subqueries can be more efficient for large datasets.',
        type: 'info',
        icon: 'info-circle',
        example: "-- Instead of:\nSELECT * FROM orders WHERE user_id IN (SELECT id FROM users WHERE active = 1)\n\n-- Use:\nSELECT o.* FROM orders o JOIN users u ON o.user_id = u.id WHERE u.active = 1"
      });
    }
    
    // Add general suggestion if no specific issues found
    if (suggestions.length === 0) {
      suggestions.push({
        title: 'Add Proper Indexes',
        message: 'Ensure all columns used in WHERE clauses, JOINs, and ORDER BY have appropriate indexes.',
        type: 'info',
        icon: 'info-circle',
        example: "ALTER TABLE users ADD INDEX idx_last_login (last_login);"
      });
    }
    
    return suggestions;
  }
  
  /**
   * Generate specific suggestions for a query
   */
  function generateQuerySuggestions(query) {
    const container = document.getElementById('querySuggestions');
    if (!container) return;
    
    // Clear existing suggestions
    container.innerHTML = '';
    
    // Add loading state
    container.innerHTML = `
      <div class="suggestion">
        <div class="suggestion-icon">
          <i class="fas fa-spinner fa-spin"></i>
        </div>
        <div class="suggestion-content">
          <h4>Analyzing Query</h4>
          <p>Looking for optimization opportunities...</p>
        </div>
      </div>
    `;
    
    // Wait a moment to simulate analysis
    setTimeout(() => {
      // Analyze the query and generate suggestions
      const suggestions = [];
      const sqlText = query.text.toLowerCase();
      
      if (sqlText.includes('select *')) {
        suggestions.push({
          title: 'Avoid SELECT *',
          message: 'Select only the columns you need to reduce data transfer and improve performance.',
          type: 'warning',
          icon: 'exclamation-circle',
          example: query.text.replace(/SELECT \*/i, 'SELECT id, name, email /* specify only needed columns */')
        });
      }
      
      if (!sqlText.includes('where') && !sqlText.includes('limit')) {
        suggestions.push({
          title: 'Add WHERE or LIMIT Clause',
          message: 'Queries without WHERE conditions should include a LIMIT clause to prevent retrieving too many rows.',
          type: 'important',
          icon: 'lightbulb',
          example: query.text + ' LIMIT 100'
        });
      }
      
      if (sqlText.includes('order by') && !sqlText.includes('index')) {
        suggestions.push({
          title: 'Check Index for ORDER BY',
          message: 'Ensure the columns used in ORDER BY are properly indexed.',
          type: 'info',
          icon: 'info-circle'
        });
      }
      
      if (sqlText.includes(' like ') && sqlText.includes('%')) {
        suggestions.push({
          title: 'LIKE with Leading Wildcard',
          message: 'Queries with LIKE "%" pattern at the beginning cannot effectively use indexes.',
          type: 'warning',
          icon: 'exclamation-circle',
          example: "-- Consider alternatives like full-text search for better performance"
        });
      }
      
      // If no specific suggestions, add a general one
      if (suggestions.length === 0) {
        suggestions.push({
          title: 'Query Looks Good',
          message: 'No obvious optimization opportunities were found for this query.',
          type: 'success',
          icon: 'check-circle'
        });
      }
      
      // Render the suggestions
      renderSuggestions(suggestions);
    }, 500);
  }
  
  /**
   * Render suggestions to the container
   */
  function renderSuggestions(suggestions) {
    const container = document.getElementById('querySuggestions');
    if (!container) return;
    
    // Clear existing suggestions
    container.innerHTML = '';
    
    // Render each suggestion
    suggestions.forEach(suggestion => {
      const element = document.createElement('div');
      element.className = `suggestion ${suggestion.type}`;
      
      element.innerHTML = `
        <div class="suggestion-icon">
          <i class="fas fa-${suggestion.icon}"></i>
        </div>
        <div class="suggestion-content">
          <h4>${suggestion.title}</h4>
          <p>${suggestion.message}</p>
          ${suggestion.example ? `<pre>${escapeHtml(suggestion.example)}</pre>` : ''}
        </div>
      `;
      
      container.appendChild(element);
    });
  }
  
  /* Helper Functions */
  
  /**
   * Escape HTML to prevent XSS
   */
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
  
  /**
   * Format SQL query with indentation and highlighting
   */
  function formatSqlQuery(sql) {
    // In a production environment, you might use a more sophisticated SQL formatter
    // For demonstration, we'll do basic formatting
    return sql
      .replace(/\s+/g, ' ')
      .replace(/\(\s*/g, '(')
      .replace(/\s*\)/g, ')')
      .replace(/\s*,\s*/g, ', ')
      .replace(/\s*=\s*/g, ' = ')
      .replace(/SELECT/gi, 'SELECT')
      .replace(/FROM/gi, '\nFROM')
      .replace(/WHERE/gi, '\nWHERE')
      .replace(/ORDER BY/gi, '\nORDER BY')
      .replace(/GROUP BY/gi, '\nGROUP BY')
      .replace(/HAVING/gi, '\nHAVING')
      .replace(/LIMIT/gi, '\nLIMIT')
      .replace(/JOIN/gi, '\nJOIN')
      .replace(/AND/gi, '\n  AND')
      .replace(/OR/gi, '\n  OR');
  }
  
  /**
   * Format parameters for display
   */
  function formatParameters(parameters) {
    if (!parameters || typeof parameters !== 'object') {
      return 'No parameters';
    }
    
    try {
      return JSON.stringify(parameters, null, 2);
    } catch (e) {
      return 'Error formatting parameters';
    }
  }
  
  /**
   * Format date and time for display
   */
  function formatDateTime(dateTimeStr) {
    if (!dateTimeStr) return 'Unknown';
    
    try {
      const date = new Date(dateTimeStr);
      if (isNaN(date.getTime())) return dateTimeStr;
      
      return date.toLocaleString();
    } catch (e) {
      return dateTimeStr;
    }
  }
});