/**
 * LUKA Errors Page JavaScript
 * Handles data fetching and visualization for error analysis
 */
document.addEventListener('DOMContentLoaded', function() {
  
  // Variables for pagination and table data
  window.errorLogData = {
    allErrors: [],
    filteredErrors: [],
    currentPage: 1,
    totalPages: 1,
    itemsPerPage: 10
  };
  
  // Hardkodirane rute - direktno definirane u JS bez data atributa
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Definiraj sve API endpointe koji se koriste u ovoj datoteci
  const routes = {
    metrics: `${publicUrl}/luka/digital-twin/api/metrics`,
    systemLogs: `${publicUrl}/luka/digital-twin/api/system-logs`
  };

  // Initialize charts
  initializeErrorTrendsChart();
  initializeErrorTypesChart();
  initializeErrorPathsChart();
  
  // Load error log data
  loadErrorLogData();
  
  // Event listeners
  document.getElementById('refreshErrorChart')?.addEventListener('click', function() {
    refreshErrorTrendsChart();
  });
  
  document.getElementById('timeRangeSelect')?.addEventListener('change', function() {
    refreshErrorTrendsChart();
    refreshErrorTypesChart();
    refreshErrorPathsChart();
    loadErrorLogData();
  });
  
  document.getElementById('errorTypeSelect')?.addEventListener('change', function() {
    loadErrorLogData();
  });
  
  document.getElementById('errorSearch')?.addEventListener('input', function() {
    filterErrorLogTable();
  });
  
  document.getElementById('closeErrorDetails')?.addEventListener('click', function() {
    hideErrorDetails();
  });
  
  // Close modal when clicking outside
  document.getElementById('errorDetailsModal')?.addEventListener('click', function(event) {
    if (event.target === this) {
      hideErrorDetails();
    }
  });
  
  // Pagination controls
  document.getElementById('prevPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) > 1) {
      navigateToPage(parseInt(this.dataset.page) - 1);
    }
  });
  
  document.getElementById('nextPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) < parseInt(this.dataset.totalPages)) {
      navigateToPage(parseInt(this.dataset.page) + 1);
    }
  });
  
  /**
   * Initialize error trends chart
   */
  function initializeErrorTrendsChart() {
    const ctx = document.getElementById('errorTrendsChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and properly destroy it
    if (window.errorTrendsChart && typeof window.errorTrendsChart.destroy === 'function') {
      window.errorTrendsChart.destroy();
    } else if (window.errorTrendsChart) {
      // If it exists but destroy is not a function, set to null
      window.errorTrendsChart = null;
    }
    
    // Create empty chart
    window.errorTrendsChart = new Chart(ctx, {
      type: 'line',
      data: {
        labels: [],
        datasets: [
          {
            label: 'Error Count',
            data: [],
            borderColor: '#e74c3c',
            backgroundColor: 'rgba(231, 76, 60, 0.1)',
            borderWidth: 2,
            tension: 0.3,
            fill: true
          },
          {
            label: 'Error Rate (%)',
            data: [],
            borderColor: '#f39c12',
            backgroundColor: 'rgba(243, 156, 18, 0.1)',
            borderWidth: 2,
            tension: 0.3,
            fill: true,
            yAxisID: 'y1'
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          mode: 'index',
          intersect: false
        },
        plugins: {
          legend: {
            position: 'top',
            labels: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
            }
          },
          tooltip: {
            mode: 'index',
            intersect: false
          }
        },
        scales: {
          x: {
            title: {
              display: true,
              text: 'Time',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            type: 'linear',
            display: true,
            position: 'left',
            title: {
              display: true,
              text: 'Error Count',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y1: {
            type: 'linear',
            display: true,
            position: 'right',
            title: {
              display: true,
              text: 'Error Rate (%)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            min: 0,
            max: 100,
            grid: {
              drawOnChartArea: false
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          }
        }
      }
    });
    
    // Load initial data
    refreshErrorTrendsChart();
  }
  
  /**
   * Initialize error types chart
   */
  function initializeErrorTypesChart() {
    const ctx = document.getElementById('errorTypesChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and properly destroy it
    if (window.errorTypesChart && typeof window.errorTypesChart.destroy === 'function') {
      window.errorTypesChart.destroy();
    } else if (window.errorTypesChart) {
      // If it exists but destroy is not a function, set to null
      window.errorTypesChart = null;
    }
    
    // Create empty chart
    window.errorTypesChart = new Chart(ctx, {
      type: 'pie',
      data: {
        labels: [],
        datasets: [{
          data: [],
          backgroundColor: [
            '#3498db', // 400 - blue
            '#f39c12', // 401/403 - orange
            '#1abc9c', // 404 - green
            '#e74c3c', // 500 - red
            '#9b59b6'  // other - purple
          ],
          borderColor: getComputedStyle(document.documentElement).getPropertyValue('--bg-secondary'),
          borderWidth: 2
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            position: 'right',
            labels: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary'),
              padding: 15,
              usePointStyle: true,
              pointStyle: 'circle'
            }
          },
          tooltip: {
            callbacks: {
              label: function(context) {
                const label = context.label || '';
                const value = context.formattedValue;
                const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
                const percentage = Math.round((context.raw / total) * 100);
                return `${label}: ${value} (${percentage}%)`;
              }
            }
          }
        }
      }
    });
    
    // Load initial data
    refreshErrorTypesChart();
  }
  
  /**
   * Initialize error paths chart
   */
  function initializeErrorPathsChart() {
    const ctx = document.getElementById('errorPathsChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and properly destroy it
    if (window.errorPathsChart && typeof window.errorPathsChart.destroy === 'function') {
      window.errorPathsChart.destroy();
    } else if (window.errorPathsChart) {
      // If it exists but destroy is not a function, set to null
      window.errorPathsChart = null;
    }
    
    // Create empty chart
    window.errorPathsChart = new Chart(ctx, {
      type: 'bar',
      data: {
        labels: [],
        datasets: [{
          label: 'Error Count',
          data: [],
          backgroundColor: 'rgba(231, 76, 60, 0.7)',
          borderColor: 'rgba(231, 76, 60, 1)',
          borderWidth: 1
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        indexAxis: 'y',
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            callbacks: {
              title: function(tooltipItems) {
                return tooltipItems[0].label;
              }
            }
          }
        },
        scales: {
          x: {
            title: {
              display: true,
              text: 'Error Count',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary'),
              callback: function(value, index, values) {
                const label = this.getLabelForValue(value);
                // Truncate long paths
                return label.length > 25 ? label.substring(0, 22) + '...' : label;
              }
            }
          }
        }
      }
    });
    
    // Load initial data
    refreshErrorPathsChart();
  }
  
  /**
   * Refresh error trends chart
   */
  function refreshErrorTrendsChart() {
    // Show loading state
    const button = document.getElementById('refreshErrorChart');
    let originalText = 'Refresh';
    
    if (button) {
      originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
      button.disabled = true;
    }
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=errors&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => response.json())
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          updateErrorTrendsChart(data.data);
          updateErrorMetrics(data.data);
        } else {
          console.error('No data returned from API or empty data array');
          
          // Show empty state
          if (window.errorTrendsChart) {
            window.errorTrendsChart.data.labels = [];
            window.errorTrendsChart.data.datasets[0].data = [];
            window.errorTrendsChart.data.datasets[1].data = [];
            window.errorTrendsChart.update();
          }
          
          // Reset metrics to zero
          document.getElementById('totalErrors').textContent = '0';
          document.getElementById('errorRate').textContent = '0.0%';
          document.getElementById('mostCommonError').textContent = 'N/A';
          document.getElementById('topErrorPath').textContent = 'N/A';
        }
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
      })
      .catch(error => {
        console.error('Error fetching error trends data:', error);
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
        
        // Show empty state
        if (window.errorTrendsChart) {
          window.errorTrendsChart.data.labels = [];
          window.errorTrendsChart.data.datasets[0].data = [];
          window.errorTrendsChart.data.datasets[1].data = [];
          window.errorTrendsChart.update();
        }
      });
  }
  
  /**
   * Update error trends chart with data
   */
  function updateErrorTrendsChart(data) {
    if (!window.errorTrendsChart) return;
    
    // Process data for trend chart
    const timeLabels = [];
    const errorCounts = [];
    const errorRates = [];
    
    // Group data by time interval
    const groupedData = {};
    
    // Sort by timestamp ascending
    data.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
    
    // Process into hourly time series for chart
    data.forEach(item => {
      // Format date for hourly grouping
      const date = new Date(item.timestamp);
      const hour = date.getHours();
      const hourKey = `${date.getFullYear()}-${date.getMonth() + 1}-${date.getDate()}-${hour}`;
      
      if (!groupedData[hourKey]) {
        groupedData[hourKey] = {
          timestamp: new Date(date.getFullYear(), date.getMonth(), date.getDate(), hour, 0, 0),
          errorCount: 0,
          requestCount: 0
        };
      }
      
      // Assuming we have error_count or count in the data
      groupedData[hourKey].errorCount += item.error_count || item.count || 1;
      
      // Assuming we have request_count in the data for error rate calculation
      groupedData[hourKey].requestCount += item.request_count || 100; // Default if not available
    });
    
    // Convert grouped data to arrays for chart
    Object.values(groupedData).sort((a, b) => a.timestamp - b.timestamp).forEach(hourData => {
      const errorRate = hourData.requestCount > 0 
        ? ((hourData.errorCount / hourData.requestCount) * 100) 
        : 0;
      
      // Format time label (e.g., "14:00")
      const timeLabel = hourData.timestamp.getHours() + ':00';
      
      timeLabels.push(timeLabel);
      errorCounts.push(hourData.errorCount);
      errorRates.push(parseFloat(errorRate.toFixed(2)));
    });
    
    // Update chart data
    window.errorTrendsChart.data.labels = timeLabels;
    window.errorTrendsChart.data.datasets[0].data = errorCounts;
    window.errorTrendsChart.data.datasets[1].data = errorRates;
    
    // Update chart
    window.errorTrendsChart.update();
  }
  
  /**
   * Refresh error types chart
   */
  function refreshErrorTypesChart() {
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=errors&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => response.json())
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Process data for pie chart
          const errorTypesData = processErrorTypesData(data.data);
          updateErrorTypesChart(errorTypesData);
        } else {
          console.error('No data returned from API or empty data array');
          
          // Show empty state
          if (window.errorTypesChart) {
            window.errorTypesChart.data.labels = [];
            window.errorTypesChart.data.datasets[0].data = [];
            window.errorTypesChart.update();
          }
        }
      })
      .catch(error => {
        console.error('Error fetching error types data:', error);
        
        // Show empty state
        if (window.errorTypesChart) {
          window.errorTypesChart.data.labels = [];
          window.errorTypesChart.data.datasets[0].data = [];
          window.errorTypesChart.update();
        }
      });
  }
  
  /**
   * Process data for error types chart
   */
  function processErrorTypesData(data) {
    const errorCodeCounts = {};
    
    // Count occurrences of each status code
    data.forEach(item => {
      const statusCode = item.status_code || 500;
      errorCodeCounts[statusCode] = (errorCodeCounts[statusCode] || 0) + (item.count || 1);
    });
    
    // Group status codes into categories
    const categoryCounts = {
      '400 Bad Request': 0,
      '401/403 Unauthorized': 0,
      '404 Not Found': 0,
      '500 Server Error': 0,
      'Other Errors': 0
    };
    
    Object.entries(errorCodeCounts).forEach(([code, count]) => {
      const numCode = parseInt(code);
      
      if (numCode === 400) {
        categoryCounts['400 Bad Request'] += count;
      } else if (numCode === 401 || numCode === 403) {
        categoryCounts['401/403 Unauthorized'] += count;
      } else if (numCode === 404) {
        categoryCounts['404 Not Found'] += count;
      } else if (numCode >= 500 && numCode < 600) {
        categoryCounts['500 Server Error'] += count;
      } else {
        categoryCounts['Other Errors'] += count;
      }
    });
    
    // Convert to array for chart
    return Object.entries(categoryCounts)
      .filter(([_, count]) => count > 0) // Remove categories with zero count
      .map(([label, count]) => ({ label, count }));
  }
  
  /**
   * Update error types chart with data
   */
  function updateErrorTypesChart(data) {
    if (!window.errorTypesChart) return;
    
    // Extract labels and data from the input
    const labels = data.map(item => item.label);
    const values = data.map(item => item.count);
    
    // Update chart data
    window.errorTypesChart.data.labels = labels;
    window.errorTypesChart.data.datasets[0].data = values;
    
    // Update chart
    window.errorTypesChart.update();
  }
  
  /**
   * Refresh error paths chart
   */
  function refreshErrorPathsChart() {
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=errors&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => response.json())
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Process data for error paths chart
          const errorPathsData = processErrorPathsData(data.data);
          updateErrorPathsChart(errorPathsData);
        } else {
          console.error('No data returned from API or empty data array');
          
          // Show empty state
          if (window.errorPathsChart) {
            window.errorPathsChart.data.labels = [];
            window.errorPathsChart.data.datasets[0].data = [];
            window.errorPathsChart.update();
          }
        }
      })
      .catch(error => {
        console.error('Error fetching error paths data:', error);
        
        // Show empty state
        if (window.errorPathsChart) {
          window.errorPathsChart.data.labels = [];
          window.errorPathsChart.data.datasets[0].data = [];
          window.errorPathsChart.update();
        }
      });
  }
  
  /**
   * Process data for error paths chart
   */
  function processErrorPathsData(data) {
    const pathCounts = {};
    
    // Count occurrences of each path
    data.forEach(item => {
      const path = item.path || '/unknown';
      pathCounts[path] = (pathCounts[path] || 0) + (item.count || 1);
    });
    
    // Convert to array for chart
    return Object.entries(pathCounts)
      .map(([path, count]) => ({ path, count }));
  }
  
  /**
   * Update error paths chart with data
   */
  function updateErrorPathsChart(data) {
    if (!window.errorPathsChart) return;
    
    // Sort data by count (descending)
    data.sort((a, b) => b.count - a.count);
    
    // Take top 10 paths
    const topPaths = data.slice(0, 10);
    
    // Extract labels and data
    const labels = topPaths.map(item => item.path);
    const values = topPaths.map(item => item.count);
    
    // Update chart data
    window.errorPathsChart.data.labels = labels;
    window.errorPathsChart.data.datasets[0].data = values;
    
    // Update chart
    window.errorPathsChart.update();
  }
  
  /**
   * Update error metrics summary
   */
  function updateErrorMetrics(data) {
    let totalErrors = 0;
    let totalRequests = 0;
    
    // Calculate total errors and requests
    data.forEach(item => {
      totalErrors += item.error_count || item.count || 0;
      totalRequests += item.request_count || 0;
    });
    
    // Calculate error rate
    const errorRate = totalRequests > 0 ? (totalErrors / totalRequests) * 100 : 0;
    
    // Find most common error code
    const errorCodeCounts = {};
    data.forEach(item => {
      const statusCode = item.status_code || 500;
      errorCodeCounts[statusCode] = (errorCodeCounts[statusCode] || 0) + (item.count || 1);
    });
    
    let mostCommonError = 'N/A';
    let maxCount = 0;
    
    Object.entries(errorCodeCounts).forEach(([code, count]) => {
      if (count > maxCount) {
        mostCommonError = code;
        maxCount = count;
      }
    });
    
    // Find top error path
    const pathCounts = {};
    data.forEach(item => {
      const path = item.path || '/unknown';
      pathCounts[path] = (pathCounts[path] || 0) + (item.count || 1);
    });
    
    let topErrorPath = 'N/A';
    maxCount = 0;
    
    Object.entries(pathCounts).forEach(([path, count]) => {
      if (count > maxCount) {
        topErrorPath = path;
        maxCount = count;
      }
    });
    
    // Update UI
    document.getElementById('totalErrors').textContent = totalErrors.toLocaleString();
    document.getElementById('errorRate').textContent = errorRate.toFixed(1) + '%';
    document.getElementById('mostCommonError').textContent = mostCommonError;
    
    // Truncate long paths for display
    document.getElementById('topErrorPath').textContent = 
      topErrorPath.length > 20 ? topErrorPath.substring(0, 17) + '...' : topErrorPath;
    
    // Set full path as title for hover
    document.getElementById('topErrorPath').title = topErrorPath;
  }
  
  /**
   * Load error log data for the table
   */
  function loadErrorLogData() {
    // Show loading state
    const table = document.getElementById('errorTable');
    if (table) {
      const tbody = table.querySelector('tbody');
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="loading-state">
            <div class="loading-spinner"></div>
            <p>Loading error data...</p>
          </td>
        </tr>
      `;
    }
    
    // Get time range and error type filter
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    const errorType = document.getElementById('errorTypeSelect')?.value || 'all'; // Default to all errors
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Get error data from system logs
    fetch(`${routes.systemLogs}?type=error&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => response.json())
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Process system logs into error data format
          const errorData = processSystemLogsToErrors(data.data);
          
          // Filter by error type if needed
          let filteredData = errorData;
          if (errorType !== 'all') {
            const statusCode = parseInt(errorType);
            filteredData = errorData.filter(error => error.status_code === statusCode);
          }
          
          // Store data for pagination and filtering
          window.errorLogData.allErrors = filteredData;
          window.errorLogData.filteredErrors = [...filteredData];
          
          // Reset to first page
          window.errorLogData.currentPage = 1;
          
          // Update pagination and render table
          updateErrorPagination();
          renderErrorLogTable();
        } else {
          console.error('No error log data returned from API or empty data array');
          
          // Show empty state
          const tbody = table?.querySelector('tbody');
          if (tbody) {
            tbody.innerHTML = `
              <tr>
                <td colspan="5" class="empty-table">
                  No errors found matching your criteria
                </td>
              </tr>
            `;
          }
          
          // Reset errorLogData
          window.errorLogData.allErrors = [];
          window.errorLogData.filteredErrors = [];
          window.errorLogData.currentPage = 1;
          window.errorLogData.totalPages = 1;
          
          // Update pagination UI
          updateErrorPagination();
        }
      })
      .catch(error => {
        console.error('Error fetching error log data:', error);
        
        // Show error state
        const tbody = table?.querySelector('tbody');
        if (tbody) {
          tbody.innerHTML = `
            <tr>
              <td colspan="5" class="empty-table">
                Error loading data. Please try again.
              </td>
            </tr>
          `;
        }
      });
  }
  
  /**
   * Process system logs into error data format
   */
  function processSystemLogsToErrors(systemLogs) {
    return systemLogs.map((log, index) => {
      // Parse status code from message if possible
      let statusCode = 500; // Default
      const statusMatch = log.message?.match(/status(?:\s+code)?[:\s]+(\d{3})/i);
      if (statusMatch) {
        statusCode = parseInt(statusMatch[1]);
      }
      
      // Parse path from message or context
      let path = '/unknown';
      const pathMatch = log.message?.match(/path[:\s]+(\/[^\s"]+)/i);
      if (pathMatch) {
        path = pathMatch[1];
      } else if (log.context && log.context.url) {
        path = log.context.url;
      }
      
      // Parse method from message or context
      let method = 'GET';
      const methodMatch = log.message?.match(/method[:\s]+([A-Z]+)/i);
      if (methodMatch) {
        method = methodMatch[1].toUpperCase();
      } else if (log.context && log.context.method) {
        method = log.context.method;
      }
      
      return {
        id: `error-${index}`,
        timestamp: log.timestamp,
        path: path,
        status_code: statusCode,
        method: method,
        message: log.message || 'Unknown error',
        user_id: log.context?.user_id || null,
        ip: log.context?.ip || '127.0.0.1',
        stack_trace: log.context?.stack_trace || null,
        context: log.context || null
      };
    });
  }
  
  /**
   * Update pagination information
   */
  function updateErrorPagination() {
    const { filteredErrors, itemsPerPage } = window.errorLogData;
    
    // Calculate total pages
    const totalPages = Math.max(1, Math.ceil(filteredErrors.length / itemsPerPage));
    
    // Update pagination state
    window.errorLogData.totalPages = totalPages;
    window.errorLogData.currentPage = Math.min(window.errorLogData.currentPage, totalPages);
    
    // Update UI
    const prevButton = document.getElementById('prevPage');
    const nextButton = document.getElementById('nextPage');
    const pageInfo = document.getElementById('pageInfo');
    
    if (prevButton) {
      prevButton.disabled = window.errorLogData.currentPage <= 1;
      prevButton.dataset.page = window.errorLogData.currentPage;
    }
    
    if (nextButton) {
      nextButton.disabled = window.errorLogData.currentPage >= totalPages;
      nextButton.dataset.page = window.errorLogData.currentPage;
      nextButton.dataset.totalPages = totalPages;
    }
    
    if (pageInfo) {
      pageInfo.textContent = `Page ${window.errorLogData.currentPage} of ${totalPages}`;
    }
  }
  
  /**
   * Navigate to a specific page
   */
  function navigateToPage(page) {
    window.errorLogData.currentPage = page;
    updateErrorPagination();
    renderErrorLogTable();
  }
  
  /**
   * Render the error log table with current data
   */
  function renderErrorLogTable() {
    const table = document.getElementById('errorTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = '';
    
    const { filteredErrors, currentPage, itemsPerPage } = window.errorLogData;
    
    // Calculate slice for current page
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageData = filteredErrors.slice(startIndex, endIndex);
    
    if (pageData.length === 0) {
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="empty-table">
            No errors found matching your criteria
          </td>
        </tr>
      `;
      return;
    }
    
    // Create rows for each error
    pageData.forEach(error => {
      const row = document.createElement('tr');
      
      // Get status code class
      let statusClass = 'error-500';
      if (error.status_code >= 400 && error.status_code < 500) {
        statusClass = `error-${error.status_code}`;
      }
      
      row.innerHTML = `
        <td>${formatDateTime(error.timestamp)}</td>
        <td title="${error.path}">${truncateText(error.path, 30)}</td>
        <td><span class="error-badge ${statusClass}">${error.status_code}</span></td>
        <td title="${error.message}">${truncateText(error.message, 50)}</td>
        <td>
          <div class="error-actions">
            <button class="error-action-btn view-error" title="View Details" data-id="${error.id}">
              <i class="fas fa-eye"></i>
            </button>
          </div>
        </td>
      `;
      
      tbody.appendChild(row);
    });
    
    // Add event listeners for action buttons
    document.querySelectorAll('.view-error').forEach(button => {
      button.addEventListener('click', function() {
        const errorId = this.dataset.id;
        const error = pageData.find(e => e.id === errorId);
        if (error) {
          showErrorDetails(error);
        }
      });
    });
  }
  
  /**
   * Filter error log table based on search term
   */
  function filterErrorLogTable() {
    const searchTerm = document.getElementById('errorSearch')?.value.toLowerCase() || '';
    
    if (!window.errorLogData) return;
    
    if (!searchTerm) {
      window.errorLogData.filteredErrors = [...window.errorLogData.allErrors];
    } else {
      window.errorLogData.filteredErrors = window.errorLogData.allErrors.filter(error => 
        error.path.toLowerCase().includes(searchTerm) ||
        error.message.toLowerCase().includes(searchTerm) ||
        String(error.status_code).includes(searchTerm)
      );
    }
    
    // Reset to first page
    window.errorLogData.currentPage = 1;
    
    // Update pagination and render table
    updateErrorPagination();
    renderErrorLogTable();
  }
  
  /**
   * Show error details in modal
   */
  function showErrorDetails(error) {
    const modal = document.getElementById('errorDetailsModal');
    if (!modal) return;
    
    // Update modal content
    document.getElementById('detailErrorStatus').textContent = error.status_code;
    document.getElementById('detailErrorTitle').textContent = getErrorTitle(error.status_code);
    document.getElementById('detailErrorPath').textContent = error.path;
    document.getElementById('detailMethod').textContent = error.method || 'GET';
    document.getElementById('detailTime').textContent = formatDateTime(error.timestamp);
    document.getElementById('detailUserId').textContent = error.user_id || 'Anonymous';
    document.getElementById('detailIp').textContent = error.ip || '127.0.0.1';
    document.getElementById('detailMessage').textContent = error.message || 'No error message available';
    
    // Update stack trace
    if (error.stack_trace) {
      document.getElementById('detailStack').textContent = error.stack_trace;
    } else {
      document.getElementById('detailStack').textContent = 'No stack trace available';
    }
    
    // Update context
    if (error.context) {
      document.getElementById('detailContext').textContent = 
        typeof error.context === 'object' ? JSON.stringify(error.context, null, 2) : error.context;
    } else {
      document.getElementById('detailContext').textContent = 'No context available';
    }
    
    // Show modal
    modal.style.display = 'flex';
  }
  
  /**
   * Hide error details modal
   */
  function hideErrorDetails() {
    const modal = document.getElementById('errorDetailsModal');
    if (modal) {
      modal.style.display = 'none';
    }
  }
  
  /* Helper Functions */
  
  /**
   * Format date and time for display
   */
  function formatDateTime(dateTimeStr) {
    if (!dateTimeStr) return 'Unknown';
    
    try {
      const date = new Date(dateTimeStr);
      return date.toLocaleString();
    } catch (e) {
      return dateTimeStr;
    }
  }
  
  /**
   * Truncate text with ellipsis
   */
  function truncateText(text, maxLength) {
    if (!text) return '';
    return text.length > maxLength ? text.substring(0, maxLength - 3) + '...' : text;
  }
  
  /**
   * Get error title based on status code
   */
  function getErrorTitle(statusCode) {
    const errorTitles = {
      400: 'Bad Request',
      401: 'Unauthorized',
      403: 'Forbidden',
      404: 'Not Found',
      500: 'Internal Server Error',
      502: 'Bad Gateway',
      503: 'Service Unavailable',
      504: 'Gateway Timeout'
    };
    
    return errorTitles[statusCode] || 'Unknown Error';
  }
});