/**
 * LUKA Overview Page JavaScript
 * Handles data fetching and chart rendering for the Digital Twin overview page
 */
document.addEventListener('DOMContentLoaded', function() {
    // Initialize variables for charts
    let performanceChart = null;
    let memoryChart = null;
  
    // Hardkodirane rute API-ja
    const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
    
    // Define all routes used in this file
    const routes = {
        performance: `${publicUrl}/luka/digital-twin/api/metrics?type=performance&from=-24%20hours&to=now`,
        memory: `${publicUrl}/luka/digital-twin/api/metrics?type=memory&from=-24%20hours&to=now`,
        errors: `${publicUrl}/luka/digital-twin/api/metrics?type=errors&from=-24%20hours&to=now`,
        state: `${publicUrl}/luka/digital-twin/api/state`
    };
  
    // Check if Chart.js is loaded
    if (typeof Chart === 'undefined') {
        console.error('Chart.js is not loaded. Please include the Chart.js library before this script.');
        return; // Exit early if Chart is not available
    }
  
    // Initialize everything
    initializeCharts();
    setupRefreshButtons();
  
    /**
     * Setup refresh button event listeners
     */
    function setupRefreshButtons() {
        const refreshPerformanceBtn = document.getElementById('refreshPerformanceChart');
        if (refreshPerformanceBtn) {
            refreshPerformanceBtn.addEventListener('click', function() {
                refreshPerformanceChart();
            });
        }
      
        const refreshMemoryBtn = document.getElementById('refreshMemoryChart');
        if (refreshMemoryBtn) {
            refreshMemoryBtn.addEventListener('click', function() {
                refreshMemoryChart();
            });
        }
    }
  
    /**
     * Initialize both charts on page load
     */
    function initializeCharts() {
        initializePerformanceChart();
        initializeMemoryChart();
    }
  
    /**
     * Initialize the performance chart
     */
    async function initializePerformanceChart() {
        const performanceCtx = document.getElementById('performanceChart')?.getContext('2d');
        
        if (!performanceCtx) {
            console.error('Performance chart canvas not found');
            return;
        }
        
        // Check if chart already exists and destroy it properly
        if (performanceChart && typeof performanceChart.destroy === 'function') {
            performanceChart.destroy();
        }
        
        try {
            // Show loading state
            showChartLoading('performanceChart');
            
            // Fetch performance data from API
            const performanceData = await fetchPerformanceData();
            
            // Hide loading state
            hideChartLoading('performanceChart');
            
            // Create the chart
            performanceChart = new Chart(performanceCtx, {
                type: 'line',
                data: {
                    labels: performanceData.labels,
                    datasets: [{
                        label: 'Response Time (ms)',
                        data: performanceData.values,
                        borderColor: getComputedStyle(document.documentElement).getPropertyValue('--accent-primary') || '#3498db',
                        backgroundColor: 'rgba(52, 152, 219, 0.1)',
                        borderWidth: 2,
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            labels: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary') || '#333'
                            }
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Response Time (ms)',
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            },
                            grid: {
                                color: 'rgba(0, 0, 0, 0.1)'
                            },
                            ticks: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Time (24h)',
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            },
                            grid: {
                                color: 'rgba(0, 0, 0, 0.1)'
                            },
                            ticks: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            }
                        }
                    }
                }
            });
        } catch (error) {
            console.error('Error initializing performance chart:', error);
            hideChartLoading('performanceChart');
            showChartError('performanceChart', 'Failed to load performance data');
        }
    }
  
    /**
     * Initialize the memory usage chart
     */
    async function initializeMemoryChart() {
        const memoryCtx = document.getElementById('memoryChart')?.getContext('2d');
        
        if (!memoryCtx) {
            console.error('Memory chart canvas not found');
            return;
        }
        
        // Check if chart already exists and destroy it properly
        if (memoryChart && typeof memoryChart.destroy === 'function') {
            memoryChart.destroy();
        }
        
        try {
            // Show loading state
            showChartLoading('memoryChart');
            
            // Fetch memory data from API
            const memoryData = await fetchMemoryData();
            
            // Hide loading state
            hideChartLoading('memoryChart');
            
            // Create the chart
            memoryChart = new Chart(memoryCtx, {
                type: 'line',
                data: {
                    labels: memoryData.labels,
                    datasets: [{
                        label: 'Memory Usage (MB)',
                        data: memoryData.values,
                        borderColor: '#f39c12',
                        backgroundColor: 'rgba(243, 156, 18, 0.1)',
                        borderWidth: 2,
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            labels: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary') || '#333'
                            }
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Memory Usage (MB)',
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            },
                            grid: {
                                color: 'rgba(0, 0, 0, 0.1)'
                            },
                            ticks: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Time (24h)',
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            },
                            grid: {
                                color: 'rgba(0, 0, 0, 0.1)'
                            },
                            ticks: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary') || '#666'
                            }
                        }
                    }
                }
            });
        } catch (error) {
            console.error('Error initializing memory chart:', error);
            hideChartLoading('memoryChart');
            showChartError('memoryChart', 'Failed to load memory data');
        }
    }
  
    /**
     * Fetch performance data from API
     * @returns {Promise<Object>} Performance data (labels and values)
     */
    async function fetchPerformanceData() {
        try {
            const response = await fetch(routes.performance);
            
            if (!response.ok) {
                throw new Error(`API error: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (!data.success || !data.data || !Array.isArray(data.data)) {
                throw new Error('Invalid API response format');
            }
            
            // Process the API response into chart data format
            const chartData = {
                labels: [],
                values: []
            };
            
            // Sort data by timestamp if needed
            const sortedData = [...data.data].sort((a, b) => {
                const dateA = new Date(a.timestamp || a.hour);
                const dateB = new Date(b.timestamp || b.hour);
                return dateA - dateB;
            });
            
            // Extract data points
            sortedData.forEach(item => {
                // Format date for label (HH:MM)
                const date = new Date(item.timestamp || item.hour);
                const hourLabel = date.getHours().toString().padStart(2, '0') + ':00';
                
                chartData.labels.push(hourLabel);
                
                // Convert seconds to milliseconds for display
                const responseTime = (item.avg_response_time || 0) * 1000;
                chartData.values.push(parseFloat(responseTime.toFixed(2)));
            });
            
            return chartData;
        } catch (error) {
            console.error('Error fetching performance data:', error);
            
            // Return empty data on error
            return {
                labels: [],
                values: []
            };
        }
    }
  
    /**
     * Fetch memory data from API
     * @returns {Promise<Object>} Memory data (labels and values)
     */
    async function fetchMemoryData() {
        try {
            const response = await fetch(routes.memory);
            
            if (!response.ok) {
                throw new Error(`API error: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (!data.success || !data.data || !Array.isArray(data.data)) {
                throw new Error('Invalid API response format');
            }
            
            // Process the API response into chart data format
            const chartData = {
                labels: [],
                values: []
            };
            
            // Sort data by timestamp if needed
            const sortedData = [...data.data].sort((a, b) => {
                const dateA = new Date(a.timestamp || a.hour);
                const dateB = new Date(b.timestamp || b.hour);
                return dateA - dateB;
            });
            
            // Extract data points
            sortedData.forEach(item => {
                // Format date for label (HH:MM)
                const date = new Date(item.timestamp || item.hour);
                const hourLabel = date.getHours().toString().padStart(2, '0') + ':00';
                
                chartData.labels.push(hourLabel);
                
                // Convert bytes to MB for display
                const memoryMB = ((item.avg_memory || 0) / (1024 * 1024)).toFixed(2);
                chartData.values.push(parseFloat(memoryMB));
            });
            
            return chartData;
        } catch (error) {
            console.error('Error fetching memory data:', error);
            
            // Return empty data on error
            return {
                labels: [],
                values: []
            };
        }
    }
  
    /**
     * Refresh the performance chart
     */
    async function refreshPerformanceChart() {
        // Show loading state in the button
        const button = document.getElementById('refreshPerformanceChart');
        if (button) {
            const originalText = button.innerHTML;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
            button.disabled = true;
            
            try {
                // Fetch new data
                const chartData = await fetchPerformanceData();
                
                // Update chart with new data
                if (performanceChart && typeof performanceChart.update === 'function') {
                    performanceChart.data.labels = chartData.labels;
                    performanceChart.data.datasets[0].data = chartData.values;
                    performanceChart.update();
                } else {
                    // Initialize chart if it doesn't exist
                    await initializePerformanceChart();
                }
                
                // Refresh the recent errors table
                refreshRecentErrorsTable();
                
            } catch (error) {
                console.error('Error refreshing performance chart:', error);
                showToast('Failed to refresh performance data', 'error');
            } finally {
                // Reset button state
                button.innerHTML = originalText;
                button.disabled = false;
            }
        }
    }
  
    /**
     * Refresh the memory chart
     */
    async function refreshMemoryChart() {
        // Show loading state in the button
        const button = document.getElementById('refreshMemoryChart');
        if (button) {
            const originalText = button.innerHTML;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
            button.disabled = true;
            
            try {
                // Fetch new data
                const chartData = await fetchMemoryData();
                
                // Update chart with new data
                if (memoryChart && typeof memoryChart.update === 'function') {
                    memoryChart.data.labels = chartData.labels;
                    memoryChart.data.datasets[0].data = chartData.values;
                    memoryChart.update();
                } else {
                    // Initialize chart if it doesn't exist
                    await initializeMemoryChart();
                }
                
            } catch (error) {
                console.error('Error refreshing memory chart:', error);
                showToast('Failed to refresh memory data', 'error');
            } finally {
                // Reset button state
                button.innerHTML = originalText;
                button.disabled = false;
            }
        }
    }
    
    /**
     * Refresh the recent errors table with latest data
     */
    async function refreshRecentErrorsTable() {
        const table = document.getElementById('recentErrorsTable');
        if (!table) return;
        
        try {
            const response = await fetch(routes.errors);
            
            if (!response.ok) {
                throw new Error(`API error: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (!data.success || !data.data) {
                throw new Error('Invalid API response format');
            }
            
            // Get table body
            const tbody = table.querySelector('tbody');
            if (!tbody) return;
            
            // Clear existing rows
            tbody.innerHTML = '';
            
            // Add new rows
            if (data.data.length === 0) {
                const emptyRow = document.createElement('tr');
                emptyRow.innerHTML = '<td colspan="3" class="empty-table">No recent errors</td>';
                tbody.appendChild(emptyRow);
            } else {
                // Only show first 5 errors
                const errors = data.data.slice(0, 5);
                
                errors.forEach(error => {
                    const row = document.createElement('tr');
                    
                    // Format date
                    const date = new Date(error.timestamp);
                    const timeStr = date.getHours().toString().padStart(2, '0') + ':' +
                                   date.getMinutes().toString().padStart(2, '0') + ':' +
                                   date.getSeconds().toString().padStart(2, '0');
                    
                    row.innerHTML = `
                        <td>${escapeHtml(error.path)}</td>
                        <td><span class="error-badge">${error.status_code}</span></td>
                        <td>${timeStr}</td>
                    `;
                    
                    tbody.appendChild(row);
                });
            }
            
        } catch (error) {
            console.error('Error refreshing errors table:', error);
        }
    }
    
    /**
     * Escape HTML to prevent XSS
     * @param {string} html - The string to escape
     * @returns {string} - The escaped string
     */
    function escapeHtml(html) {
        const div = document.createElement('div');
        div.textContent = html;
        return div.innerHTML;
    }
    
    /**
     * Show loading state for a chart
     * @param {string} chartId - Chart canvas ID
     */
    function showChartLoading(chartId) {
        const canvas = document.getElementById(chartId);
        if (!canvas) return;
        
        // Create loading overlay if it doesn't exist
        let loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
        if (!loadingOverlay) {
            loadingOverlay = document.createElement('div');
            loadingOverlay.className = 'chart-loading-overlay';
            loadingOverlay.innerHTML = '<div class="loading-spinner"></div>';
            canvas.parentNode.appendChild(loadingOverlay);
            
            // Add CSS if not already present
            if (!document.getElementById('chart-loading-css')) {
                const style = document.createElement('style');
                style.id = 'chart-loading-css';
                style.textContent = `
                    .chart-container { position: relative; }
                    .chart-loading-overlay {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        background: rgba(255, 255, 255, 0.7);
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        z-index: 10;
                    }
                    .loading-spinner {
                        border: 5px solid rgba(0, 0, 0, 0.1);
                        border-radius: 50%;
                        border-top: 5px solid #3498db;
                        width: 30px;
                        height: 30px;
                        animation: spin 1s linear infinite;
                    }
                    @keyframes spin {
                        0% { transform: rotate(0deg); }
                        100% { transform: rotate(360deg); }
                    }
                    .chart-error-message {
                        position: absolute;
                        top: 50%;
                        left: 50%;
                        transform: translate(-50%, -50%);
                        color: #e74c3c;
                        text-align: center;
                        z-index: 5;
                    }
                `;
                document.head.appendChild(style);
            }
        } else {
            loadingOverlay.style.display = 'flex';
        }
    }
    
    /**
     * Hide loading state for a chart
     * @param {string} chartId - Chart canvas ID
     */
    function hideChartLoading(chartId) {
        const canvas = document.getElementById(chartId);
        if (!canvas) return;
        
        const loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
        if (loadingOverlay) {
            loadingOverlay.style.display = 'none';
        }
    }
    
    /**
     * Show error message for a chart
     * @param {string} chartId - Chart canvas ID
     * @param {string} message - Error message
     */
    function showChartError(chartId, message) {
        const canvas = document.getElementById(chartId);
        if (!canvas) return;
        
        // Remove existing error message
        const existingError = canvas.parentNode.querySelector('.chart-error-message');
        if (existingError) {
            existingError.remove();
        }
        
        // Create error message
        const errorMessage = document.createElement('div');
        errorMessage.className = 'chart-error-message';
        errorMessage.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
        canvas.parentNode.appendChild(errorMessage);
    }
    
    /**
     * Show toast notification
     * @param {string} message - Toast message
     * @param {string} type - Toast type (success, error, info)
     */
    function showToast(message, type = 'info') {
        // Create toast container if it doesn't exist
        let toastContainer = document.getElementById('toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.id = 'toast-container';
            document.body.appendChild(toastContainer);
            
            // Add CSS if not already present
            if (!document.getElementById('toast-css')) {
                const style = document.createElement('style');
                style.id = 'toast-css';
                style.textContent = `
                    #toast-container {
                        position: fixed;
                        bottom: 20px;
                        right: 20px;
                        z-index: 9999;
                    }
                    .toast {
                        min-width: 250px;
                        margin-top: 10px;
                        padding: 15px;
                        border-radius: 4px;
                        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
                        color: white;
                        display: flex;
                        align-items: center;
                        animation: toast-fadein 0.5s, toast-fadeout 0.5s 2.5s;
                        animation-fill-mode: forwards;
                        opacity: 0;
                    }
                    .toast.success { background-color: #2ecc71; }
                    .toast.error { background-color: #e74c3c; }
                    .toast.info { background-color: #3498db; }
                    .toast i { margin-right: 10px; }
                    @keyframes toast-fadein {
                        from { opacity: 0; transform: translateY(20px); }
                        to { opacity: 1; transform: translateY(0); }
                    }
                    @keyframes toast-fadeout {
                        from { opacity: 1; transform: translateY(0); }
                        to { opacity: 0; transform: translateY(-20px); }
                    }
                `;
                document.head.appendChild(style);
            }
        }
        
        // Create toast
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        
        // Set icon based on type
        let icon = 'info-circle';
        if (type === 'success') icon = 'check-circle';
        if (type === 'error') icon = 'exclamation-circle';
        
        toast.innerHTML = `<i class="fas fa-${icon}"></i> ${message}`;
        toastContainer.appendChild(toast);
        
        // Remove toast after animation
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
  });