/**
 * LUKA Requests Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Check if Chart.js is loaded
  if (typeof Chart === 'undefined') {
      console.error('Chart.js is not loaded. Please include the Chart.js library before this script.');
      return; // Exit early if Chart is not available
  }

  // Hardkodirane rute API-ja
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    metrics: `${publicUrl}/luka/digital-twin/api/metrics`
  };
  
  // Initialize requests chart
  initializeRequestsChart();
  
  // Load routes data
  loadRoutesData();
  
  // Event listeners
  document.getElementById('refreshRequestChart')?.addEventListener('click', function() {
    refreshRequestsChart();
  });
  
  document.getElementById('timeRangeSelect')?.addEventListener('change', function() {
    refreshRequestsChart();
    loadRoutesData();
  });
  
  document.getElementById('routeSearch')?.addEventListener('input', function() {
    filterRoutes();
  });
  
  document.getElementById('exportRoutesData')?.addEventListener('click', function() {
    exportRoutesData();
  });
  
  document.getElementById('closeRouteDetails')?.addEventListener('click', function() {
    hideRouteDetails();
  });
  
  // Pagination controls
  document.getElementById('prevPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) > 1) {
      navigateToPage(parseInt(this.dataset.page) - 1);
    }
  });
  
  document.getElementById('nextPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) < parseInt(this.dataset.totalPages)) {
      navigateToPage(parseInt(this.dataset.page) + 1);
    }
  });
  
  // Table sorting
  document.querySelectorAll('.luka-table th.sortable').forEach(header => {
    header.addEventListener('click', function() {
      sortTable(this.dataset.sort);
    });
  });
  
  // Variables for pagination and table data
  window.routesData = {
    allRoutes: [],
    filteredRoutes: [],
    currentPage: 1,
    totalPages: 1,
    itemsPerPage: 10,
    sortField: 'count',
    sortDirection: 'desc'
  };
  
  /**
   * Initialize the requests chart
   */
  function initializeRequestsChart() {
    const ctx = document.getElementById('requestsChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and destroy it properly
    if (window.requestsChart && typeof window.requestsChart.destroy === 'function') {
      window.requestsChart.destroy();
    }
    
    // Create empty chart with loading state
    window.requestsChart = new Chart(ctx, {
      type: 'bar',
      data: {
        labels: [],
        datasets: [
          {
            label: 'Successful Requests',
            data: [],
            backgroundColor: 'rgba(46, 204, 113, 0.7)',
            borderColor: 'rgba(46, 204, 113, 1)',
            borderWidth: 1
          },
          {
            label: 'Failed Requests',
            data: [],
            backgroundColor: 'rgba(231, 76, 60, 0.7)',
            borderColor: 'rgba(231, 76, 60, 1)',
            borderWidth: 1
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          x: {
            stacked: true,
            title: {
              display: true,
              text: 'Time Period',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            stacked: true,
            title: {
              display: true,
              text: 'Number of Requests',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          }
        },
        plugins: {
          legend: {
            position: 'top',
            labels: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
            }
          },
          tooltip: {
            mode: 'index',
            intersect: false
          }
        }
      }
    });
    
    // Load initial data
    refreshRequestsChart();
  }
  
  /**
   * Refresh the requests chart
   */
  function refreshRequestsChart() {
    // Show loading state
    const button = document.getElementById('refreshRequestChart');
    let originalText = 'Refresh';
    
    if (button) {
      originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
      button.disabled = true;
    }
    
    // Show loading state on chart
    showChartLoading('requestsChart');
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=requests&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          updateRequestsChart(data.data);
        } else {
          console.error('No data returned from API');
          showChartError('requestsChart', 'No request data available');
        }
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
      })
      .catch(error => {
        console.error('Error fetching requests data:', error);
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
        
        // Show error state
        showChartError('requestsChart', 'Failed to load request data');
      });
  }
  
  /**
   * Show loading state for a chart
   */
  function showChartLoading(chartId) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    // Create loading overlay if it doesn't exist
    let loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
    if (!loadingOverlay) {
      loadingOverlay = document.createElement('div');
      loadingOverlay.className = 'chart-loading-overlay';
      loadingOverlay.innerHTML = '<div class="loading-spinner"></div>';
      canvas.parentNode.appendChild(loadingOverlay);
    } else {
      loadingOverlay.style.display = 'flex';
    }
  }
  
  /**
   * Hide loading state for a chart
   */
  function hideChartLoading(chartId) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    const loadingOverlay = canvas.parentNode.querySelector('.chart-loading-overlay');
    if (loadingOverlay) {
      loadingOverlay.style.display = 'none';
    }
  }
  
  /**
   * Show error message for a chart
   */
  function showChartError(chartId, message) {
    const canvas = document.getElementById(chartId);
    if (!canvas) return;
    
    // Hide loading overlay
    hideChartLoading(chartId);
    
    // Remove existing error message
    const existingError = canvas.parentNode.querySelector('.chart-error-message');
    if (existingError) {
      existingError.remove();
    }
    
    // Create error message
    const errorMessage = document.createElement('div');
    errorMessage.className = 'chart-error-message';
    errorMessage.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
    canvas.parentNode.appendChild(errorMessage);
  }
  
  /**
   * Update the requests chart with new data
   */
  function updateRequestsChart(data) {
    if (!window.requestsChart || typeof window.requestsChart.update !== 'function') return;
    
    // Hide loading and error states
    hideChartLoading('requestsChart');
    const errorMsg = document.querySelector('#requestsChart').parentNode.querySelector('.chart-error-message');
    if (errorMsg) errorMsg.remove();
    
    // Process data for chart
    const labels = [];
    const successData = [];
    const failureData = [];
    
    // Group by time interval based on data length
    const groupedData = {};
    data.forEach(item => {
      // Use path as the group key for this chart
      const key = item.path || 'Unknown';
      
      if (!groupedData[key]) {
        groupedData[key] = {
          success: 0,
          failure: 0,
          total: 0
        };
      }
      
      const successCount = item.status_code && item.status_code < 400 ? item.count : 0;
      const failureCount = item.status_code && item.status_code >= 400 ? item.count : 0;
      
      groupedData[key].success += successCount;
      groupedData[key].failure += failureCount;
      groupedData[key].total += item.count;
    });
    
    // Convert to arrays for chart
    // Take top 10 paths by total request count
    const sortedPaths = Object.keys(groupedData).sort((a, b) => 
      groupedData[b].total - groupedData[a].total
    ).slice(0, 10);
    
    // Create chart data
    sortedPaths.forEach(path => {
      // Truncate long paths
      const displayPath = path.length > 30 ? path.substring(0, 27) + '...' : path;
      labels.push(displayPath);
      successData.push(groupedData[path].success);
      failureData.push(groupedData[path].failure);
    });
    
    // Update chart data
    window.requestsChart.data.labels = labels;
    window.requestsChart.data.datasets[0].data = successData;
    window.requestsChart.data.datasets[1].data = failureData;
    
    // Update chart
    window.requestsChart.update();
  }
  
  /**
   * Load routes data for the table
   */
  function loadRoutesData() {
    // Show loading state
    const table = document.getElementById('routesTable');
    if (table) {
      const tbody = table.querySelector('tbody');
      tbody.innerHTML = `
        <tr>
          <td colspan="6" class="loading-state">
            <div class="loading-spinner"></div>
            <p>Loading routes data...</p>
          </td>
        </tr>
      `;
    }
    
    // Get time range
    const timeRange = document.getElementById('timeRangeSelect')?.value || 24; // Default to 24 hours
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=requests&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          window.routesData.allRoutes = processRoutesData(data.data);
          window.routesData.filteredRoutes = [...window.routesData.allRoutes];
          updateRoutesPagination();
          renderRoutesTable();
        } else {
          showEmptyRoutesState('No request data available');
        }
      })
      .catch(error => {
        console.error('Error fetching routes data:', error);
        showEmptyRoutesState('Failed to load request data. Please try again later.');
      });
  }
  
  /**
   * Process routes data from API
   */
  function processRoutesData(data) {
    // Group data by path and method to calculate success rates
    const groupedByPathMethod = {};
    
    data.forEach(item => {
      const key = `${item.path}|${item.method}`;
      
      if (!groupedByPathMethod[key]) {
        groupedByPathMethod[key] = {
          path: item.path,
          method: item.method,
          count: 0,
          success_count: 0,
          error_count: 0,
          avg_time: 0,
          max_time: 0,
          min_time: 0,
          total_time: 0
        };
      }
      
      const entry = groupedByPathMethod[key];
      entry.count += item.count || 0;
      
      // Calculate success vs error counts
      if (item.status_code && item.status_code < 400) {
        entry.success_count += item.count || 0;
      } else {
        entry.error_count += item.count || 0;
      }
      
      // Track timing information
      entry.total_time += (item.avg_time || 0) * (item.count || 0);
      entry.max_time = Math.max(entry.max_time, item.max_time || 0);
      entry.min_time = Math.min(entry.min_time || Number.MAX_VALUE, item.min_time || Number.MAX_VALUE);
    });
    
    // Calculate averages and convert to array
    return Object.values(groupedByPathMethod).map(entry => ({
      ...entry,
      avg_time: entry.count > 0 ? entry.total_time / entry.count : 0,
      min_time: entry.min_time === Number.MAX_VALUE ? 0 : entry.min_time,
      success_rate: entry.count > 0 ? (entry.success_count / entry.count) * 100 : 100
    }));
  }
  
  /**
   * Update pagination information
   */
  function updateRoutesPagination() {
    const { filteredRoutes, itemsPerPage } = window.routesData;
    
    // Calculate total pages
    const totalPages = Math.max(1, Math.ceil(filteredRoutes.length / itemsPerPage));
    
    // Update pagination state
    window.routesData.totalPages = totalPages;
    window.routesData.currentPage = Math.min(window.routesData.currentPage, totalPages);
    
    // Update UI
    const prevButton = document.getElementById('prevPage');
    const nextButton = document.getElementById('nextPage');
    const pageInfo = document.getElementById('pageInfo');
    
    if (prevButton) {
      prevButton.disabled = window.routesData.currentPage <= 1;
      prevButton.dataset.page = window.routesData.currentPage;
    }
    
    if (nextButton) {
      nextButton.disabled = window.routesData.currentPage >= totalPages;
      nextButton.dataset.page = window.routesData.currentPage;
      nextButton.dataset.totalPages = totalPages;
    }
    
    if (pageInfo) {
      pageInfo.textContent = `Page ${window.routesData.currentPage} of ${totalPages}`;
    }
  }
  
  /**
   * Navigate to a specific page
   */
  function navigateToPage(page) {
    window.routesData.currentPage = page;
    updateRoutesPagination();
    renderRoutesTable();
  }
  
  /**
   * Render the routes table with current data
   */
  function renderRoutesTable() {
    const table = document.getElementById('routesTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = '';
    
    const { filteredRoutes, currentPage, itemsPerPage } = window.routesData;
    
    // Calculate slice for current page
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageData = filteredRoutes.slice(startIndex, endIndex);
    
    if (pageData.length === 0) {
      tbody.innerHTML = `
        <tr>
          <td colspan="6" class="empty-table">
            No routes data available
          </td>
        </tr>
      `;
      return;
    }
    
    // Create rows for each item
    pageData.forEach(route => {
      const row = document.createElement('tr');
      row.dataset.path = route.path;
      row.dataset.method = route.method;
      
      // Add click event to show details
      row.addEventListener('click', function() {
        showRouteDetails(route);
      });
      
      // Success rate color
      const successRateClass = route.success_rate >= 98 
        ? '' 
        : (route.success_rate >= 90 ? 'warning' : 'danger');
      
      row.innerHTML = `
        <td>${route.path}</td>
        <td><span class="method-badge ${route.method.toLowerCase()}">${route.method}</span></td>
        <td>${route.count.toLocaleString()}</td>
        <td>${(route.avg_time * 1000).toFixed(2)} ms</td>
        <td>${(route.max_time * 1000).toFixed(2)} ms</td>
        <td>
          <div class="success-rate">
            <div class="success-bar">
              <div class="success-bar-value ${successRateClass}" data-progress="${route.success_rate}"></div>
            </div>
            <div class="success-text">${route.success_rate.toFixed(1)}%</div>
          </div>
        </td>
      `;
      
      tbody.appendChild(row);
    });

    updateSuccessRateBars();
  }

  /**
   * Update progress bars with CSS variables based on data-progress attributes
   */
  function updateSuccessRateBars() {
    const progressBars = document.querySelectorAll('.success-bar-value[data-progress]');
    console.log('Found success rate bars:', progressBars.length);
    
    progressBars.forEach(bar => {
      const progressValue = parseFloat(bar.getAttribute('data-progress') || 0);
      bar.style.setProperty('--progress-width', progressValue + '%');
    });
  }
  
  /**
   * Show empty state for routes table
   */
  function showEmptyRoutesState(message = 'No routes data available') {
    const table = document.getElementById('routesTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = `
      <tr>
        <td colspan="6" class="empty-table">
          ${message}
        </td>
      </tr>
    `;
    
    // Disable pagination
    const prevButton = document.getElementById('prevPage');
    const nextButton = document.getElementById('nextPage');
    
    if (prevButton) prevButton.disabled = true;
    if (nextButton) nextButton.disabled = true;
  }
  
  /**
   * Filter routes based on search query
   */
  function filterRoutes() {
    const searchQuery = document.getElementById('routeSearch')?.value.toLowerCase() || '';
    
    if (!searchQuery) {
      window.routesData.filteredRoutes = [...window.routesData.allRoutes];
    } else {
      window.routesData.filteredRoutes = window.routesData.allRoutes.filter(route => 
        route.path.toLowerCase().includes(searchQuery) ||
        route.method.toLowerCase().includes(searchQuery)
      );
    }
    
    // Reset pagination
    window.routesData.currentPage = 1;
    
    // Update UI
    updateRoutesPagination();
    renderRoutesTable();
  }
  
  /**
   * Sort table by column
   */
  function sortTable(field) {
    // Get current sort state
    const { sortField, sortDirection } = window.routesData;
    
    // Determine new sort direction
    const newDirection = field === sortField && sortDirection === 'asc' ? 'desc' : 'asc';
    
    // Update sort state
    window.routesData.sortField = field;
    window.routesData.sortDirection = newDirection;
    
    // Sort data
    window.routesData.filteredRoutes.sort((a, b) => {
      let result = 0;
      
      // Handle different field types
      if (field === 'path' || field === 'method') {
        result = a[field].localeCompare(b[field]);
      } else {
        result = a[field] - b[field];
      }
      
      // Apply direction
      return newDirection === 'asc' ? result : -result;
    });
    
    // Update UI
    updateSortIndicators(field, newDirection);
    renderRoutesTable();
  }
  
  /**
   * Update sort indicators in the table headers
   */
  function updateSortIndicators(field, direction) {
    const headers = document.querySelectorAll('.luka-table th.sortable');
    
    headers.forEach(header => {
      header.classList.remove('sort-asc', 'sort-desc');
      
      if (header.dataset.sort === field) {
        header.classList.add(`sort-${direction}`);
      }
    });
  }
  
  /**
   * Show route details
   */
  function showRouteDetails(route) {
    const detailsPanel = document.getElementById('routeDetailsPanel');
    if (!detailsPanel) return;
    
    // Update title
    document.getElementById('routeDetailsTitle').textContent = `${route.method} ${route.path}`;
    
    // Update details
    document.getElementById('detailPath').textContent = route.path;
    document.getElementById('detailMethod').textContent = route.method;
    document.getElementById('detailCount').textContent = route.count.toLocaleString();
    document.getElementById('detailAvgTime').textContent = `${(route.avg_time * 1000).toFixed(2)} ms`;
    document.getElementById('detailMaxTime').textContent = `${(route.max_time * 1000).toFixed(2)} ms`;
    document.getElementById('detailMinTime').textContent = `${(route.min_time * 1000).toFixed(2)} ms`;
    
    // Show panel
    detailsPanel.style.display = 'block';
    
    // Initialize chart
    initializeRouteDetailChart(route);
    
    // Highlight selected row
    const tableRows = document.querySelectorAll('.luka-table tbody tr');
    tableRows.forEach(row => row.classList.remove('selected'));
    
    const selectedRow = Array.from(tableRows).find(
      row => row.dataset.path === route.path && row.dataset.method === route.method
    );
    
    if (selectedRow) {
      selectedRow.classList.add('selected');
      selectedRow.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }

    updateSuccessRateBars();
  }
  
  /**
   * Hide route details
   */
  function hideRouteDetails() {
    const detailsPanel = document.getElementById('routeDetailsPanel');
    if (detailsPanel) {
      detailsPanel.style.display = 'none';
    }
    
    // Remove selected class from rows
    const tableRows = document.querySelectorAll('.luka-table tbody tr');
    tableRows.forEach(row => row.classList.remove('selected'));
  }
  
  /**
   * Initialize route detail chart
   */
  function initializeRouteDetailChart(route) {
    const ctx = document.getElementById('routeDetailChart')?.getContext('2d');
    if (!ctx) return;
    
    // Destroy existing chart if present
    if (window.routeDetailChart && typeof window.routeDetailChart.destroy === 'function') {
      window.routeDetailChart.destroy();
    }
    
    // Create bins for response time distribution
    const bins = 8;
    const range = route.max_time - route.min_time;
    // Ensure there's always some range to prevent division by zero
    const binSize = range > 0 ? range / bins : 1;
    
    const labels = [];
    // We'll create a simple normal-like distribution around the average time
    // This is an approximation since we don't have actual distribution data
    const data = [];
    
    const avgMs = route.avg_time * 1000;
    const minMs = route.min_time * 1000;
    const maxMs = route.max_time * 1000;
    
    for (let i = 0; i < bins; i++) {
      const start = minMs + i * (binSize * 1000);
      const end = start + (binSize * 1000);
      
      labels.push(`${Math.round(start)} - ${Math.round(end)}`);
      
      // Create a bell curve distribution around the average
      const midpoint = (start + end) / 2;
      const distance = Math.abs(midpoint - avgMs);
      const maxDistance = (maxMs - minMs) / 2;
      // Normalize the distance to be between 0 and 1
      const normalizedDistance = maxDistance > 0 ? distance / maxDistance : 0;
      const factor = Math.max(0, 1 - normalizedDistance);
      
      // Estimate a count based on the bell curve and total request count
      const count = Math.round(route.count * Math.pow(factor, 2) / 2);
      
      data.push(count);
    }
    
    // Create chart
    window.routeDetailChart = new Chart(ctx, {
      type: 'bar',
      data: {
        labels: labels,
        datasets: [{
          label: 'Request Count (estimated)',
          data: data,
          backgroundColor: 'rgba(var(--accent-primary-rgb), 0.7)',
          borderColor: 'rgba(var(--accent-primary-rgb), 1)',
          borderWidth: 1
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          x: {
            title: {
              display: true,
              text: 'Response Time (ms)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            title: {
              display: true,
              text: 'Number of Requests (est.)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          }
        },
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            callbacks: {
              title: function(tooltipItems) {
                const item = tooltipItems[0];
                const label = item.label.split(' - ');
                return `${label[0]} - ${label[1]} ms`;
              }
            }
          }
        }
      }
    });
  }
  
  /**
   * Export routes data to CSV
   */
  function exportRoutesData() {
    const { filteredRoutes } = window.routesData;
    
    if (filteredRoutes.length === 0) {
      alert('No data to export');
      return;
    }
    
    // Create CSV content
    const headers = ['Path', 'Method', 'Count', 'Avg Time (ms)', 'Max Time (ms)', 'Min Time (ms)', 'Success Rate (%)'];
    
    let csvContent = headers.join(',') + '\n';
    
    filteredRoutes.forEach(route => {
      const row = [
        `"${route.path}"`,
        route.method,
        route.count,
        (route.avg_time * 1000).toFixed(2),
        (route.max_time * 1000).toFixed(2),
        (route.min_time * 1000).toFixed(2),
        route.success_rate.toFixed(1)
      ];
      
      csvContent += row.join(',') + '\n';
    });
    
    // Create and download file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', `routes_export_${new Date().toISOString().slice(0, 10)}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }
});