/**
 * LUKA System Page JavaScript
 * Handles system information display and maintenance actions
 */
document.addEventListener('DOMContentLoaded', function() {
  // Variables for pagination and table data - Initialize BEFORE calling any functions
  window.systemLogData = {
    allLogs: [],
    filteredLogs: [],
    currentPage: 1,
    totalPages: 1,
    itemsPerPage: 10
  };
  
  // Hardkodirane rute API-ja
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    metrics: `${publicUrl}/luka/digital-twin/api/metrics`,
    systemLogs: `${publicUrl}/luka/digital-twin/api/system-logs`,
    clearMetrics: `${publicUrl}/luka/digital-twin/api/clear-metrics`,
    restartWebsocket: `${publicUrl}/luka/digital-twin/api/restart-websocket`,
    state: `${publicUrl}/luka/digital-twin/api/state`
  };
  
  // Load system logs
  loadSystemLogs();
  
  // Event listeners
  document.getElementById('logSearch')?.addEventListener('input', function() {
    filterSystemLogs();
  });
  
  document.getElementById('logTypeSelect')?.addEventListener('change', function() {
    filterSystemLogs();
  });
  
  document.getElementById('closeLogDetails')?.addEventListener('click', function() {
    hideLogDetails();
  });
  
  document.getElementById('clearMetricsBtn')?.addEventListener('click', function() {
    showConfirmModal(
      'Clear Metrics Data',
      'Are you sure you want to clear all metrics data? This cannot be undone.',
      clearMetricsData
    );
  });
  
  document.getElementById('restartWebSocketBtn')?.addEventListener('click', function() {
    showConfirmModal(
      'Restart WebSocket Server',
      'Are you sure you want to restart the WebSocket server? This may temporarily disconnect users.',
      restartWebSocketServer
    );
  });
  
  document.getElementById('downloadReportBtn')?.addEventListener('click', function() {
    downloadReport();
  });
  
  document.getElementById('clearCacheBtn')?.addEventListener('click', function() {
    showConfirmModal(
      'Clear Cache',
      'Are you sure you want to clear all system cache files?',
      clearCacheFiles
    );
  });
  
  document.getElementById('closeConfirmModal')?.addEventListener('click', function() {
    hideConfirmModal();
  });
  
  document.getElementById('cancelActionBtn')?.addEventListener('click', function() {
    hideConfirmModal();
  });
  
  // Close modals when clicking outside
  document.getElementById('logDetailsModal')?.addEventListener('click', function(event) {
    if (event.target === this) {
      hideLogDetails();
    }
  });
  
  document.getElementById('confirmActionModal')?.addEventListener('click', function(event) {
    if (event.target === this) {
      hideConfirmModal();
    }
  });
  
  // Pagination controls
  document.getElementById('prevPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) > 1) {
      navigateToPage(parseInt(this.dataset.page) - 1);
    }
  });
  
  document.getElementById('nextPage')?.addEventListener('click', function() {
    if (parseInt(this.dataset.page) < parseInt(this.dataset.totalPages)) {
      navigateToPage(parseInt(this.dataset.page) + 1);
    }
  });
  
  /**
   * Load system log data
   */
  function loadSystemLogs() {
    // Show loading state
    const table = document.getElementById('systemLogTable');
    if (table) {
      const tbody = table.querySelector('tbody');
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="loading-state">
            <div class="loading-spinner"></div>
            <p>Loading system logs...</p>
          </td>
        </tr>
      `;
    }
    
    // Make API request to get system logs
    fetch(routes.systemLogs)
      .then(response => response.json())
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          // Process and store data for pagination and filtering
          const logs = processSystemLogs(data.data);
          window.systemLogData.allLogs = logs;
          window.systemLogData.filteredLogs = [...logs];
          
          // Reset to first page
          window.systemLogData.currentPage = 1;
          
          // Update pagination and render table
          updateLogPagination();
          renderSystemLogsTable();
        } else {
          // Show empty state
          const tbody = table?.querySelector('tbody');
          if (tbody) {
            tbody.innerHTML = `
              <tr>
                <td colspan="5" class="empty-table">
                  No system logs found.
                </td>
              </tr>
            `;
          }
          
          console.error('No system logs found or API returned an error');
        }
      })
      .catch(error => {
        console.error('Error fetching system logs:', error);
        
        // Show error state
        const tbody = table?.querySelector('tbody');
        if (tbody) {
          tbody.innerHTML = `
            <tr>
              <td colspan="5" class="empty-table">
                Error loading system logs. Please try again.
              </td>
            </tr>
          `;
        }
      });
  }
  
  /**
   * Process system logs data to standardized format
   */
  function processSystemLogs(logs) {
    return logs.map((log, index) => {
      return {
        id: `log-${index}`,
        timestamp: log.timestamp,
        type: log.log_type.charAt(0).toUpperCase() + log.log_type.slice(1), // Capitalize type
        component: log.component,
        message: log.message,
        context: log.context
      };
    });
  }
  
  /**
   * Update pagination information
   */
  function updateLogPagination() {
    const { filteredLogs, itemsPerPage } = window.systemLogData;
    
    // Calculate total pages
    const totalPages = Math.max(1, Math.ceil(filteredLogs.length / itemsPerPage));
    
    // Update pagination state
    window.systemLogData.totalPages = totalPages;
    window.systemLogData.currentPage = Math.min(window.systemLogData.currentPage, totalPages);
    
    // Update UI
    const prevButton = document.getElementById('prevPage');
    const nextButton = document.getElementById('nextPage');
    const pageInfo = document.getElementById('pageInfo');
    
    if (prevButton) {
      prevButton.disabled = window.systemLogData.currentPage <= 1;
      prevButton.dataset.page = window.systemLogData.currentPage;
    }
    
    if (nextButton) {
      nextButton.disabled = window.systemLogData.currentPage >= totalPages;
      nextButton.dataset.page = window.systemLogData.currentPage;
      nextButton.dataset.totalPages = totalPages;
    }
    
    if (pageInfo) {
      pageInfo.textContent = `Page ${window.systemLogData.currentPage} of ${totalPages}`;
    }
  }
  
  /**
   * Navigate to a specific page
   */
  function navigateToPage(page) {
    window.systemLogData.currentPage = page;
    updateLogPagination();
    renderSystemLogsTable();
  }
  
  /**
   * Render the system logs table with current data
   */
  function renderSystemLogsTable() {
    const table = document.getElementById('systemLogTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = '';
    
    const { filteredLogs, currentPage, itemsPerPage } = window.systemLogData;
    
    // Calculate slice for current page
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageData = filteredLogs.slice(startIndex, endIndex);
    
    if (pageData.length === 0) {
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="empty-table">
            No logs found matching your criteria
          </td>
        </tr>
      `;
      return;
    }
    
    // Create rows for each log
    pageData.forEach(log => {
      const row = document.createElement('tr');
      
      row.innerHTML = `
        <td>${formatDateTime(log.timestamp)}</td>
        <td>
          <div class="log-type ${log.type.toLowerCase()}">
            <i class="fas fa-${getLogTypeIcon(log.type)}"></i> ${log.type}
          </div>
        </td>
        <td>${log.component}</td>
        <td title="${log.message}">${truncateText(log.message, 50)}</td>
        <td>
          <div class="log-actions">
            <button class="log-action-btn view-log" title="View Details" data-id="${log.id}">
              <i class="fas fa-eye"></i>
            </button>
          </div>
        </td>
      `;
      
      tbody.appendChild(row);
    });
    
    // Add event listeners for action buttons
    document.querySelectorAll('.view-log').forEach(button => {
      button.addEventListener('click', function() {
        const logId = this.dataset.id;
        const log = pageData.find(l => l.id === logId);
        if (log) {
          showLogDetails(log);
        }
      });
    });
  }
  
  /**
   * Filter system logs based on search term and type
   */
  function filterSystemLogs() {
    const searchTerm = document.getElementById('logSearch')?.value.toLowerCase() || '';
    const logType = document.getElementById('logTypeSelect')?.value || 'all';
    
    if (!window.systemLogData) return;
    
    let filteredData = [...window.systemLogData.allLogs];
    
    // Filter by log type
    if (logType !== 'all') {
      filteredData = filteredData.filter(log => log.type.toLowerCase() === logType.toLowerCase());
    }
    
    // Filter by search term
    if (searchTerm) {
      filteredData = filteredData.filter(log => 
        log.message.toLowerCase().includes(searchTerm) ||
        log.component.toLowerCase().includes(searchTerm)
      );
    }
    
    // Update filtered data
    window.systemLogData.filteredLogs = filteredData;
    
    // Reset to first page
    window.systemLogData.currentPage = 1;
    
    // Update pagination and render table
    updateLogPagination();
    renderSystemLogsTable();
  }
  
  /**
   * Show log details in modal
   */
  function showLogDetails(log) {
    const modal = document.getElementById('logDetailsModal');
    if (!modal) return;
    
    // Update modal content
    document.getElementById('logComponent').textContent = log.component;
    document.getElementById('logTimestamp').textContent = formatDateTime(log.timestamp);
    document.getElementById('logDetailMessage').textContent = log.message;
    
    // Update log type icon and color
    const typeIcon = document.getElementById('logTypeIcon');
    typeIcon.className = `fas fa-${getLogTypeIcon(log.type)}`;
    
    // Update context data
    if (log.context) {
      document.getElementById('logContext').textContent = 
        typeof log.context === 'object' ? JSON.stringify(log.context, null, 2) : log.context;
    } else {
      document.getElementById('logContext').textContent = 'No context data available';
    }
    
    // Set appropriate background color for log type badge
    const logTypeBadge = document.querySelector('.log-type-badge');
    
    // Remove existing type classes
    logTypeBadge.classList.remove('error', 'warning', 'info', 'debug');
    
    // Add appropriate class based on log type
    const type = log.type.toLowerCase();
    if (logTypeBadge) {
      switch (type) {
        case 'error':
          logTypeBadge.style.backgroundColor = 'rgba(231, 76, 60, 0.2)';
          typeIcon.style.color = '#e74c3c';
          break;
        case 'warning':
          logTypeBadge.style.backgroundColor = 'rgba(243, 156, 18, 0.2)';
          typeIcon.style.color = '#f39c12';
          break;
        case 'info':
          logTypeBadge.style.backgroundColor = 'rgba(52, 152, 219, 0.2)';
          typeIcon.style.color = '#3498db';
          break;
        case 'debug':
          logTypeBadge.style.backgroundColor = 'rgba(149, 165, 166, 0.2)';
          typeIcon.style.color = '#95a5a6';
          break;
      }
    }
    
    // Show modal
    modal.style.display = 'flex';
  }
  
  /**
   * Hide log details modal
   */
  function hideLogDetails() {
    const modal = document.getElementById('logDetailsModal');
    if (modal) {
      modal.style.display = 'none';
    }
  }
  
  /**
   * Show confirmation modal
   */
  function showConfirmModal(title, message, confirmAction) {
    const modal = document.getElementById('confirmActionModal');
    if (!modal) return;
    
    // Update modal content
    document.getElementById('confirmTitle').textContent = title;
    document.getElementById('confirmMessage').textContent = message;
    
    // Set confirm action
    const confirmButton = document.getElementById('confirmActionBtn');
    if (confirmButton) {
      // Remove existing event listeners
      const newConfirmButton = confirmButton.cloneNode(true);
      confirmButton.parentNode.replaceChild(newConfirmButton, confirmButton);
      
      // Add new event listener
      newConfirmButton.addEventListener('click', function() {
        hideConfirmModal();
        if (typeof confirmAction === 'function') {
          confirmAction();
        }
      });
    }
    
    // Show modal
    modal.style.display = 'flex';
  }
  
  /**
   * Hide confirmation modal
   */
  function hideConfirmModal() {
    const modal = document.getElementById('confirmActionModal');
    if (modal) {
      modal.style.display = 'none';
    }
  }
  
  /**
   * Clear metrics data action
   */
  function clearMetricsData() {
    const button = document.getElementById('clearMetricsBtn');
    if (button) {
      button.disabled = true;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Processing...</span>';
    }
    
    // Make API request
    fetch(routes.clearMetrics, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Requested-With': 'XMLHttpRequest'
      }
    })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          showSuccessMessage('Metrics data cleared successfully!');
          
          // Reload the page to refresh the metrics count
          setTimeout(() => {
            window.location.reload();
          }, 1500);
        } else {
          showErrorMessage('Failed to clear metrics data: ' + (data.message || 'Unknown error'));
          
          // Reset button state
          if (button) {
            button.disabled = false;
            button.innerHTML = '<i class="fas fa-trash"></i><span>Clear Metrics Data</span>';
          }
        }
      })
      .catch(error => {
        console.error('Error clearing metrics:', error);
        showErrorMessage('An error occurred while clearing metrics data');
        
        // Reset button state
        if (button) {
          button.disabled = false;
          button.innerHTML = '<i class="fas fa-trash"></i><span>Clear Metrics Data</span>';
          }
      });
  }
  
  /**
   * Restart WebSocket server action
   */
  function restartWebSocketServer() {
    const button = document.getElementById('restartWebSocketBtn');
    if (button) {
      button.disabled = true;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Restarting...</span>';
    }
    
    // Make API request
    fetch(routes.restartWebsocket, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Requested-With': 'XMLHttpRequest'
      }
    })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          showSuccessMessage('WebSocket server restarted successfully!');
          
          // Reload the page after a short delay
          setTimeout(() => {
            window.location.reload();
          }, 1500);
        } else {
          showErrorMessage('Failed to restart WebSocket server: ' + (data.message || 'Unknown error'));
          
          // Reset button state
          if (button) {
            button.disabled = false;
            button.innerHTML = '<i class="fas fa-sync-alt"></i><span>Restart WebSocket</span>';
          }
        }
      })
      .catch(error => {
        console.error('Error restarting WebSocket server:', error);
        showErrorMessage('An error occurred while restarting the WebSocket server');
        
        // Reset button state
        if (button) {
          button.disabled = false;
          button.innerHTML = '<i class="fas fa-sync-alt"></i><span>Restart WebSocket</span>';
        }
      });
  }
  
  /**
   * Download performance report action
   */
  function downloadReport() {
    const button = document.getElementById('downloadReportBtn');
    if (button) {
      button.disabled = true;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Generating...</span>';
    }
    
    // Get current date for filename
    const now = new Date();
    const dateString = now.getFullYear() + '-' + 
      String(now.getMonth() + 1).padStart(2, '0') + '-' + 
      String(now.getDate()).padStart(2, '0');
    const filename = 'performance_report_' + dateString + '.json';
    
    // Get performance data
    fetch(routes.metrics + '?type=performance&from=-7 days&to=now')
      .then(response => response.json())
      .then(data => {
        // Create and download file
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        // Reset button state
        if (button) {
          button.disabled = false;
          button.innerHTML = '<i class="fas fa-download"></i><span>Download Report</span>';
        }
        
        showSuccessMessage('Report downloaded successfully!');
      })
      .catch(error => {
        console.error('Error downloading report:', error);
        showErrorMessage('An error occurred while generating the report');
        
        // Reset button state
        if (button) {
          button.disabled = false;
          button.innerHTML = '<i class="fas fa-download"></i><span>Download Report</span>';
        }
      });
  }
  
  /**
   * Clear cache files action
   */
  function clearCacheFiles() {
    const button = document.getElementById('clearCacheBtn');
    if (button) {
      button.disabled = true;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Processing...</span>';
    }
    
    // In real implementation, this would make an API request
    // For now, just simulate success after a short delay
    setTimeout(() => {
      showSuccessMessage('Cache files cleared successfully!');
      
      // Reset button state
      if (button) {
        button.disabled = false;
        button.innerHTML = '<i class="fas fa-broom"></i><span>Clear Cache</span>';
      }
    }, 1500);
  }
  
  /**
   * Show success message toast
   */
  function showSuccessMessage(message) {
    // Create toast element
    const toast = document.createElement('div');
    toast.className = 'luka-toast success';
    toast.innerHTML = `
      <div class="toast-icon">
        <i class="fas fa-check-circle"></i>
      </div>
      <div class="toast-content">
        <p>${message}</p>
      </div>
    `;
    
    // Add to DOM
    document.body.appendChild(toast);
    
    // Trigger animation
    setTimeout(() => {
      toast.classList.add('show');
    }, 10);
    
    // Remove after 3 seconds
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => {
        document.body.removeChild(toast);
      }, 300); // Match transition duration
    }, 3000);
  }
  
  /**
   * Show error message toast
   */
  function showErrorMessage(message) {
    // Create toast element
    const toast = document.createElement('div');
    toast.className = 'luka-toast error';
    toast.innerHTML = `
      <div class="toast-icon">
        <i class="fas fa-exclamation-circle"></i>
      </div>
      <div class="toast-content">
        <p>${message}</p>
      </div>
    `;
    
    // Add to DOM
    document.body.appendChild(toast);
    
    // Trigger animation
    setTimeout(() => {
      toast.classList.add('show');
    }, 10);
    
    // Remove after 5 seconds
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => {
        document.body.removeChild(toast);
      }, 300); // Match transition duration
    }, 5000);
  }
  
  /* Helper Functions */
  
  /**
   * Get icon for log type
   */
  function getLogTypeIcon(type) {
    switch (type.toLowerCase()) {
      case 'error':
        return 'exclamation-circle';
      case 'warning':
        return 'exclamation-triangle';
      case 'info':
        return 'info-circle';
      case 'debug':
        return 'bug';
      default:
        return 'info-circle';
    }
  }
  
  /**
   * Format date and time for display
   */
  function formatDateTime(dateTimeStr) {
    if (!dateTimeStr) return 'Unknown';
    
    try {
      const date = new Date(dateTimeStr);
      return date.toLocaleString();
    } catch (e) {
      return dateTimeStr;
    }
  }
  
  /**
   * Truncate text with ellipsis
   */
  function truncateText(text, maxLength) {
    if (!text) return '';
    return text.length > maxLength ? text.substring(0, maxLength - 3) + '...' : text;
  }
});