// csrf-fetch.js - Enhanced with notification functionality
let isRefreshing = false;

// Create notification element
function createNotificationElement() {
    const notificationDiv = document.createElement('div');
    notificationDiv.className = 'session-notification notification-default';
    document.body.appendChild(notificationDiv);
    return notificationDiv;
}

// Show notification
function showNotification(message, type = 'info') {
    let notificationDiv = document.querySelector('.session-notification');
    if (!notificationDiv) {
        notificationDiv = createNotificationElement();
    }
    
    // Reset classes and add correct type
    notificationDiv.className = 'session-notification';
    notificationDiv.classList.add(type === 'error' ? 'notification-error' : 'notification-info');
    
    notificationDiv.innerHTML = message;
    
    // Show with fade effect
    notificationDiv.style.display = 'block';
    notificationDiv.style.opacity = '1';
    
    // Auto-hide error messages after 3 seconds
    if (type === 'error') {
        setTimeout(() => {
            notificationDiv.style.opacity = '0';
            setTimeout(() => {
                notificationDiv.style.display = 'none';
            }, 300);
        }, 3000);
    }
}

window.fetchWithCsrf = async function(url, options = {}) {
    // Get CSRF token from hidden input
    const csrfToken = document.querySelector('input[name="csrf_token"]')?.value;
    if (!csrfToken) {
        console.error('CSRF token not found');
        throw new Error('CSRF token not found');
    }
    
    // Set up options
    options = options || {};
    options.headers = options.headers || {};
    options.headers['X-CSRF-Token'] = csrfToken;
    options.headers['X-Requested-With'] = 'XMLHttpRequest';
    
    // For POST requests, include token in body too
    if (options.method && options.method.toUpperCase() === 'POST') {
        if (options.body instanceof FormData) {
            options.body.append('csrf_token', csrfToken);
        } else if (typeof options.body === 'string') {
            try {
                // Try to parse JSON
                const bodyObj = JSON.parse(options.body);
                bodyObj.csrf_token = csrfToken;
                options.body = JSON.stringify(bodyObj);
            } catch (e) {
                // Not JSON, append as string
                options.body += '&csrf_token=' + csrfToken;
            }
        } else if (typeof options.body === 'object' && options.body !== null) {
            options.body.csrf_token = csrfToken;
            if (!(options.body instanceof FormData)) {
                options.body = JSON.stringify(options.body);
            }
        } else {
            // No body, create one
            options.body = JSON.stringify({ csrf_token: csrfToken });
            options.headers['Content-Type'] = 'application/json';
        }
    }
    
    // Make the request with error handling
    try {
        const response = await fetch(url, options);
        
        // Handle 403 errors (CSRF validation failures)
        if (response.status === 403 && !isRefreshing) {
            isRefreshing = true;
            console.log('CSRF validation failed, refreshing page...');
            
            showNotification(`
                <div class="notification-content">
                    <div>Refreshing your session...</div>
                    <div class="spinner-border spinner-border-sm" role="status">
                        <span class="sr-only">Loading...</span>
                    </div>
                </div>
            `, 'info');
            
            // Refresh after delay
            setTimeout(() => {
                window.location.reload();
            }, 2000);
            
            return response;
        } else if (response.status === 404) {
            showNotification('Unable to process your request. Please try again.', 'error');
        }
        
        return response;
    } catch (error) {
        showNotification('Connection error. Please check your network.', 'error');
        throw error;
    }
};