$(document).ready(function() {
    let isRefreshing = false;

    // Get CSRF token from hidden field
    const csrfToken = $('input[name="csrf_token"]').val();
    
    if (!csrfToken) {
        console.warn('CSRF token not found in the page. Make sure @csrf directive is present.');
        return;
    }
    
    // Setup CSRF token for all AJAX requests
    $.ajaxSetup({
        headers: {
            'X-CSRF-Token': csrfToken
        },
        beforeSend: function(xhr, settings) {
            // Always set the header token
            if (!xhr.getResponseHeader('X-CSRF-Token')) {
                xhr.setRequestHeader('X-CSRF-Token', csrfToken);
            }

            // For POST requests, also include token in the data
            if (settings.type === 'POST') {
                // Handle FormData
                if (settings.data instanceof FormData) {
                    settings.data.append('csrf_token', csrfToken);
                } 
                // Handle string data
                else if (typeof settings.data === 'string') {
                    settings.data += '&csrf_token=' + csrfToken;
                } 
                // Handle object data
                else if (typeof settings.data === 'object' && settings.data !== null) {
                    settings.data.csrf_token = csrfToken;
                }
                // If no data, create new object with token
                else {
                    settings.data = { csrf_token: csrfToken };
                }
            }
        }
    });

    // Create a reusable notification element
    const notificationDiv = $('<div>')
        .addClass('session-notification')
        .addClass('notification-default')
        .appendTo('body');

    // Error handler for all AJAX requests
    $(document).ajaxError(function(event, jqXHR, settings, error) {
        if (jqXHR.status === 403 && !isRefreshing) {
            isRefreshing = true;
            console.log('CSRF validation failed, refreshing page...');

            // Show notification
            notificationDiv
                .removeClass('notification-error')
                .addClass('notification-info')
                .html(`
                    <div class="notification-content">
                        <div>Refreshing your session...</div>
                        <div class="spinner-border spinner-border-sm" role="status">
                            <span class="sr-only">Loading...</span>
                        </div>
                    </div>
                `)
                .fadeIn();

            // Gently refresh after a short delay
            setTimeout(function() {
                window.location.reload();
            }, 2000);
        } else if (jqXHR.status === 404 && !isRefreshing) {
            notificationDiv
                .removeClass('notification-info')
                .addClass('notification-error')
                .html('Unable to process your request. Please try again.')
                .fadeIn();

            setTimeout(function() {
                notificationDiv.fadeOut();
            }, 3000);
        }
    });
});