/**
 * FirebaseRealtimeHandler.js
 * Handles Firebase Realtime Database operations for notifications
 */
class FirebaseRealtimeHandler {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;
        this.userId = this.isLoggedIn ? this.userData.id : null;

        // Skip initialization if user is not logged in
        if (!this.isLoggedIn) {
            console.log('User not logged in, skipping Firebase Realtime DB initialization');
            return;
        }

        // API Routes
        this.routes = {
            updateReadStatus: `${this.publicUrl}/firebase-realtime/api/update-read-status`,
            updateStickyStatus: `${this.publicUrl}/firebase-realtime/api/update-sticky-status`,
            syncFirebase: `${this.publicUrl}/firebase-realtime/api/sync-firebase`
        };

        // Firebase database reference
        this.realtimeDBRef = null;
        this.initialized = false;
        this.initAttempts = 0;
        this.authInitialized = false;
        
        // Track seen notification IDs to prevent duplicates
        this.seenNotificationIds = new Set();
        
        // Logger
        this.logger = this.createLogger();

        // Initialize
        this.waitForFirebase();
    }

    /**
     * Create logger function with debug mode check
     * @returns {Object} Logger object with log, warn, error methods
     */
    createLogger() {
        return {
            log: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.log(`[FirebaseRealtimeHandler] ${message}`, ...args);
                }
            },
            warn: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.warn(`[FirebaseRealtimeHandler] ${message}`, ...args);
                }
            },
            error: (message, ...args) => {
                // Always log errors
                console.error(`[FirebaseRealtimeHandler] ${message}`, ...args);
            }
        };
    }

    /**
     * Wait for Firebase to be initialized before proceeding
     */
    waitForFirebase() {
        // Check if Firebase is already initialized
        if (this.isFirebaseInitialized()) {
            this.logger.log('Firebase detected as already initialized');
            // Don't initialize database yet, wait for auth
            this.waitForAuth();
            return;
        }
    
        // Listen for events
        document.addEventListener('firebase-ready', this.handleFirebaseReady.bind(this));
        document.addEventListener('firebaseInitialized', this.handleFirebaseReady.bind(this));
        document.addEventListener('firebase-auth-success', this.handleFirebaseAuthSuccess.bind(this));
        
        // Timeout to check again
        setTimeout(() => {
            this.checkFirebaseAndInitialize();
        }, 1000);
    }    

    /**
     * Handle Firebase auth success event
     */
    handleFirebaseAuthSuccess(event) {
        this.logger.log('Firebase authentication success event received', event.detail);
        
        // If already initialized, don't do it again
        if (this.initialized) return;
        
        this.authInitialized = true;
        
        // Initialize database after a short delay to ensure auth is fully processed
        setTimeout(() => {
            this.initializeRealtimeDB();
        }, 500);
    }
    
    /**
     * Check if Firebase is initialized and try again if needed
     */
    checkFirebaseAndInitialize() {
        if (this.initialized) return;
        
        this.initAttempts++;
        this.logger.log(`Checking Firebase initialization, attempt ${this.initAttempts}`);
        
        if (this.isFirebaseInitialized()) {
            this.logger.log('Firebase now initialized, proceeding with database setup');
            this.waitForAuth();
            return;
        }
        
        // Give up after too many attempts
        if (this.initAttempts >= 5) {
            this.logger.error('Giving up on waiting for Firebase after 5 attempts');
            return;
        }
        
        // Try again after delay
        setTimeout(() => {
            this.checkFirebaseAndInitialize();
        }, 1000);
    }

    /**
     * Check if Firebase is properly initialized
     */
    isFirebaseInitialized() {
        return (
            typeof firebase !== 'undefined' && 
            firebase.apps && 
            firebase.apps.length > 0 && 
            window.firebaseInitialized === true
        );
    }

    /**
     * Wait for authentication before initializing database
     */
    waitForAuth() {
        // Check if already authenticated
        if (firebase.auth().currentUser) {
            this.logger.log('User is already authenticated with Firebase');
            this.authInitialized = true;
            this.initializeRealtimeDB();
            return;
        }
    
        // Listen for auth state changes
        firebase.auth().onAuthStateChanged((user) => {
            if (user) {
                this.logger.log('Firebase auth state changed: user authenticated', user.uid);
                this.authInitialized = true;
                this.initializeRealtimeDB();
            } else {
                this.logger.log('Firebase auth state changed: user signed out');
                // We'll wait for the auth success event
            }
        });
    
        // Set a timeout in case auth never completes
        setTimeout(() => {
            if (!this.authInitialized) {
                this.logger.error('Timed out waiting for Firebase authentication');
            }
        }, 10000);
    }

    /**
     * Handle Firebase ready event
     */
    handleFirebaseReady(event) {
        this.logger.log('Firebase ready event received', event.detail);
        if (event.detail?.success) {
            // Don't initialize database yet, wait for auth
            this.waitForAuth();
        }
    }

    /**
     * Initialize Firebase Realtime Database
     */
    initializeRealtimeDB() {
        if (this.initialized || !this.userId) return;
    
        // Check if authenticated
        if (!firebase.auth().currentUser) {
            this.logger.error('User is not authenticated with Firebase');
            return;
        }
    
        try {
            this.logger.log('Initializing Firebase Realtime Database');
    
            // Double check Firebase is initialized
            if (!this.isFirebaseInitialized()) {
                this.logger.error('Firebase not initialized yet, cannot initialize database');
                return;
            }
    
            // Verify database is available
            if (typeof firebase.database !== 'function') {
                this.logger.error('Firebase database not available');
                return;
            }
            
            this.setupDatabaseReferences();
    
        } catch (error) {
            this.logger.error('Error initializing Firebase Realtime Database:', error);
        }
    }

    /**
     * Set up database references and listeners
     */
    setupDatabaseReferences() {
        try {
            // Get the database instance
            const database = firebase.database();
    
            // Reference to user's notifications node (parent)
            const userNotificationsRef = database.ref(`/notifications/${this.userId}`);
    
            // Reference to user's notifications items
            this.realtimeDBRef = userNotificationsRef.child('items');
    
            // Listen for new notifications
            this.realtimeDBRef.on('child_added', this.handleNewNotification.bind(this));
    
            // Listen for changes in existing notifications
            this.realtimeDBRef.on('child_changed', this.handleUpdatedNotification.bind(this));
    
            // Log existing notifications for diagnosis
            this.realtimeDBRef.once('value', (snapshot) => {
                this.logger.log(`Found ${snapshot.numChildren()} existing notifications`);
                
                snapshot.forEach((childSnapshot) => {
                    const notificationId = childSnapshot.key;
                    this.seenNotificationIds.add(notificationId);
                });
            });
            
            // Listen for changes in notification count
            userNotificationsRef.child('count').on('value', (snapshot) => {
                const count = snapshot.val() || 0;
                this.logger.log('Current notification count from Firebase:', count);
            });
            
            // Setup presence system for online/offline detection
            this.setupPresenceSystem(database);
    
            this.initialized = true;
            
            // Broadcast that Firebase Realtime Database is ready
            document.dispatchEvent(new CustomEvent('firebase-realtime-ready', { 
                detail: { success: true } 
            }));
            
            this.logger.log('Firebase Realtime Database initialized successfully');
        } catch (error) {
            this.logger.error('Error setting up Firebase Realtime Database:', error);
        }
    }

    /**
     * Setup Firebase presence system to detect connection state
     */
    setupPresenceSystem(database) {
        // Skip if no user ID
        if (!this.userId) {
            return;
        }
        
        // Get a reference to the special /.info/connected path
        const connectedRef = database.ref('.info/connected');
        
        connectedRef.on('value', (snap) => {
            const connected = snap.val();
            if (connected) {
                this.logger.log('Connected to Firebase Realtime Database');
                
                // Update user's online status
                const userStatusRef = database.ref(`/users/${this.userId}/status`);
                
                // When disconnected, update the status
                userStatusRef.onDisconnect().set('offline');
                
                // Set the current status
                userStatusRef.set('online');
            } else {
                this.logger.log('Disconnected from Firebase Realtime Database');
            }
        });
    }

    /**
     * Handle a new notification from Firebase
     */
    handleNewNotification(snapshot) {
        try {
            const notificationId = snapshot.key;
            const notification = snapshot.val();

            // Add ID to notification object
            notification.id = notificationId;

            this.logger.log('New notification received from Firebase:', notificationId);

            // Check if we've already seen this notification
            if (this.seenNotificationIds.has(notificationId)) {
                this.logger.log('Ignoring already processed notification:', notificationId);
                return;
            }

            // Add to set of seen notifications
            this.seenNotificationIds.add(notificationId);

            // Normalize is_read value to boolean
            notification.is_read = !!notification.is_read;

            // Skip already read notifications
            if (notification.is_read) {
                this.logger.log('Ignoring already read notification:', notificationId);
                return;
            }

            // Dispatch event for notification handlers
            document.dispatchEvent(new CustomEvent('newNotificationReceived', {
                detail: notification
            }));

            this.logger.log('Dispatched newNotificationReceived event for:', notificationId);
        } catch (error) {
            this.logger.error('Error handling new notification:', error);
        }
    }

    /**
     * Handle an updated notification from Firebase
     */
    handleUpdatedNotification(snapshot) {
        try {
            const notificationId = snapshot.key;
            const notification = snapshot.val();

            // Add ID to notification object
            notification.id = notificationId;

            this.logger.log('Notification updated in Firebase:', notificationId);

            // Dispatch event for notification handlers
            document.dispatchEvent(new CustomEvent('notificationUpdated', {
                detail: notification
            }));

            this.logger.log('Dispatched notificationUpdated event for:', notificationId);
        } catch (error) {
            this.logger.error('Error handling updated notification:', error);
        }
    }

    /**
     * Mark a notification as read in Firebase
     */
    async markAsRead(notificationId) {
        if (!this.initialized || !this.realtimeDBRef) {
            return false;
        }

        try {
            // Update in Firebase directly
            await this.realtimeDBRef.child(notificationId).update({
                is_read: true,
                updated_at: Date.now()
            });

            // Also sync with server
            await this.fetchWithCsrf(this.routes.updateReadStatus, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify({ notification_id: notificationId })
            });

            this.logger.log('Notification marked as read in Firebase:', notificationId);
            return true;
        } catch (error) {
            this.logger.error('Error marking notification as read in Firebase:', error);
            return false;
        }
    }

    /**
     * Mark all notifications as read in Firebase
     */
    async markAllAsRead() {
        if (!this.initialized || !this.realtimeDBRef) {
            return false;
        }

        try {
            // Get current notifications
            const snapshot = await this.realtimeDBRef.once('value');
            let updatePromises = [];

            // Update each non-sticky unread notification
            snapshot.forEach((childSnapshot) => {
                const notification = childSnapshot.val();
                const id = childSnapshot.key;

                // Skip already read or sticky notifications
                if (notification.is_read || notification.is_sticky) {
                    return;
                }

                // Add update promise
                updatePromises.push(
                    this.realtimeDBRef.child(id).update({
                        is_read: true,
                        updated_at: Date.now()
                    })
                );
            });

            // Wait for all updates to complete
            if (updatePromises.length > 0) {
                await Promise.all(updatePromises);
                
                // Also sync with server
                await this.fetchWithCsrf(this.routes.updateReadStatus, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    body: JSON.stringify({})
                });
                
                this.logger.log(`Marked ${updatePromises.length} notifications as read in Firebase`);
            } else {
                this.logger.log('No unread non-sticky notifications to mark as read');
            }

            return true;
        } catch (error) {
            this.logger.error('Error marking all notifications as read in Firebase:', error);
            return false;
        }
    }

    /**
     * Toggle sticky status for a notification in Firebase
     */
    async toggleSticky(notificationId) {
        if (!this.initialized || !this.realtimeDBRef) {
            return false;
        }

        try {
            // Get current notification
            const snapshot = await this.realtimeDBRef.child(notificationId).once('value');
            const notification = snapshot.val();

            if (!notification) {
                this.logger.error('Notification not found:', notificationId);
                return false;
            }

            // Toggle sticky status
            const newStickyStatus = !notification.is_sticky;

            // Update in Firebase
            await this.realtimeDBRef.child(notificationId).update({
                is_sticky: newStickyStatus,
                updated_at: Date.now()
            });
            
            // Also sync with server
            await this.fetchWithCsrf(this.routes.updateStickyStatus, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify({ 
                    notification_id: notificationId,
                    is_sticky: newStickyStatus
                })
            });

            this.logger.log(`Notification ${notificationId} sticky status set to ${newStickyStatus}`);
            return true;
        } catch (error) {
            this.logger.error('Error toggling sticky status in Firebase:', error);
            return false;
        }
    }
    
    /**
     * Trigger a full sync with the server
     */
    async fullSync() {
        try {
            const response = await this.fetchWithCsrf(this.routes.syncFirebase, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            const result = await response.json();
            
            this.logger.log('Full Firebase sync result:', result);
            
            return result.success;
        } catch (error) {
            this.logger.error('Error triggering full Firebase sync:', error);
            return false;
        }
    }

    /**
     * Fetch with CSRF token
     */
    fetchWithCsrf(url, options = {}) {
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
        
        // Add CSRF token to headers
        const headers = {
            ...options.headers,
            'X-CSRF-TOKEN': csrfToken
        };
        
        // Return fetch with updated options
        return fetch(url, {
            ...options,
            headers,
            credentials: 'same-origin'
        });
    }

    /**
     * Cleanup event listeners when destroying the handler
     */
    destroy() {
        if (this.realtimeDBRef) {
            this.realtimeDBRef.off();
            this.logger.log('Firebase Realtime Database listeners removed');
        }
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.firebaseRealtimeHandler = new FirebaseRealtimeHandler();
});