/**
 * InitializationManager.js
 * Manages initialization of client-side components with centralized data loading
 */
class InitializationManager {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;

        // Create logger
        this.logger = this.createLogger();

        // Initialize
        if (this.isLoggedIn) {
            this.initialize();
        }
    }

    /**
     * Create logger function with debug mode check
     */
    createLogger() {
        return {
            log: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.log(`[InitializationManager] ${message}`, ...args);
                }
            },
            warn: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.warn(`[InitializationManager] ${message}`, ...args);
                }
            },
            error: (message, ...args) => {
                // Always log errors
                console.error(`[InitializationManager] ${message}`, ...args);
            }
        };
    }

    /**
     * Initialize all components with a single API call
     */
    async initialize() {
        try {
            // Try to load from cache first
            const cachedData = localStorage.getItem('app_initial_data');
            const cachedTimestamp = localStorage.getItem('app_initial_data_timestamp');
            const currentTime = Date.now();
            const CACHE_EXPIRY = 300000; // 5 minutes
            
            let initialData = null;
            
            // Use cached data if it exists and is not expired
            if (cachedData && cachedTimestamp && 
                (currentTime - parseInt(cachedTimestamp, 10) < CACHE_EXPIRY)) {
                try {
                    initialData = JSON.parse(cachedData);
                    this.logger.log('Using cached initial data');
                } catch (e) {
                    this.logger.error('Error parsing cached initial data:', e);
                }
            }
            
            // If no valid cache, fetch from API
            if (!initialData) {
                this.logger.log('Fetching initial data from API');
                
                // Get CSRF token
                const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
                
                const response = await fetch(`${this.publicUrl}/api/initial-data`, {
                    method: 'GET',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': csrfToken
                    },
                    credentials: 'same-origin'
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error ${response.status}`);
                }
                
                const responseData = await response.json();
                
                if (responseData.success) {
                    initialData = responseData.data;
                    
                    // Cache the data
                    localStorage.setItem('app_initial_data', JSON.stringify(initialData));
                    localStorage.setItem('app_initial_data_timestamp', currentTime.toString());
                } else {
                    throw new Error('API returned error');
                }
            }
            
            // Initialize components with the data
            this.initializeComponents(initialData);
            
        } catch (error) {
            this.logger.error('Error initializing application:', error);
            
            // Fallback to individual component initialization
            this.initializeComponentsIndividually();
        }
    }
    
    /**
     * Initialize components with the fetched data
     */
    initializeComponents(data) {
        // Set global data
        if (data.notification_types) {
            window.notificationTypes = data.notification_types.types;
            window.defaultNotificationIcon = data.notification_types.defaultIcon;
        }
        
        if (data.firebase_config) {
            window.firebaseConfig = data.firebase_config.config;
            window.firebaseFeatures = data.firebase_config.features;
        }
        
        // Initialize unread notification count
        if (data.unread_notifications) {
            const unreadCount = data.unread_notifications.length;
            localStorage.setItem('notification_unread_count', unreadCount.toString());
            localStorage.setItem('notification_unread_count_timestamp', Date.now().toString());
            
            // Update badge if handler exists
            if (window.notificationBadgeHandler) {
                window.notificationBadgeHandler.updateBadge(unreadCount);
            }
        }
        
        // Dispatch initialization event
        document.dispatchEvent(new CustomEvent('app-initialized', { 
            detail: { 
                success: true,
                initialData: data
            } 
        }));
    }
    
    /**
     * Fallback to individual component initialization
     */
    initializeComponentsIndividually() {
        this.logger.log('Falling back to individual component initialization');
        // Components will initialize themselves as before
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.initManager = new InitializationManager();
});