/**
 * NotificationUtils.js
 * Utility functions for notifications
 */
(function() {
    // Function to get notification icon based on type
    window.getNotificationIconByType = function(type, basePath) {
        if (window.notificationTypes && window.notificationTypes[type]) {
            return window.notificationTypes[type].icon;
        }
        return `${basePath}/staticfirebase/notification-icon.png`;
    };
    
    // Load notification types and icons from API with caching
    document.addEventListener('DOMContentLoaded', function() {
        const appConfig = document.getElementById('app-config');
        const publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';
        
        // Skip if user is not logged in
        const userDataElement = document.getElementById('user-data');
        const isLoggedIn = userDataElement && JSON.parse(userDataElement.textContent).id;
        
        if (!isLoggedIn) return;
        
        // Check for app-initialized event (if InitializationManager is active)
        document.addEventListener('app-initialized', function(event) {
            // Data is already set by InitializationManager
            console.log('Notification types already loaded by InitializationManager');
            return;
        });
        
        // Try to load from cache first
        const cachedTypes = localStorage.getItem('notification_types');
        const cachedTimestamp = localStorage.getItem('notification_types_timestamp');
        const currentTime = Date.now();
        const CACHE_EXPIRY = 86400000; // 24 hours in milliseconds
        
        // Use cached types if they exist and are not expired
        if (cachedTypes && cachedTimestamp && 
            (currentTime - parseInt(cachedTimestamp, 10) < CACHE_EXPIRY)) {
            try {
                const data = JSON.parse(cachedTypes);
                window.notificationTypes = data.types;
                window.defaultNotificationIcon = data.defaultIcon;
                console.log('Using cached notification types');
                return;
            } catch (e) {
                console.error('Error parsing cached notification types:', e);
                // Continue to fetch from API if cache parsing fails
            }
        }
        
        // Fetch from API if cache doesn't exist or is expired
        fetch(`${publicUrl}/notification/api/get-notification-types`, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            credentials: 'same-origin'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                // Save to window object for immediate use
                window.notificationTypes = data.data.types;
                window.defaultNotificationIcon = data.data.defaultIcon;
                
                // Cache for future use
                localStorage.setItem('notification_types', JSON.stringify(data.data));
                localStorage.setItem('notification_types_timestamp', currentTime.toString());
                console.log('Notification types cached');
            }
        })
        .catch(error => {
            console.error('Error loading notification types:', error);
        });
    });
})();