/**
 * RealtimeNotificationHandler.js
 * Handles real-time notification updates from Firebase Realtime Database
 */
class RealtimeNotificationHandler {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;

        // Skip initialization if user is not logged in
        if (!this.isLoggedIn) {
            console.log('User not logged in, skipping realtime notification handler initialization');
            return;
        }

        // Notification counter
        this.unreadCount = 0;

        // Initialize
        this.init();
    }

    /**
     * Initialize the handler
     */
    init() {
        // Listen for new notification events
        document.addEventListener('newNotificationReceived', this.handleNewNotification.bind(this));
        
        // Listen for notification update events
        document.addEventListener('notificationUpdated', this.handleUpdatedNotification.bind(this));
        
        // Listen for Firebase Realtime Database ready event
        document.addEventListener('firebase-realtime-ready', () => {
            console.log('Firebase Realtime Database ready, realtime notification handler initialized');
        });
    }

    /**
     * Handle a new notification from Firebase
     */
    handleNewNotification(event) {
        const notification = event.detail;
        
        // If notification is not read, increment unread count
        if (!notification.is_read) {
            this.unreadCount++;
            this.updateNotificationBadge();
        }
        
        // Update page title to indicate new notification
        this.updatePageTitle();
        
        // Play notification sound if available
        this.playNotificationSound();
    }

    /**
     * Handle an updated notification from Firebase
     */
    handleUpdatedNotification(event) {
        const notification = event.detail;
        
        // If notification was marked as read and we were tracking it as unread
        if (notification.is_read && this.unreadCount > 0) {
            this.unreadCount--;
            this.updateNotificationBadge();
            this.updatePageTitle();
        }
    }

    /**
     * Update the notification badge count
     */
    updateNotificationBadge() {
        const badgeElement = document.getElementById('notification-count');
        if (badgeElement) {
            badgeElement.textContent = this.unreadCount.toString();
            badgeElement.style.display = this.unreadCount > 0 ? 'flex' : 'none';
        }
    }

    /**
     * Update the page title to indicate new notifications
     */
    updatePageTitle() {
        // Get original title without count prefix
        const originalTitle = document.title.replace(/^\(\d+\) /, '');
        
        // Add count to title if there are unread notifications
        if (this.unreadCount > 0) {
            document.title = `(${this.unreadCount}) ${originalTitle}`;
        } else {
            document.title = originalTitle;
        }
    }

    /**
     * Play a notification sound
     */
    playNotificationSound() {
        try {
            // Check if notification sound is enabled
            const soundEnabled = localStorage.getItem('notification_sound_enabled') !== 'false';
            
            if (soundEnabled) {
                // Create audio element
                const audio = new Audio(`${this.publicUrl}/staticfirebase/notification.mp3`);
                audio.volume = 0.5;
                audio.play().catch(error => {
                    // Autoplay might be blocked, that's fine
                    console.log('Could not play notification sound:', error.message);
                });
            }
        } catch (error) {
            console.error('Error playing notification sound:', error);
        }
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.realtimeNotificationHandler = new RealtimeNotificationHandler();
});